-- =============================================
-- EPCommerce User Database Schema
-- Role-based User Management System
-- =============================================

-- ---------------------------------------------
-- ตาราง sys_seq สำหรับ sequence number
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `sys_seq` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `seq` BIGINT NOT NULL DEFAULT 0,
    `table_name` VARCHAR(50) NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert initial sequence for users table
INSERT INTO `sys_seq` (`id`, `seq`, `table_name`) VALUES (1, 0, 'users');

-- ---------------------------------------------
-- ตาราง user_roles สำหรับเก็บประเภท role
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `user_roles` (
    `role_id` INT NOT NULL AUTO_INCREMENT,
    `role_code` VARCHAR(20) NOT NULL UNIQUE,
    `role_name` VARCHAR(50) NOT NULL,
    `role_description` TEXT,
    `can_access_admin` TINYINT(1) DEFAULT 0,
    `can_manage_users` TINYINT(1) DEFAULT 0,
    `can_manage_products` TINYINT(1) DEFAULT 0,
    `can_view_reports` TINYINT(1) DEFAULT 0,
    `can_view_representative_price` TINYINT(1) DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`role_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default roles
INSERT INTO `user_roles` (`role_code`, `role_name`, `role_description`, `can_access_admin`, `can_manage_users`, `can_manage_products`, `can_view_reports`, `can_view_representative_price`) VALUES
('super_admin', 'Super Admin', 'สำหรับ Programmer และผู้ขายระบบให้ลูกค้า - มีสิทธิ์เข้าถึงทุกอย่าง', 1, 1, 1, 1, 1),
('admin', 'Admin', 'สำหรับเจ้าของร้าน - มีสิทธิ์จัดการร้านค้าของตัวเอง', 1, 1, 1, 1, 0),
('staff', 'Staff', 'สำหรับพนักงานร้าน - มีสิทธิ์ตามที่ admin กำหนด', 1, 0, 1, 0, 0),
('representative', 'Representative', 'สำหรับตัวแทนจำหน่าย - ดูสินค้าราคาตัวแทน', 0, 0, 0, 0, 1);

-- ---------------------------------------------
-- ตาราง users หลัก
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `users` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `US_id` VARCHAR(30) NOT NULL UNIQUE,
    `username` VARCHAR(50) NOT NULL UNIQUE,
    `email` VARCHAR(100) NOT NULL UNIQUE,
    `password` VARCHAR(255) NOT NULL,
    `first_name` VARCHAR(100),
    `last_name` VARCHAR(100),
    `phone` VARCHAR(20),
    `avatar` VARCHAR(255),
    `role_id` INT NOT NULL,
    `shop_id` INT DEFAULT NULL COMMENT 'ร้านที่สังกัด (สำหรับ admin, staff)',
    `parent_user_id` INT DEFAULT NULL COMMENT 'ผู้สร้าง user นี้',
    `status` ENUM('active', 'inactive', 'suspended', 'pending') DEFAULT 'active',
    `last_login` TIMESTAMP NULL,
    `login_attempts` INT DEFAULT 0,
    `locked_until` TIMESTAMP NULL,
    `email_verified` TINYINT(1) DEFAULT 0,
    `email_verified_at` TIMESTAMP NULL,
    `remember_token` VARCHAR(100),
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `deleted_at` TIMESTAMP NULL COMMENT 'Soft delete',
    PRIMARY KEY (`id`),
    INDEX `idx_US_id` (`US_id`),
    INDEX `idx_username` (`username`),
    INDEX `idx_email` (`email`),
    INDEX `idx_role_id` (`role_id`),
    INDEX `idx_shop_id` (`shop_id`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`role_id`) REFERENCES `user_roles`(`role_id`) ON DELETE RESTRICT ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- Trigger สำหรับสร้าง US_id อัตโนมัติ
-- ---------------------------------------------
DELIMITER $$
CREATE TRIGGER `trg_users_before_insert` BEFORE INSERT ON `users` FOR EACH ROW 
BEGIN
    DECLARE v_time BIGINT;
    DECLARE v_seq BIGINT;

    SET v_time = FLOOR(UNIX_TIMESTAMP(NOW(3)) * 1000);
    UPDATE sys_seq SET seq = seq + 1 WHERE id = 1;
    SELECT seq INTO v_seq FROM sys_seq WHERE id = 1;

    SET NEW.US_id = CONCAT('US_', v_time + (v_seq MOD 1000));
END
$$
DELIMITER ;

-- ---------------------------------------------
-- ตาราง user_permissions สำหรับ permission เฉพาะ user
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `user_permissions` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `user_id` INT NOT NULL,
    `permission_code` VARCHAR(50) NOT NULL,
    `granted` TINYINT(1) DEFAULT 1,
    `granted_by` INT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_user_permission` (`user_id`, `permission_code`),
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    FOREIGN KEY (`granted_by`) REFERENCES `users`(`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- ตาราง user_sessions สำหรับเก็บ session
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `user_sessions` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `user_id` INT NOT NULL,
    `session_token` VARCHAR(255) NOT NULL,
    `ip_address` VARCHAR(45),
    `user_agent` TEXT,
    `expires_at` TIMESTAMP NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_session_token` (`session_token`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_expires_at` (`expires_at`),
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- ตาราง user_activity_logs สำหรับเก็บ log กิจกรรม
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `user_activity_logs` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `user_id` INT,
    `action` VARCHAR(100) NOT NULL,
    `description` TEXT,
    `ip_address` VARCHAR(45),
    `user_agent` TEXT,
    `metadata` JSON,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_action` (`action`),
    INDEX `idx_created_at` (`created_at`),
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- ตาราง password_resets สำหรับ reset password
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `password_resets` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `email` VARCHAR(100) NOT NULL,
    `token` VARCHAR(255) NOT NULL,
    `expires_at` TIMESTAMP NOT NULL,
    `used` TINYINT(1) DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_email` (`email`),
    INDEX `idx_token` (`token`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- Insert default super_admin account
-- Password: admin123 (hashed with password_hash)
-- ---------------------------------------------
INSERT INTO `users` (`username`, `email`, `password`, `first_name`, `last_name`, `role_id`, `status`, `email_verified`) VALUES
('superadmin', 'superadmin@epcommerce.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Super', 'Admin', 1, 'active', 1);

-- =============================================
-- VIEWS สำหรับดึงข้อมูล
-- =============================================

-- View แสดง users พร้อม role
CREATE OR REPLACE VIEW `v_users_with_roles` AS
SELECT 
    u.id,
    u.US_id,
    u.username,
    u.email,
    u.first_name,
    u.last_name,
    u.phone,
    u.avatar,
    u.status,
    u.last_login,
    u.created_at,
    r.role_code,
    r.role_name,
    r.can_access_admin,
    r.can_manage_users,
    r.can_manage_products,
    r.can_view_reports,
    r.can_view_representative_price
FROM users u
INNER JOIN user_roles r ON u.role_id = r.role_id
WHERE u.deleted_at IS NULL;

-- View แสดงเฉพาะ active users
CREATE OR REPLACE VIEW `v_active_users` AS
SELECT * FROM `v_users_with_roles`
WHERE status = 'active';
