-- =============================================
-- EPCommerce Shop Database Schema
-- ร้านค้าและการเชื่อมต่อกับ User
-- =============================================

-- ---------------------------------------------
-- เพิ่ม sequence สำหรับ shops ใน sys_seq
-- ---------------------------------------------
INSERT INTO `sys_seq` (`id`, `seq`, `table_name`) VALUES (2, 0, 'shops');

-- ---------------------------------------------
-- ตาราง shops (ร้านค้า)
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `shops` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `SH_id` VARCHAR(30) NOT NULL UNIQUE COMMENT 'Shop ID เช่น SH_1736290000123',
    `shop_name` VARCHAR(150) NOT NULL COMMENT 'ชื่อร้าน',
    `shop_slug` VARCHAR(150) UNIQUE COMMENT 'URL-friendly name',
    `province` VARCHAR(100) COMMENT 'จังหวัด',
    `logo` VARCHAR(255) COMMENT 'โลโก้ร้าน',
    `cover_image` VARCHAR(255) COMMENT 'รูปปก',
    
    -- Contact Information
    `phone` VARCHAR(20) COMMENT 'เบอร์โทรศัพท์',
    `whatsapp` VARCHAR(20) COMMENT 'เบอร์ WhatsApp',
    `email` VARCHAR(100) COMMENT 'Email ร้าน',
    `address` TEXT COMMENT 'ที่อยู่',
    
    -- Social Media
    `facebook` VARCHAR(150) COMMENT 'ชื่อ Facebook',
    `facebook_url` VARCHAR(255) COMMENT 'URL Facebook',
    `tiktok` VARCHAR(100) COMMENT 'ชื่อ TikTok',
    `tiktok_url` VARCHAR(255) COMMENT 'URL TikTok',
    `instagram` VARCHAR(100) COMMENT 'ชื่อ Instagram',
    `instagram_url` VARCHAR(255) COMMENT 'URL Instagram',
    `line_id` VARCHAR(100) COMMENT 'Line ID',
    `website` VARCHAR(255) COMMENT 'Website URL',
    
    -- Business Info
    `description` TEXT COMMENT 'รายละเอียดร้าน',
    `business_type` VARCHAR(100) COMMENT 'ประเภทธุรกิจ',
    
    -- Status & Plan
    `status` ENUM('active', 'inactive', 'suspended', 'pending') DEFAULT 'pending',
    `plan_type` ENUM('free', 'basic', 'pro', 'enterprise') DEFAULT 'free',
    `plan_expires_at` TIMESTAMP NULL COMMENT 'วันหมดอายุ plan',
    
    -- Owner
    `owner_user_id` INT NOT NULL COMMENT 'User ID ของเจ้าของร้าน',
    
    -- Settings (JSON)
    `settings` JSON COMMENT 'การตั้งค่าร้าน',
    
    -- Timestamps
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `deleted_at` TIMESTAMP NULL COMMENT 'Soft delete',
    
    PRIMARY KEY (`id`),
    INDEX `idx_SH_id` (`SH_id`),
    INDEX `idx_shop_name` (`shop_name`),
    INDEX `idx_shop_slug` (`shop_slug`),
    INDEX `idx_province` (`province`),
    INDEX `idx_status` (`status`),
    INDEX `idx_plan_type` (`plan_type`),
    INDEX `idx_owner_user_id` (`owner_user_id`),
    FOREIGN KEY (`owner_user_id`) REFERENCES `users`(`id`) ON DELETE RESTRICT ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- Trigger สำหรับสร้าง SH_id อัตโนมัติ
-- ---------------------------------------------
DELIMITER $$
CREATE TRIGGER `trg_shops_before_insert` BEFORE INSERT ON `shops` FOR EACH ROW 
BEGIN
    DECLARE v_time BIGINT;
    DECLARE v_seq BIGINT;

    SET v_time = FLOOR(UNIX_TIMESTAMP(NOW(3)) * 1000);
    UPDATE sys_seq SET seq = seq + 1 WHERE id = 2;
    SELECT seq INTO v_seq FROM sys_seq WHERE id = 2;

    SET NEW.SH_id = CONCAT('SH_', v_time + (v_seq MOD 1000));
    
    -- Generate slug if not provided
    IF NEW.shop_slug IS NULL OR NEW.shop_slug = '' THEN
        SET NEW.shop_slug = LOWER(REPLACE(REPLACE(NEW.shop_name, ' ', '-'), '.', ''));
    END IF;
END
$$
DELIMITER ;

-- ---------------------------------------------
-- ตาราง shop_users (เชื่อมต่อร้านกับ user)
-- พนักงานของร้าน
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `shop_users` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `shop_id` INT NOT NULL COMMENT 'ID ของร้าน',
    `user_id` INT NOT NULL COMMENT 'ID ของ user (staff)',
    `position` VARCHAR(100) COMMENT 'ตำแหน่งในร้าน',
    `permissions` JSON COMMENT 'สิทธิ์เฉพาะในร้าน',
    `is_active` TINYINT(1) DEFAULT 1,
    `joined_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `invited_by` INT COMMENT 'ผู้เชิญเข้าร้าน',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_shop_user` (`shop_id`, `user_id`),
    INDEX `idx_shop_id` (`shop_id`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_is_active` (`is_active`),
    FOREIGN KEY (`shop_id`) REFERENCES `shops`(`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    FOREIGN KEY (`invited_by`) REFERENCES `users`(`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ---------------------------------------------
-- ตาราง shop_representatives (ตัวแทนร้าน)
-- สำหรับเชื่อมต่อ representative กับร้าน
-- ---------------------------------------------
CREATE TABLE IF NOT EXISTS `shop_representatives` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `shop_id` INT NOT NULL COMMENT 'ID ของร้าน',
    `user_id` INT NOT NULL COMMENT 'ID ของ representative',
    `discount_percent` DECIMAL(5,2) DEFAULT 0 COMMENT 'ส่วนลดเป็น %',
    `commission_percent` DECIMAL(5,2) DEFAULT 0 COMMENT 'ค่าคอมมิชชัน %',
    `rep_code` VARCHAR(20) UNIQUE COMMENT 'รหัสตัวแทน',
    `status` ENUM('active', 'inactive', 'pending') DEFAULT 'pending',
    `approved_by` INT COMMENT 'ผู้อนุมัติ',
    `approved_at` TIMESTAMP NULL,
    `notes` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_shop_rep` (`shop_id`, `user_id`),
    INDEX `idx_shop_id` (`shop_id`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_rep_code` (`rep_code`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`shop_id`) REFERENCES `shops`(`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    FOREIGN KEY (`approved_by`) REFERENCES `users`(`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- VIEWS สำหรับดึงข้อมูล
-- =============================================

-- View แสดงร้านพร้อมข้อมูลเจ้าของ
CREATE OR REPLACE VIEW `v_shops_with_owner` AS
SELECT 
    s.id,
    s.SH_id,
    s.shop_name,
    s.shop_slug,
    s.logo,
    s.province,
    s.phone,
    s.whatsapp,
    s.address,
    s.facebook,
    s.tiktok,
    s.instagram,
    s.website,
    s.status,
    s.plan_type,
    s.plan_expires_at,
    s.created_at,
    u.US_id AS owner_US_id,
    u.username AS owner_username,
    u.email AS owner_email,
    CONCAT(u.first_name, ' ', u.last_name) AS owner_name
FROM shops s
INNER JOIN users u ON s.owner_user_id = u.id
WHERE s.deleted_at IS NULL;

-- View แสดง staff ของร้าน
CREATE OR REPLACE VIEW `v_shop_staff` AS
SELECT 
    su.id,
    su.shop_id,
    su.position,
    su.is_active,
    su.joined_at,
    s.shop_name,
    s.SH_id,
    u.id AS user_id,
    u.US_id,
    u.username,
    u.email,
    u.first_name,
    u.last_name,
    u.phone,
    u.avatar
FROM shop_users su
INNER JOIN shops s ON su.shop_id = s.id
INNER JOIN users u ON su.user_id = u.id
WHERE s.deleted_at IS NULL AND u.deleted_at IS NULL;

-- View แสดง representatives ของร้าน
CREATE OR REPLACE VIEW `v_shop_representatives` AS
SELECT 
    sr.id,
    sr.shop_id,
    sr.discount_percent,
    sr.commission_percent,
    sr.rep_code,
    sr.status AS rep_status,
    sr.created_at AS joined_at,
    s.shop_name,
    s.SH_id,
    u.id AS user_id,
    u.US_id,
    u.username,
    u.email,
    u.first_name,
    u.last_name,
    u.phone
FROM shop_representatives sr
INNER JOIN shops s ON sr.shop_id = s.id
INNER JOIN users u ON sr.user_id = u.id
WHERE s.deleted_at IS NULL AND u.deleted_at IS NULL;
