<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

// ตรวจสอบการล็อกอิน
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

// เชื่อมต่อฐานข้อมูล
require_once 'includes/db_con.php';

// รับ order_id จาก URL
$order_id = isset($_GET['order_id']) ? $_GET['order_id'] : '';

if (empty($order_id)) {
    echo "ไม่พบเลขที่ออเดอร์";
    exit();
}

// ดึงข้อมูลออเดอร์ (รวมฟิลด์ invoice approval)
$order_query = "SELECT o.*, c.cus_name, c.cus_tel, c.cus_address, 
                       cur.currencys_symbol, cur.currencys_name,
                       u.full_name as user_name
                FROM orders o 
                LEFT JOIN customers c ON o.cus_id = c.cus_id
                LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id
                LEFT JOIN user u ON o.user_id = u.id
                WHERE o.order_id = ?";

$stmt = $conn->prepare($order_query);
$stmt->bind_param("s", $order_id);
$stmt->execute();
$order_result = $stmt->get_result();

if ($order_result->num_rows == 0) {
    echo "ไม่พบข้อมูลออเดอร์";
    exit();
}

$order = $order_result->fetch_assoc();

// กำหนดตัวแปร $inv_status จากข้อมูลออเดอร์
$inv_status = $order['invoice_approval_status'] ?? '';

// ดึงรายการสินค้า (รวมหน่วยจากตาราง units)
$detail_query = "SELECT od.*, p.pro_name, p.types_id, p.unit_id, u.unit_name,
                        tp.types_name, r.remark_text, dp.detail_text
                 FROM orders_detail od
                 LEFT JOIN products p ON od.pro_id = p.pro_id
                 LEFT JOIN types_product tp ON p.types_id = tp.types_id
                 LEFT JOIN units u ON p.unit_id = u.unit_id
                 LEFT JOIN remark r ON p.remark_id = r.remark_id
                 LEFT JOIN detail_product dp ON p.detail_product_id = dp.detail_product_id
                 WHERE od.order_id = ?
                 ORDER BY od.id";

$detail_stmt = $conn->prepare($detail_query);
$detail_stmt->bind_param("s", $order_id);
$detail_stmt->execute();
$detail_result = $detail_stmt->get_result();

$order_items = [];
while ($item = $detail_result->fetch_assoc()) {
    $order_items[] = $item;
}

$stmt->close();
$detail_stmt->close();
$conn->close();

// ฟังก์ชันจัดรูปแบบตัวเลข
function formatNumber($number, $currency_id = 'CUR_1') {
    if ($currency_id === 'CUR_3') { // USD
        return number_format($number, 2);
    } else {
        return number_format($number, 0);
    }
}

// ฟังก์ชันแปลงวันที่
function formatLaoDate($date) {
    $months = [
        1 => 'ມ.ກ.', 2 => 'ກ.ພ.', 3 => 'ມ.ນ.', 4 => 'ມ.ສ.',
        5 => 'ພ.ພ.', 6 => 'ມິ.ຖ.', 7 => 'ກ.ລ.', 8 => 'ສ.ຫ.',
        9 => 'ກ.ຍ.', 10 => 'ຕ.ລ.', 11 => 'ພ.ຈ.', 12 => 'ທ.ວ.'
    ];
    $date_obj = new DateTime($date);
    $day = $date_obj->format('j');
    $month = $months[(int)$date_obj->format('n')];
    $year = $date_obj->format('Y');
    return "$day $month $year";
}
?>

<!DOCTYPE html>
<html lang="lo">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ໃບເກັບເງິນ / Invoice</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;700&family=Noto+Sans:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="css/print_invoice.css">
</head>
<body>
    <div class="controls">
        <button class="btn" onclick="window.print()">🖨️ Print</button>
        <button class="btn" onclick="exportToPDF()">📄 Export PDF</button>
        <button class="btn btn-email" onclick="sendInvoiceEmail()">📧 ສົ່ງອີເມລອະນຸມັດ</button>
        <span class="quotation-info">ໃບເກັບເງິນ: <?php echo htmlspecialchars(!empty($order['invoice_no']) ? $order['invoice_no'] : $order['order_id']); ?></span>
        <?php if (!empty($order['invoice_approval_status'])): ?>
            <?php if ($order['invoice_approval_status'] === 'approved' || strtolower($order['invoice_approval_status']) === 'approved'): ?>
                <span class="status-badge approved">✓ ອະນຸມັດແລ້ວ</span>
            <?php elseif ($order['invoice_approval_status'] === 'rejected'): ?>
                <span class="status-badge rejected">✗ ປະຕິເສດ</span>
            <?php elseif ($order['invoice_approval_status'] === 'pending'): ?>
                <span class="status-badge pending">⏳ ລໍຖ້າອະນຸມັດ</span>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <div class="main-content">
        <div class="container">
            <!-- Header Section -->
            <div class="header">
                <div class="logo-section">
                    <div class="logo-placeholder">
                        <img src="img/wayha.png" alt="Company Logo" style="width: 90px; height: 90px; object-fit: contain;">
                    </div>
                    <div class="company-info">
                        <h1>Wayha Sokxay Technology CO.,LTD</h1>
                        <p>ຖະໜົນ ກຳເເພງເມືອງ, ບ້ານ ທາດຫຼວງກາງ, ເມືອງ ໄຊເສດຖາ ນະຄອນຫຼວງວຽງຈັນ</p>
                        <p><strong>Mobile:</strong> 020 2992 2634, <strong>Tel:</strong> 030 552 7260</p>
                        <p><strong>Email:</strong> info@wayha.la</p>
                    </div>
                </div>
            </div>

            <?php
                // เตรียมข้อความเหตุผลการปฏิเสธ (ถ้ามี)
                $invoice_status = trim($order['invoice_approval_status'] ?? '');
                $rejected_reason = '';
                if ($invoice_status === 'rejected') {
                    $rejected_reason = $order['invoice_not_approve_reason'] ?? '';
                }
            ?>

            <?php if (!empty($rejected_reason)): ?>
                <div class="rejection-banner" role="alert">
                    ຖືກປະຕິເສດ ຍ້ອນ : <?php echo nl2br(htmlspecialchars($rejected_reason)); ?>
                </div>
            <?php endif; ?>

            <div class="main-title">ໃບເກັບເງິນ / Invoice</div>

            <div class="customer-info-section">
                <div class="header-fields_2">
                    <div class="customer-field">
                        <label>ລູກຄ້າ:</label>
                        <span><?php echo htmlspecialchars($order['cus_name'] ?? 'ไม่ระบุ'); ?></span>
                    </div>
                    <div class="customer-field">
                        <label>ຫົວຂໍ້:</label>
                        <span><?php echo htmlspecialchars($order['section'] ?? 'ໃບເກັບເງິນ'); ?></span>
                    </div>
                </div>

                <div class="header-fields">
                    <div class="field">
                        <label>No:</label>
                        <span><?php echo htmlspecialchars(!empty($order['invoice_no']) ? $order['invoice_no'] : $order['order_id']); ?></span>
                    </div>
                    <div class="field">
                        <label>Date:</label>
                        <span><?php echo formatLaoDate($order['order_date']); ?></span>
                    </div>
                    <div class="field">
                        <label>Currency:</label>
                        <span><?php echo htmlspecialchars($order['currencys_symbol'] ?? '₭'); ?></span>
                    </div>
                </div>
            </div>

            <!-- Invoice Table -->
            <div class="table-container">
                <table class="invoice-table">
                    <thead>
                        <tr>
                            <th class="col-no">ລຳດັບ<br>No</th>
                            <th class="col-description">ລາຍການ<br>Descriptions</th>
                            <th class="col-detail">ລາຍລະອຽດ<br>Detail</th>
                            <th class="col-amount">ຈຳນວນ<br>Amount</th>
                            <th class="col-unit">ຫົວໜ່ວຍ<br>Unit</th>
                            <th class="col-price">ລາຄາ<br>Price</th>
                            <th class="col-total">ຈຳນວນເງີນ<br>Total</th>
                        </tr>
                    </thead>
                    <tbody id="invoiceTableBody">
                        <?php if (!empty($order_items)): ?>
                            <?php $i = 1; foreach ($order_items as $item): ?>
                            <tr>
                                <td class="col-no"><?php echo $i; ?></td>
                                <td class="col-description"><?php echo htmlspecialchars($item['pro_name'] ?? 'ไม่ระบุสินค้า'); ?></td>
                                <td class="col-detail">
                                    <?php 
                                        $detail = htmlspecialchars($item['detail_text'] ?? '');
                                        $remark = htmlspecialchars($item['remark_text'] ?? '');
                                        echo nl2br($detail);
                                        if (!empty($remark)) {
                                            echo (!empty($detail) ? '<br>' : '') . nl2br($remark);
                                        }
                                    ?>
                                </td>
                                <td class="col-amount"><?php echo htmlspecialchars($item['order_qty']); ?></td>
                                <td class="col-unit"><?php echo !empty($item['unit_name']) ? htmlspecialchars($item['unit_name']) : '-'; ?></td>
                                <td class="col-price"><?php echo formatNumber($item['pro_price'], $order['currencys_id']); ?></td>
                                <td class="col-total"><?php echo formatNumber($item['total'], $order['currencys_id']); ?></td>
                            </tr>
                            <?php $i++; endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" style="text-align: center; padding: 20px;">ไม่พบรายการสินค้า</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Totals Section -->
            <div class="totals-section">
                <div class="notes-section">
                    <h3>ໝາຍເຫດ:</h3>
                    <ul class="notes-list">
                        <li>ກຳນົດຈ່າຍພາຍໃນ 7 ວັນ, ຫຼັງຈາກໄດ້ຮັບໃບບີນຮຽກເກັບເງີນ</li>
                        <li>BCEL Bank</li>
                        <li>Account Name: Wayha Sokxay Technology Co.,Ltd</li>
                        <li>LAK: 010 110 014 0011</li>
                        <li>USD: 010 110 014 0039</li>
                        <li>THB: 010 110 014 0020</li>
                    </ul>
                </div>
                <div class="totals-block">
                    <div class="total-row">
                        <span class="label">Total:</span>
                        <span class="amount"><?php echo htmlspecialchars($order['currencys_symbol'] ?? '₭'); ?> <?php echo formatNumber($order['order_subtotal'], $order['currencys_id']); ?></span>
                    </div>
                    <div class="total-row">
                        <span class="label">ສ່ວນຫຼຸດ / Discount:</span>
                        <span class="amount"><?php echo htmlspecialchars($order['currencys_symbol'] ?? '₭'); ?> <?php echo formatNumber($order['order_discount'], $order['currencys_id']); ?></span>
                    </div>
                    <?php if ($order['order_tax_percent'] > 0): ?>
                    <div class="total-row">
                        <span class="label">VAT <?php echo $order['order_tax_percent']; ?>%:</span>
                        <span class="amount"><?php echo htmlspecialchars($order['currencys_symbol'] ?? '₭'); ?> <?php echo formatNumber($order['order_tax_amount'], $order['currencys_id']); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="total-row">
                        <span class="label">Grand Total:</span>
                        <span class="amount"><?php echo htmlspecialchars($order['currencys_symbol'] ?? '₭'); ?> <?php echo formatNumber($order['order_total'], $order['currencys_id']); ?></span>
                    </div>
                </div>
            </div>

            <!-- Signature Section -->
            <div class="signature-section">
                <div class="signature-block">
                    <h4>ຜູ້ຮັບເອກະສານ(Received by):</h4>
                    <div class="stamp-area">
                        
                    </div>
                    <div class="signature-fields">
                        <div class="signature-field">
                            <label>Name:</label>
                            <div class="line"></div>
                        </div>
                        <div class="signature-field">
                            <label>Date:</label>
                            <div class="line"></div>
                        </div>
                    </div>
                </div>

                <div class="signature-block">
                    <h4>ບໍລິສັດ ເວຫາ ໂຊກໄຊ ເທັກໂນໂລຍີ ຈຳກັດ </h4>
                    <div class="stamp-area">
                        <?php
                            if ($inv_status === 'approved' || strtolower($inv_status) === 'approved' || $inv_status === 'ອະນຸມັດ'):
                        ?>
                            <img src="img/st.png" alt="Signature & Stamp" style="width: 120%; height: auto; max-height: 90px; object-fit: contain;">
                        <?php endif; ?>
                    </div>
                    <div class="signature-fields">
                        <div class="signature-field">
                            <label>Name:</label>
                            <div class="line"><?php echo htmlspecialchars($order['user_name'] ?? ''); ?></div>
                        </div>
                        <div class="signature-field">
                            <label>Date:</label>
                            <div class="line"><?php echo formatLaoDate(date('Y-m-d')); ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Sweet Alert -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <!-- html2pdf.js library -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
    <script>
        const orderId = '<?php echo htmlspecialchars($order['order_id']); ?>';

        function exportToPDF() {
            const controls = document.querySelector('.controls');
            const originalVisibility = controls.style.visibility;
            controls.style.visibility = 'hidden';
            const element = document.querySelector('.container');
            const opt = {
                margin: [5,5,5,5],
                filename: 'invoice_' + orderId + '.pdf',
                image: { type: 'jpeg', quality: 0.98 },
                html2canvas: { scale: 2, useCORS: true, letterRendering: true, logging: false, windowHeight: 1100 },
                jsPDF: { unit: 'mm', format: 'a4', orientation: 'portrait' },
                pagebreak: { mode: 'avoid-all' }
            };

            Swal.fire({ title: 'ກຳລັງສ້າງ PDF...', text: 'ກະລຸນາລໍຖ້າ', allowOutsideClick: false, didOpen: () => { Swal.showLoading(); } });

            html2pdf().set(opt).from(element).toPdf().get('pdf').then(function(pdf){
                const totalPages = pdf.internal.getNumberOfPages();
                if (totalPages > 1) {
                    for (let i = totalPages; i > 1; i--) pdf.deletePage(i);
                }
                pdf.save('invoice_' + orderId + '.pdf');
                controls.style.visibility = originalVisibility;
                Swal.close();
            }).catch(error => {
                controls.style.visibility = originalVisibility;
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: 'ບໍ່ສາມາດສ້າງ PDF ໄດ້', confirmButtonText: 'ຕົກລົງ' });
            });
        }

        function sendInvoiceEmail() {
            Swal.fire({ title: 'ກຳລັງສົ່ງອີເມລ...', text: 'ກະລຸນາລໍຖ້າ', allowOutsideClick: false, didOpen: () => { Swal.showLoading(); } });

            fetch('api/send_invoice_email.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ order_id: orderId })
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    Swal.fire({ icon: 'success', title: 'ສຳເລັດ!', text: 'ສົ່ງອີເມລໄປຫາຫົວໜ້າແລ້ວ', confirmButtonText: 'ຕົກລົງ' });
                } else {
                    throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
                }
            }).catch(err => {
                console.error(err);
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: 'ບໍ່ສາມາດສົ່ງອີເມລໄດ້: ' + err.message, confirmButtonText: 'ຕົກລົງ' });
            });
        }
    </script>

    <style>
        .stamp-area {
    text-align: center;
    display: flex;
    justify-content: center;
    align-items: center;
    height: 100px; /* ปรับความสูงตามต้องการ */
}
.stamp-area .stamp-img {
    display: block;
    margin: 0 auto;
    max-width: 180px;
    max-height: 90px;
    object-fit: contain;
}
        .controls .quotation-info { font-weight: bold; color: #333; margin-left: 20px; }
        .controls .btn-email { background: #17a2b8; }
        .controls .btn-email:hover { background: #138496; }
        .status-badge { display: inline-block; padding: 8px 16px; border-radius: 20px; font-weight: bold; font-size: 14px; margin-left: 15px; }
        .status-badge.approved { background: #28a745; color: white; }
        .status-badge.pending { background: #ffc107; color: #333; }
        .status-badge.rejected { background: #dc3545; color: #fff; }
        
        /* Rejection banner - very prominent red text */
        .rejection-banner {
            border: 2px solid #e74c3c;
            background: #fff5f5;
            color: #c0392b;
            font-weight: 800;
            font-size: 20px;
            padding: 14px 18px;
            border-radius: 6px;
            text-align: center;
            margin: 12px 0;
            line-height: 1.3;
        }
        
        @media print { .controls { display: none !important; } }
    </style>

</body>
</html>