<?php
/**
 * Gmail SMTP Mailer - Simple implementation without external libraries
 */

class GmailSMTPMailer {
    private $smtp_host;
    private $smtp_port;
    private $smtp_user;
    private $smtp_pass;
    private $from_email;
    private $from_name;
    
    public function __construct() {
        require_once __DIR__ . '/../includes/email_config.php';
        
        $this->smtp_host = SMTP_HOST;
        $this->smtp_port = SMTP_PORT;
        $this->smtp_user = SMTP_USERNAME;
        $this->smtp_pass = SMTP_PASSWORD;
        $this->from_email = FROM_EMAIL;
        $this->from_name = FROM_NAME;
    }
    
    public function send($to_email, $subject, $html_body) {
        return $this->sendWithAttachment($to_email, $subject, $html_body, null, null);
    }
    
    public function sendWithAttachment($to_email, $subject, $html_body, $attachment_path = null, $attachment_name = null) {
        try {
            // เชื่อมต่อกับ Gmail SMTP
            $socket = fsockopen(SMTP_HOST, SMTP_PORT, $errno, $errstr, 30);
            
            if (!$socket) {
                throw new Exception("Cannot connect to SMTP: $errstr ($errno)");
            }
            
            // รับ response
            $this->getResponse($socket);
            
            // EHLO
            fputs($socket, "EHLO " . $_SERVER['SERVER_NAME'] . "\r\n");
            $this->getResponse($socket);
            
            // STARTTLS
            fputs($socket, "STARTTLS\r\n");
            $this->getResponse($socket);
            
            // Enable crypto
            stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
            
            // EHLO again after TLS
            fputs($socket, "EHLO " . $_SERVER['SERVER_NAME'] . "\r\n");
            $this->getResponse($socket);
            
            // AUTH LOGIN
            fputs($socket, "AUTH LOGIN\r\n");
            $this->getResponse($socket);
            
            // Username
            fputs($socket, base64_encode($this->smtp_user) . "\r\n");
            $this->getResponse($socket);
            
            // Password
            fputs($socket, base64_encode($this->smtp_pass) . "\r\n");
            $this->getResponse($socket);
            
            // MAIL FROM
            fputs($socket, "MAIL FROM: <{$this->from_email}>\r\n");
            $this->getResponse($socket);
            
            // RCPT TO
            fputs($socket, "RCPT TO: <{$to_email}>\r\n");
            $this->getResponse($socket);
            
            // DATA
            fputs($socket, "DATA\r\n");
            $this->getResponse($socket);
            
            // Email headers and body
            if ($attachment_path && file_exists($attachment_path)) {
                $email_content = $this->buildEmailWithAttachment($to_email, $subject, $html_body, $attachment_path, $attachment_name);
            } else {
                $email_content = $this->buildEmail($to_email, $subject, $html_body);
            }
            fputs($socket, $email_content . "\r\n.\r\n");
            $this->getResponse($socket);
            
            // QUIT
            fputs($socket, "QUIT\r\n");
            fclose($socket);
            
            return true;
            
        } catch (Exception $e) {
            error_log("Gmail SMTP Error: " . $e->getMessage());
            return false;
        }
    }
    
    private function getResponse($socket) {
        $response = '';
        while ($line = fgets($socket, 515)) {
            $response .= $line;
            if (substr($line, 3, 1) == ' ') {
                break;
            }
        }
        return $response;
    }
    
    private function buildEmail($to_email, $subject, $html_body) {
        $boundary = md5(time());
        
        $headers = "From: {$this->from_name} <{$this->from_email}>\r\n";
        $headers .= "To: <{$to_email}>\r\n";
        $headers .= "Subject: {$subject}\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "Content-Transfer-Encoding: 8bit\r\n";
        $headers .= "\r\n";
        
        return $headers . $html_body;
    }
    
    private function buildEmailWithAttachment($to_email, $subject, $html_body, $attachment_path, $attachment_name) {
        $boundary = md5(time());
        
        $headers = "From: {$this->from_name} <{$this->from_email}>\r\n";
        $headers .= "To: <{$to_email}>\r\n";
        $headers .= "Subject: {$subject}\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: multipart/mixed; boundary=\"{$boundary}\"\r\n";
        $headers .= "\r\n";
        
        // HTML Body Part
        $body = "--{$boundary}\r\n";
        $body .= "Content-Type: text/html; charset=UTF-8\r\n";
        $body .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
        $body .= $html_body . "\r\n\r\n";
        
        // Attachment Part
        if ($attachment_name) {
            $file_content = file_get_contents($attachment_path);
            $encoded_content = chunk_split(base64_encode($file_content));
            
            $body .= "--{$boundary}\r\n";
            $body .= "Content-Type: application/pdf; name=\"{$attachment_name}\"\r\n";
            $body .= "Content-Transfer-Encoding: base64\r\n";
            $body .= "Content-Disposition: attachment; filename=\"{$attachment_name}\"\r\n\r\n";
            $body .= $encoded_content . "\r\n\r\n";
        }
        
        $body .= "--{$boundary}--\r\n";
        
        return $headers . $body;
    }
}
