// sidebar.js - Shared Sidebar JavaScript

// Global function สำหรับ Coming Soon
function showComingSoon(feature) {
    Swal.fire({
        title: 'ກຳລັງພັດທະນາ',
        text: `ຟີເຈີ "${feature}" ກຳລັງຢູ່ໃນຂັ້ນຕອນພັດທະນາ`,
        icon: 'info',
        confirmButtonText: 'ເຂົ້າໃຈແລ້ວ',
        confirmButtonColor: '#2c5aa0',
        customClass: {
            popup: 'swal-popup',
            title: 'swal-title',
            content: 'swal-content'
        }
    });
}

// Global function สำหรับ toggle dropdown
function toggleDropdown(element) {
    const dropdownMenu = element.nextElementSibling;
    const arrow = element.querySelector('.dropdown-arrow');
    
    // Close other dropdowns
    document.querySelectorAll('.dropdown-menu.open').forEach(menu => {
        if (menu !== dropdownMenu) {
            menu.classList.remove('open');
            const otherToggle = menu.previousElementSibling;
            otherToggle.classList.remove('open');
        }
    });
    
    // Toggle current dropdown
    dropdownMenu.classList.toggle('open');
    element.classList.toggle('open');
}

// Sidebar functionality
class SidebarManager {
    constructor() {
        this.sidebar = document.getElementById('sidebar');
        this.toggleButton = document.getElementById('sidebar-toggle');
        this.navItems = document.querySelectorAll('.nav-item');
        
        this.init();
    }

    init() {
        // เริ่มต้น event listeners
        this.initEventListeners();
        
        // เริ่มต้น mobile functionality
        this.initMobile();
        
        // เริ่มต้น hover effects
        this.initHoverEffects();
    }

    initEventListeners() {
        // Mobile sidebar toggle
        if (this.toggleButton && this.sidebar) {
            this.toggleButton.addEventListener('click', () => {
                this.toggleSidebar();
            });
            
            // Close sidebar when clicking outside (mobile)
            document.addEventListener('click', (e) => {
                if (window.innerWidth <= 768 && 
                    !this.sidebar.contains(e.target) && 
                    !this.toggleButton.contains(e.target) &&
                    this.sidebar.classList.contains('open')) {
                    this.closeSidebar();
                }
            });
        }

        // Navigation items click handling (regular nav items, not dropdown)
        this.navItems.forEach(item => {
            // Skip dropdown toggles
            if (item.classList.contains('dropdown-toggle')) {
                return;
            }
            
            item.addEventListener('click', (e) => {
                // ถ้าไม่มี href หรือเป็น # ให้ป้องกันการ navigate
                const href = item.getAttribute('href');
                if (!href || href === '#') {
                    e.preventDefault();
                }
                
                // Update active state (เฉพาะ link ที่ไม่มี onclick)
                if (!item.hasAttribute('onclick') && href && href !== '#') {
                    this.updateActiveState(item);
                }
                
                // Close sidebar on mobile after clicking
                if (window.innerWidth <= 768) {
                    this.closeSidebar();
                }
            });
        });

        // Dropdown items click handling
        const dropdownItems = document.querySelectorAll('.dropdown-item');
        dropdownItems.forEach(item => {
            item.addEventListener('click', (e) => {
                const href = item.getAttribute('href');
                if (href && href !== '#') {
                    // Remove active from all dropdown items
                    dropdownItems.forEach(di => di.classList.remove('active'));
                    // Add active to clicked item
                    item.classList.add('active');
                    
                    // Keep parent dropdown active
                    const parentDropdown = item.closest('.nav-item-dropdown');
                    if (parentDropdown) {
                        parentDropdown.classList.add('active');
                    }
                }
                
                // Close sidebar on mobile after clicking
                if (window.innerWidth <= 768) {
                    this.closeSidebar();
                }
            });
        });

        // Close dropdowns when clicking outside
        document.addEventListener('click', (e) => {
            if (!e.target.closest('.nav-item-dropdown')) {
                document.querySelectorAll('.dropdown-menu.open').forEach(menu => {
                    menu.classList.remove('open');
                    const toggle = menu.previousElementSibling;
                    toggle.classList.remove('open');
                });
            }
        });
    }

    initMobile() {
        // Handle window resize
        window.addEventListener('resize', () => {
            if (window.innerWidth > 768) {
                this.sidebar.classList.remove('open');
            }
        });
    }

    initHoverEffects() {
        // Add hover effects for nav items (ไม่รวม dropdown toggle)
        this.navItems.forEach(item => {
            if (!item.classList.contains('dropdown-toggle')) {
                item.addEventListener('mouseenter', () => {
                    if (!item.classList.contains('active') && !item.closest('.nav-item-dropdown.active')) {
                        item.style.backgroundColor = '#f8f9fa';
                    }
                });
                
                item.addEventListener('mouseleave', () => {
                    if (!item.classList.contains('active') && !item.closest('.nav-item-dropdown.active')) {
                        item.style.backgroundColor = '';
                    }
                });
            }
        });

        // Add hover effects for dropdown toggles
        const dropdownToggles = document.querySelectorAll('.dropdown-toggle');
        dropdownToggles.forEach(toggle => {
            toggle.addEventListener('mouseenter', () => {
                if (!toggle.closest('.nav-item-dropdown').classList.contains('active')) {
                    toggle.style.backgroundColor = '#f8f9fa';
                }
            });
            
            toggle.addEventListener('mouseleave', () => {
                if (!toggle.closest('.nav-item-dropdown').classList.contains('active')) {
                    toggle.style.backgroundColor = '';
                }
            });
        });
    }

    toggleSidebar() {
        this.sidebar.classList.toggle('open');
    }

    closeSidebar() {
        this.sidebar.classList.remove('open');
    }

    updateActiveState(activeItem) {
        // Remove active class from all items
        this.navItems.forEach(nav => nav.classList.remove('active'));
        
        // Add active class to clicked item
        activeItem.classList.add('active');
    }

    // Method to programmatically set active menu item
    setActiveMenuItem(pageName) {
        this.navItems.forEach(item => {
            item.classList.remove('active');
            const href = item.getAttribute('href');
            if (href && href.includes(pageName)) {
                item.classList.add('active');
            }
        });
    }
}

// Initialize sidebar when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize sidebar
    window.sidebarManager = new SidebarManager();
    
    // Add SweetAlert styles
    const style = document.createElement('style');
    style.textContent = `
        .swal-popup {
            font-family: 'Noto Sans Lao', sans-serif !important;
        }
        
        /* Mobile sidebar styles */
        @media (max-width: 768px) {
            .mobile-sidebar-toggle {
                display: block !important;
                position: fixed;
                top: 1rem;
                left: 1rem;
                z-index: 1001;
                background: var(--bg-white);
                border: 1px solid var(--border-color);
                border-radius: 6px;
                padding: 0.75rem;
                cursor: pointer;
                box-shadow: var(--shadow);
                color: var(--text-dark);
                font-size: 1rem;
            }
        }
    `;
    
    if (!document.getElementById('sidebar-styles')) {
        style.id = 'sidebar-styles';
        document.head.appendChild(style);
    }
});