 // Initialize with current date
        document.getElementById('quotationDate').value = new Date().toISOString().split('T')[0];

        // Number formatting function
        function formatNumber(num) {
            if (!num || num === 0) return '-';
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(num);
        }

        // Number formatting function for Total column (no decimals for whole numbers)
        function formatNumberTotal(num) {
            if (!num || num === 0) return '-';
            
            // Check if the number is a whole number
            if (num % 1 === 0) {
                // Format without decimals for whole numbers
                return new Intl.NumberFormat('en-US', {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 0
                }).format(num);
            } else {
                // Format with 2 decimals for numbers with decimal places
                return new Intl.NumberFormat('en-US', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                }).format(num);
            }
        }

        // Currency symbol management
        let currentCurrency = '₭';
        let vatEnabled = true; // VAT toggle state

        function toggleVAT() {
            vatEnabled = !vatEnabled;
            const vatButton = document.getElementById('vatToggle');
            const vatRow = document.getElementById('vatRow');
            
            if (vatEnabled) {
                vatButton.textContent = '✓ VAT 10%';
                vatButton.style.backgroundColor = 'var(--primary-color)';
                vatRow.style.display = 'flex';
            } else {
                vatButton.textContent = '✕ VAT 10%';
                vatButton.style.backgroundColor = '#dc3545';
                vatRow.style.display = 'none';
            }
            
            calculateTotals();
        }

        function updateCurrency() {
            const select = document.getElementById('currencySelect');
            currentCurrency = select.value;
            document.getElementById('selectedCurrency').textContent = currentCurrency;
            
            // Update all currency displays
            const currencySpans = document.querySelectorAll('.currency');
            currencySpans.forEach(span => {
                span.textContent = currentCurrency;
            });

            // Recalculate totals to update currency display
            calculateTotals();
        }

        // Debounce function for input handling
        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        // Calculate individual row total
        function calculateRowTotal(input) {
            const row = input.closest('tr');
            const amountInput = row.querySelector('.col-amount input');
            const priceInput = row.querySelector('.col-price input');
            const totalSpan = row.querySelector('.total-display');

            const amount = parseFloat(getNumericValue(amountInput.value)) || 0;
            const price = parseFloat(getNumericValue(priceInput.value)) || 0;
            const total = amount * price;

            totalSpan.textContent = formatNumberTotal(total);
            
            // Debounced total calculation
            debouncedCalculateTotals();
        }

        // Calculate overall totals
        function calculateTotals() {
            const totalDisplays = document.querySelectorAll('.total-display');
            let subtotal = 0;

            totalDisplays.forEach(display => {
                const text = display.textContent;
                if (text !== '-') {
                    // Remove thousand separators and convert to number
                    const value = parseFloat(text.replace(/,/g, '')) || 0;
                    subtotal += value;
                }
            });

            // Get discount value
            const discountInput = document.getElementById('discountInput');
            const discount = parseFloat(getNumericValue(discountInput.value)) || 0;
            const subtotalAfterDiscount = subtotal - discount;

            const vatRate = 0.1; // 10% VAT
            const vatAmount = vatEnabled ? (subtotalAfterDiscount * vatRate) : 0;
            const grandTotal = subtotalAfterDiscount + vatAmount;

            // Update displays
            document.getElementById('subtotal').innerHTML = `<span class="currency">${currentCurrency}</span> ${formatNumberTotal(subtotal)}`;
            document.getElementById('vatAmount').innerHTML = `<span class="currency">${currentCurrency}</span> ${formatNumberTotal(vatAmount)}`;
            document.getElementById('grandTotal').innerHTML = `<span class="currency">${currentCurrency}</span> ${formatNumberTotal(grandTotal)}`;
        }

        // Create debounced version of calculateTotals
        const debouncedCalculateTotals = debounce(calculateTotals, 300);

        // Add event listeners to all amount and price inputs
        document.addEventListener('DOMContentLoaded', function() {
            const inputs = document.querySelectorAll('.col-amount input, .col-price input');
            inputs.forEach(input => {
                input.addEventListener('input', function() {
                    calculateRowTotal(this);
                });
                
                // Also handle blur event for better UX
                input.addEventListener('blur', function() {
                    calculateRowTotal(this);
                });
            });

            // Add number formatting to amount and price inputs
            const numberInputs = document.querySelectorAll('.col-amount input, .col-price input');
            numberInputs.forEach(input => {
                input.addEventListener('input', function() {
                    // Store cursor position
                    const cursorPos = this.selectionStart;
                    const oldLength = this.value.length;
                    
                    // Format the value
                    const formattedValue = formatNumberInput(this.value);
                    this.value = formattedValue;
                    
                    // Adjust cursor position
                    const newLength = this.value.length;
                    const newCursorPos = cursorPos + (newLength - oldLength);
                    this.setSelectionRange(newCursorPos, newCursorPos);
                    
                    // Calculate row total
                    calculateRowTotal(this);
                });

                // Handle paste events
                input.addEventListener('paste', function(e) {
                    setTimeout(() => {
                        this.value = formatNumberInput(this.value);
                        calculateRowTotal(this);
                    }, 10);
                });
            });

            // Add auto-resize functionality to customer info inputs
            const customerInputs = document.querySelectorAll('.customer-field input');
            customerInputs.forEach(input => {
                // Set initial size
                autoResizeInput(input);
                
                // Add event listener for input changes
                input.addEventListener('input', function() {
                    autoResizeInput(this);
                });
            });

            // Add discount input formatting and calculation
            const discountInput = document.getElementById('discountInput');
            discountInput.addEventListener('input', function() {
                // Store cursor position
                const cursorPos = this.selectionStart;
                const oldLength = this.value.length;
                
                // Format the value
                const formattedValue = formatNumberInput(this.value);
                this.value = formattedValue;
                
                // Adjust cursor position
                const newLength = this.value.length;
                const newCursorPos = cursorPos + (newLength - oldLength);
                this.setSelectionRange(newCursorPos, newCursorPos);
                
                // Recalculate totals
                debouncedCalculateTotals();
            });

            discountInput.addEventListener('paste', function(e) {
                setTimeout(() => {
                    this.value = formatNumberInput(this.value);
                    debouncedCalculateTotals();
                }, 10);
            });

            // Initialize totals
            calculateTotals();
        });

        // Format number with comma separators for display
        function formatNumberInput(value) {
            // Remove any non-digit characters except decimal point
            const cleanValue = value.replace(/[^\d.]/g, '');
            
            // Split by decimal point
            const parts = cleanValue.split('.');
            
            // Add comma separators to the integer part
            parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
            
            // Rejoin with decimal point if it exists
            return parts.join('.');
        }

        // Get raw numeric value from formatted string
        function getNumericValue(value) {
            return value.replace(/,/g, '');
        }

        // Auto-resize input function
        function autoResizeInput(input) {
            // Create a temporary span to measure text width
            const tempSpan = document.createElement('span');
            tempSpan.style.visibility = 'hidden';
            tempSpan.style.position = 'absolute';
            tempSpan.style.whiteSpace = 'nowrap';
            tempSpan.style.font = window.getComputedStyle(input).font;
            tempSpan.textContent = input.value || input.placeholder;
            
            document.body.appendChild(tempSpan);
            const textWidth = tempSpan.offsetWidth;
            document.body.removeChild(tempSpan);
            
            // Set input width with some padding
            const minWidth = 150; // Minimum width in pixels
            const padding = 20; // Extra padding
            const newWidth = Math.max(minWidth, textWidth + padding);
            
            input.style.width = newWidth + 'px';
        }

        // Print functionality
        function printQuotation() {
            window.print();
        }

        

        // Auto-save to localStorage (optional feature)
        function saveFormData() {
            const formData = {
                quotationNo: document.getElementById('quotationNo').value,
                quotationDate: document.getElementById('quotationDate').value,
                currency: document.getElementById('currencySelect').value,
                rows: []
            };

            const rows = document.querySelectorAll('#invoiceTableBody tr');
            rows.forEach(row => {
                const inputs = row.querySelectorAll('input');
                formData.rows.push({
                    description: inputs[0].value,
                    amount: getNumericValue(inputs[1].value), // Store raw numeric value
                    unit: inputs[2].value,
                    price: getNumericValue(inputs[3].value) // Store raw numeric value
                });
            });

            localStorage.setItem('quotationForm', JSON.stringify(formData));
        }

        function loadFormData() {
            const saved = localStorage.getItem('quotationForm');
            if (saved) {
                const formData = JSON.parse(saved);
                document.getElementById('quotationNo').value = formData.quotationNo || '';
                document.getElementById('quotationDate').value = formData.quotationDate || '';
                document.getElementById('currencySelect').value = formData.currency || '₭';
                updateCurrency();

                const rows = document.querySelectorAll('#invoiceTableBody tr');
                formData.rows.forEach((rowData, index) => {
                    if (rows[index]) {
                        const inputs = rows[index].querySelectorAll('input');
                        inputs[0].value = rowData.description || '';
                        inputs[1].value = formatNumberInput(rowData.amount || ''); // Format on load
                        inputs[2].value = rowData.unit || '';
                        inputs[3].value = formatNumberInput(rowData.price || ''); // Format on load
                        calculateRowTotal(inputs[1]);
                    }
                });
            }
        }

        // Auto-save on input change
        document.addEventListener('input', debounce(saveFormData, 1000));

        // Load saved data on page load
        document.addEventListener('DOMContentLoaded', loadFormData);

        // Export to PDF functionality
        function exportToPDF() {
            // Get current date and quotation number for filename
            const quotationNo = document.getElementById('quotationNo').value || 'Quotation';
            const date = new Date().toISOString().split('T')[0];
            const filename = quotationNo + '_' + date + '.pdf';
            
            // Show loading message
            const button = event.target;
            const originalText = button.textContent;
            button.textContent = '📄 กำลังสร้าง PDF...';
            button.disabled = true;
            
            // Show instruction dialog
            const confirmed = confirm(
                'กรุณาทำตามขั้นตอนนี้:\n\n' +
                '1. กดปุ่ม OK เพื่อเปิดหน้าต่าง Print\n' +
                '2. เลือก "Save as PDF" หรือ "Microsoft Print to PDF"\n' +
                '3. ตั้งชื่อไฟล์: ' + filename + '\n' +
                '4. กด Save เพื่อบันทึกไฟล์ PDF'
            );
            
            if (confirmed) {
                // Use window.print() with custom page settings
                setTimeout(() => {
                    window.print();
                    
                    // Reset button after print dialog
                    setTimeout(() => {
                        button.textContent = originalText;
                        button.disabled = false;
                    }, 1000);
                }, 300);
            } else {
                // User cancelled
                button.textContent = originalText;
                button.disabled = false;
            }
        }