// POS System JavaScript

class POSSystem {
    constructor() {
        this.cart = [];
        this.products = [];
        this.customers = [];
        this.currencies = [];
        this.selectedCurrency = 'CUR_1'; // Default to LAK (Lao Kip)
        this.selectedCurrencyRate = 1;
        this.selectedCurrencySymbol = '₭';
        this.selectedCurrencyRateId = null; // Track rate_id for selected currency
        this.baseCurrency = 'CUR_1'; // All prices stored in LAK - CUR_1
        this.vatEnabled = true; // เพิ่มสถานะ VAT
        this.initializeEventListeners();
        this.loadCurrencies();
    }

    initializeEventListeners() {
        // Search functionality
        const searchInput = document.getElementById('productSearch');
        if (searchInput) {
            searchInput.addEventListener('input', () => this.filterProducts());
        }

        // Category filter
        const categoryFilter = document.getElementById('categoryFilter');
        if (categoryFilter) {
            categoryFilter.addEventListener('change', () => this.filterProducts());
        }

        // Currency filter
        const currencyFilter = document.getElementById('currencyFilter');
        if (currencyFilter) {
            currencyFilter.addEventListener('change', () => this.changeCurrency());
        }

        // Customer selection
        const customerSelect = document.getElementById('customerSelect');
        if (customerSelect) {
            customerSelect.addEventListener('change', () => this.handleCustomerSelection());
        }

        // Modal events
        window.addEventListener('click', (event) => {
            const modal = document.getElementById('orderModal');
            if (event.target === modal) {
                this.closeModal();
            }
        });
    }

    async loadCurrencies() {
        try {
            // เริ่มต้นด้วยเงินกีบเป็นสกุลหลัก (ไม่ต้องดึงจากฐานข้อมูล)
            this.currencies = [
                { id: 'CUR_1', name: 'ກີບລາວ', symbol: '₭', rate: 1, is_base: true, rate_id: null }
            ];
            
            // ดึงข้อมูลสกุลเงินอื่นๆ จากฐานข้อมูล
            const response = await fetch('api/get_currencies.php');
            const data = await response.json();
            
            if (data.success) {
                // เพิ่มสกุลเงินอื่นๆ จากฐานข้อมูล (ข้ามเงินกีบ)
                data.data.forEach(currency => {
                    if (currency.id !== 'CUR_1' && currency.id !== 'CUR_6') {
                        this.currencies.push({
                            id: currency.id,
                            name: currency.name,
                            symbol: currency.symbol,
                            rate: currency.rate,
                            rate_id: currency.rate_id // Store rate_id from API
                        });
                    }
                });
                
                console.log('Loaded currencies from database:', this.currencies);
            } else {
                console.warn('ไม่สามารถดึงข้อมูลสกุลเงินจากฐานข้อมูลได้:', data.message);
            }
        } catch (error) {
            console.error('เกิดข้อผิดพลาดในการดึงข้อมูลสกุลเงิน:', error);
        }
        
        this.populateCurrencySelect();
    }

    populateCurrencySelect() {
        const currencySelect = document.getElementById('currencyFilter');
        if (currencySelect) {
            currencySelect.innerHTML = '';
            // เพิ่มตัวเลือกเงินกีบเป็นตัวแรก (ค่าเริ่มต้น)
            const lakOption = document.createElement('option');
            lakOption.value = 'CUR_1';
            lakOption.textContent = '₭ ກີບລາວ (ສະກຸນຫຼັກ)';
            lakOption.dataset.symbol = '₭';
            lakOption.dataset.rate = '1';
            lakOption.dataset.rateId = '';
            lakOption.selected = true;
            currencySelect.appendChild(lakOption);
            // เพิ่มตัวเลือกสกุลเงินอื่นๆ (เพื่อความสะดวกของลูกค้า) จากฐานข้อมูล
            this.currencies.forEach(currency => {
                if (currency.id !== 'CUR_1') {
                    const option = document.createElement('option');
                    option.value = currency.id;
                    option.textContent = `${currency.symbol} ${currency.name} (ສະດວກສໍາລັບລູກຄ້າ)`;
                    option.dataset.symbol = currency.symbol;
                    option.dataset.rate = currency.rate;
                    option.dataset.rateId = currency.rate_id || '';
                    currencySelect.appendChild(option);
                }
            });
            // ตั้งค่าเริ่มต้นเป็นเงินกีบ
            this.selectedCurrency = 'CUR_1';
            this.selectedCurrencyRate = 1;
            this.selectedCurrencySymbol = '₭';
            this.selectedCurrencyRateId = null;
        }
    }

    setDefaultCurrency() {
        const currencySelect = document.getElementById('currencyFilter');
        if (currencySelect) {
            currencySelect.innerHTML = '<option value="CUR_1">₭ ກີບລາວ (ສະກຸນຫຼັກ)</option>';
            // ตั้งค่าเริ่มต้น
            this.selectedCurrency = 'CUR_1';
            this.selectedCurrencyRate = 1;
            this.selectedCurrencySymbol = '₭';
        }
        console.log('ใช้เงินกีบเป็นค่าเริ่มต้น (ไม่มีสกุลเงินอื่นจากฐานข้อมูล)');
    }

    changeCurrency() {
        const currencySelect = document.getElementById('currencyFilter');
        const selectedOption = currencySelect.options[currencySelect.selectedIndex];
        
        if (selectedOption) {
            this.selectedCurrency = selectedOption.value;
            this.selectedCurrencyRate = parseFloat(selectedOption.dataset.rate) || 1;
            this.selectedCurrencySymbol = selectedOption.dataset.symbol || '₭';
            this.selectedCurrencyRateId = selectedOption.dataset.rateId || null;
            // อัปเดตค่าที่ซ่อนไว้ใน modal
            const hiddenCurrency = document.getElementById('selectedCurrency');
            const hiddenRate = document.getElementById('selectedCurrencyRate');
            const hiddenSymbol = document.getElementById('selectedCurrencySymbol');
            const hiddenRateId = document.getElementById('selectedCurrencyRateId');
            if (hiddenCurrency) hiddenCurrency.value = this.selectedCurrency;
            if (hiddenRate) hiddenRate.value = this.selectedCurrencyRate;
            if (hiddenSymbol) hiddenSymbol.value = this.selectedCurrencySymbol;
            if (hiddenRateId) hiddenRateId.value = this.selectedCurrencyRateId;
            // อัปเดตราคาทั้งหมดบนหน้าจอ (เพื่อแสดงให้ลูกค้าเห็นเป็นสกุลเงินที่ต้องการจ่าย)
            this.updateAllPrices();
        }
    }

    updateAllPrices() {
        // Update product prices
        const productCards = document.querySelectorAll('.product-card');
        productCards.forEach(card => {
            const priceElement = card.querySelector('.product-price');
            
            if (priceElement && priceElement.dataset.basePrice) {
                const basePrice = parseFloat(priceElement.dataset.basePrice);
                const convertedPrice = this.convertPrice(basePrice);
                
                priceElement.innerHTML = `${this.selectedCurrencySymbol} ${this.formatNumber(convertedPrice)}`;

                // Update the onclick handler with converted price and unit info
                // Use a direct function assignment to avoid escaping issues
                const unitId = card.dataset.unitId || '';
                const unitName = card.dataset.unitName || '';
                const remarkId = card.dataset.remarkId || '';
                const detailProductId = card.dataset.detailProductId || '';
                const detailText = card.dataset.detailText || '';
                card.onclick = function() { addToCart(card.dataset.id, card.dataset.name, convertedPrice, unitId, unitName, remarkId, detailProductId, detailText); };
            }
        });
        
        // Update cart display
        this.updateCartDisplay();
        
        // Update discount symbol
        const discountSymbol = document.getElementById('discountSymbol');
        if (discountSymbol) {
            discountSymbol.textContent = this.selectedCurrencySymbol;
        }
        
        // Update currency note
        const currencyNote = document.getElementById('currencyNote');
        if (currencyNote) {
            const currencyName = this.getCurrencyName(this.selectedCurrency);
            currencyNote.textContent = `ລາຄາເປັນ: ${currencyName}`;
        }
    }

    convertPrice(basePriceLAK) {
        // แปลงจากเงินกีบ (LAK) เป็นสกุลเงินที่เลือก
        // หากเป็นเงินกีบ (CUR_1) ให้คืนค่าเดิม
        if (this.selectedCurrency === 'CUR_1' || this.selectedCurrencyRate === 1) {
            return basePriceLAK;
        }
        
        // สำหรับสกุลเงินอื่น: LAK ÷ อัตราแลกเปลี่ยน
        // เช่น: 200,000 LAK ÷ 663 (THB rate) = 301.66 THB
        return basePriceLAK / this.selectedCurrencyRate;
    }

    getCurrencyName(currencyId) {
        const currency = this.currencies.find(c => c.id === currencyId);
        return currency ? currency.name : 'ກີບລາວ';
    }

    filterProducts() {
        const searchTerm = document.getElementById('productSearch')?.value.toLowerCase() || '';
        const selectedCategory = document.getElementById('categoryFilter')?.value || '';
        
        const productCards = document.querySelectorAll('.product-card');
        
        productCards.forEach(card => {
            const productName = card.dataset.name?.toLowerCase() || '';
            const productCategory = card.dataset.category || '';
            
            const matchesSearch = productName.includes(searchTerm);
            const matchesCategory = !selectedCategory || productCategory === selectedCategory;
            
            if (matchesSearch && matchesCategory) {
                card.style.display = 'flex';
            } else {
                card.style.display = 'none';
            }
        });
    }

    addToCart(productId, productName, productPrice, unitId = '', unitName = '', remarkId = '', detailProductId = '', detailText = '') {
        // Check if item already exists in cart
        const existingItem = this.cart.find(item => item.id === productId);
        
        // เก็บราคาในสกุลเงินฐาน (LAK) สำหรับการคำนวณ
        // หากราคาที่ส่งมาถูกแปลงแล้ว ต้องแปลงกลับเป็น LAK
        let basePriceLAK;
        if (this.selectedCurrency === 'CUR_1' || this.selectedCurrencyRate === 1) {
            // หากเป็นเงินกีบ ใช้ราคาเดิม
            basePriceLAK = parseFloat(productPrice);
        } else {
            // หากเป็นสกุลเงินอื่น แปลงกลับเป็น LAK
            basePriceLAK = parseFloat(productPrice) * this.selectedCurrencyRate;
        }
        
        if (existingItem) {
            existingItem.quantity += 1;
        } else {
            this.cart.push({
                id: productId,
                name: productName,
                basePriceLAK: basePriceLAK, // Store in LAK
                price: parseFloat(productPrice), // Display price in selected currency
                quantity: 1,
                unitId: unitId || '',
                unitName: unitName || '',
                remarkId: remarkId || '',
                detailProductId: detailProductId || '',
                detailText: detailText || ''
            });
        }
        
        this.updateCartDisplay();
        this.showAddToCartAnimation();
    }

    removeFromCart(productId) {
        this.cart = this.cart.filter(item => item.id !== productId);
        this.updateCartDisplay();
    }

    updateQuantity(productId, newQuantity) {
        if (newQuantity <= 0) {
            this.removeFromCart(productId);
            return;
        }
        
        const item = this.cart.find(item => item.id === productId);
        if (item) {
            item.quantity = parseInt(newQuantity);
            this.updateCartDisplay();
        }
    }

    // Toggle VAT functionality
    toggleVAT() {
        this.vatEnabled = !this.vatEnabled;
        
        const vatButton = document.getElementById('vatToggle');
        const vatRow = document.getElementById('vatRow');
        const modalVatRow = document.getElementById('modalVatRow');
        
        if (this.vatEnabled) {
            vatButton.classList.add('active');
            vatButton.innerHTML = '<i class="fas fa-percent"></i> VAT';
            vatRow.classList.remove('hidden');
            modalVatRow.classList.remove('hidden');
        } else {
            vatButton.classList.remove('active');
            vatButton.innerHTML = '<i class="fas fa-percent"></i> ບໍ່ເອົາ VAT';
            vatRow.classList.add('hidden');
            modalVatRow.classList.add('hidden');
        }
        
        this.updateCartDisplay();
    }

    updateCartDisplay() {
        const cartItemsContainer = document.getElementById('cartItems');
        
        if (this.cart.length === 0) {
            cartItemsContainer.innerHTML = `
                <div class="empty-cart">
                    <i class="fas fa-shopping-cart"></i>
                    <p>ບໍ່ມີສິນຄ້າໃນຕະກຮ້າ</p>
                </div>
            `;
        } else {
            cartItemsContainer.innerHTML = this.cart.map(item => {
                // Convert price to selected currency for display
                const displayPrice = this.convertPrice(item.basePriceLAK);
                return `
                <div class="cart-item">
                    <div class="cart-item-header">
                        <div class="cart-item-name">${item.name} ${item.unitName ? `<small class="unit-text">(${item.unitName})</small>` : ''}</div>
                        <button class="cart-item-remove" onclick="posSystem.removeFromCart('${item.id}')">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="cart-item-controls">
                        <div class="quantity-controls">
                            <button class="qty-btn" onclick="posSystem.updateQuantity('${item.id}', ${item.quantity - 1})" ${item.quantity <= 1 ? 'disabled' : ''}>
                                <i class="fas fa-minus"></i>
                            </button>
                            <span class="cart-item-quantity">${item.quantity}</span>
                            <button class="qty-btn" onclick="posSystem.updateQuantity('${item.id}', ${item.quantity + 1})">
                                <i class="fas fa-plus"></i>
                            </button>
                        </div>
                        <div class="cart-item-price">${this.selectedCurrencySymbol} ${this.formatNumber(displayPrice * item.quantity)}</div>
                    </div>
                </div>
            `}).join('');
        }
        
        this.updateTotal();
    }

    updateTotal() {
        // Calculate in LAK first, then convert to selected currency
        const subtotalLAK = this.cart.reduce((sum, item) => sum + (item.basePriceLAK * item.quantity), 0);
        const discount = parseFloat(document.getElementById('discountInput')?.value || 0);
        
        // Convert discount to LAK if in different currency
        let discountLAK;
        if (this.selectedCurrency === 'CUR_1' || this.selectedCurrencyRate === 1) {
            // หากเป็นเงินกีบ ใช้ส่วนลดเดิม
            discountLAK = discount;
        } else {
            // หากเป็นสกุลเงินอื่น แปลงเป็น LAK
            discountLAK = discount * this.selectedCurrencyRate;
        }
        
        const discountedAmountLAK = subtotalLAK - discountLAK;
        
        // คำนวณ VAT ตามสถานะที่เลือก
        const vatAmountLAK = this.vatEnabled ? discountedAmountLAK * 0.1 : 0; // 10% VAT หรือ 0
        const grandTotalLAK = discountedAmountLAK + vatAmountLAK;
        
        // Convert to selected currency for display
        const subtotal = this.convertPrice(subtotalLAK);
        const vatAmount = this.convertPrice(vatAmountLAK);
        const grandTotal = this.convertPrice(grandTotalLAK);
        
        document.getElementById('subtotal').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(subtotal)}`;
        document.getElementById('vatAmount').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(vatAmount)}`;
        document.getElementById('grandTotal').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(grandTotal)}`;
    }

    clearCart() {
        if (this.cart.length === 0) {
            Swal.fire({
                icon: 'info',
                title: 'ແຈ້ງເຕືອນ',
                text: 'ຕະກຮ້າວ່າງເປົ່າແລ້ວ',
                confirmButtonText: 'ຕົກລົງ'
            });
            return;
        }

        Swal.fire({
            title: 'ຢືນຢັນການລ້າງ',
            text: "ຕ້ອງການລ້າງສິນຄ້າທັງຫມົດໃນຕະກຮ້າບໍ່?",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'ລ້າງທັງຫມົດ',
            cancelButtonText: 'ຍົກເລີກ'
        }).then((result) => {
            if (result.isConfirmed) {
                this.cart = [];
                this.updateCartDisplay();
                
                Swal.fire({
                    icon: 'success',
                    title: 'ສຳເລັດ',
                    text: 'ລ້າງຕະກຮ້າແລ້ວ',
                    timer: 1500,
                    showConfirmButton: false
                });
            }
        });
    }

    confirmOrder() {
        if (this.cart.length === 0) {
            Swal.fire({
                icon: 'warning',
                title: 'ແຈ້ງເຕືອນ',
                text: 'ກະລຸນາເລືອກສິນຄ້າກ່ອນ',
                confirmButtonText: 'ຕົກລົງ'
            });
            return;
        }
        
        // Update modal summary
        this.updateModalSummary();
        
        // Show modal
        document.getElementById('orderModal').style.display = 'block';
    }

    updateModalSummary() {
        const modalOrderItems = document.getElementById('modalOrderItems');
        
        // Calculate in LAK first
        const subtotalLAK = this.cart.reduce((sum, item) => sum + (item.basePriceLAK * item.quantity), 0);
        const discount = parseFloat(document.getElementById('discountInput')?.value || 0);
        
        // Convert discount to LAK if in different currency
        let discountLAK;
        if (this.selectedCurrency === 'CUR_1' || this.selectedCurrency === 'CUR_6' || this.selectedCurrencyRate === 1) {
            discountLAK = discount;
        } else {
            discountLAK = discount * this.selectedCurrencyRate;
        }
        
        const discountedAmountLAK = subtotalLAK - discountLAK;
        const vatAmountLAK = this.vatEnabled ? discountedAmountLAK * 0.1 : 0; // คำนวณ VAT ตามสถานะ
        const grandTotalLAK = discountedAmountLAK + vatAmountLAK;
        
        // Convert to selected currency for display
        const subtotal = this.convertPrice(subtotalLAK);
        const vatAmount = this.convertPrice(vatAmountLAK);
        const grandTotal = this.convertPrice(grandTotalLAK);
        
        modalOrderItems.innerHTML = this.cart.map(item => {
            const displayPrice = this.convertPrice(item.basePriceLAK);
            return `
            <div class="modal-order-item">
                <span>${item.name} ${item.unitName ? `<small class=\"unit-text\">(${item.unitName})</small>` : ''} × ${item.quantity}</span>
                <span>${this.selectedCurrencySymbol} ${this.formatNumber(displayPrice * item.quantity)}</span>
            </div>
        `}).join('');
        
        document.getElementById('modalSubtotal').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(subtotal)}`;
        document.getElementById('modalDiscount').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(discount)}`;
        document.getElementById('modalVat').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(vatAmount)}`;
        document.getElementById('modalTotal').textContent = `${this.selectedCurrencySymbol} ${this.formatNumber(grandTotal)}`;
    }

    closeModal() {
        document.getElementById('orderModal').style.display = 'none';
        document.getElementById('newCustomerForm').style.display = 'none';
    }

    showNewCustomerForm() {
        const form = document.getElementById('newCustomerForm');
        form.style.display = form.style.display === 'none' ? 'block' : 'none';
    }

    handleCustomerSelection() {
        const select = document.getElementById('customerSelect');
        const selectedOption = select.options[select.selectedIndex];
        
        if (selectedOption && selectedOption.value) {
            // Hide new customer form if customer is selected
            document.getElementById('newCustomerForm').style.display = 'none';
        }
    }

    addNewCustomer() {
        const name = document.getElementById('newCustomerName').value.trim();
        const tel = document.getElementById('newCustomerTel').value.trim();
        const address = document.getElementById('newCustomerAddress').value.trim();
        
        if (!name) {
            Swal.fire({
                icon: 'warning',
                title: 'ແຈ້ງເຕືອນ',
                text: 'ກະລຸນາໃສ່ຊື່ລູກຄ້າ',
                confirmButtonText: 'ຕົກລົງ'
            });
            return;
        }
        
        // Add customer to database (let trigger generate cus_id)
        fetch('api/add_customer.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                cus_name: name,
                cus_tel: tel,
                cus_address: address
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Add to customer select
                const customerSelect = document.getElementById('customerSelect');
                const option = document.createElement('option');
                option.value = data.customer_id;
                option.textContent = `${data.customer_name} - ${data.customer_tel || 'ບໍ່ມີເບີ'}`;
                option.selected = true;
                customerSelect.appendChild(option);
                
                // Clear form
                document.getElementById('newCustomerName').value = '';
                document.getElementById('newCustomerTel').value = '';
                document.getElementById('newCustomerAddress').value = '';
                document.getElementById('newCustomerForm').style.display = 'none';
                
                Swal.fire({
                    icon: 'success',
                    title: 'ສຳເລັດ',
                    text: 'ເພີ່ມລູກຄ້າໃໝ່ແລ້ວ',
                    timer: 1500,
                    showConfirmButton: false
                });
            } else {
                throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'ຜິດພາດ',
                text: 'ບໍ່ສາມາດເພີ່ມລູກຄ້າໄດ້',
                confirmButtonText: 'ຕົກລົງ'
            });
        });
    }

    submitOrder() {
        const customerSelect = document.getElementById('customerSelect');
        const section = document.getElementById('sectionInput').value.trim();
        
        // Validate form
        if (!customerSelect.value && !document.getElementById('newCustomerName').value) {
            Swal.fire({
                icon: 'warning',
                title: 'ແຈ້ງເຕືອນ',
                text: 'ກະລຸນາເລືອກລູກຄ້າຫຼືເພີ່ມລູກຄ້າໃໝ່',
                confirmButtonText: 'ຕົກລົງ'
            });
            return;
        }

        // Prepare order data
        const subtotalLAK = this.cart.reduce((sum, item) => sum + (item.basePriceLAK * item.quantity), 0);
        const discount = parseFloat(document.getElementById('discountInput')?.value || 0);
        
        // Convert discount to LAK if in different currency
        let discountLAK;
        if (this.selectedCurrency === 'CUR_1' || this.selectedCurrency === 'CUR_6' || this.selectedCurrencyRate === 1) {
            discountLAK = discount;
        } else {
            discountLAK = discount * this.selectedCurrencyRate;
        }
        
        const discountedAmountLAK = subtotalLAK - discountLAK;
        const vatAmountLAK = this.vatEnabled ? discountedAmountLAK * 0.1 : 0; // คำนวณ VAT ตามสถานะ
        const grandTotalLAK = discountedAmountLAK + vatAmountLAK;
        
        // Find the first remark_id from cart items
        const remarkId = this.cart.find(item => item.remarkId)?.remarkId || null;
        // Find the first detail_product_id from cart items
        const detailProductId = this.cart.find(item => item.detailProductId)?.detailProductId || null;

        const orderData = {
            customer_id: customerSelect.value,
            section: section,
            remark_id: remarkId,
            detail_product_id: detailProductId,
            order_items: this.cart.map(item => ({
                ...item,
                price_lak: item.basePriceLAK // Ensure we send LAK prices
            })),
            subtotal_lak: subtotalLAK,
            discount_lak: discountLAK,
            vat_amount_lak: vatAmountLAK,
            total_amount_lak: grandTotalLAK,
            currency_id: this.selectedCurrency,
            currency_rate: this.selectedCurrencyRate,
            currency_symbol: this.selectedCurrencySymbol,
            rate_id: this.selectedCurrencyRateId || null
        };

        // Show loading
        Swal.fire({
            title: 'ກໍາລັງດໍາເນີນການ...',
            text: 'ກະລຸນາລໍຖ້າ',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });

        // Submit order
        fetch('api/submit_order.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(orderData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                this.closeModal();
                this.cart = [];
                this.updateCartDisplay();
                
                Swal.fire({
                    icon: 'success',
                    title: 'ສຳເລັດ!',
                    text: `ສ້າງອໍເດີເລກທີ ${data.order_id} ແລ້ວ`,
                    showCancelButton: true,
                    confirmButtonText: 'ພິມໃບບິນ',
                    cancelButtonText: 'ຕົກລົງ'
                }).then((result) => {
                    if (result.isConfirmed) {
                        window.open(`print_quotation.php?order_id=${data.order_id}`, '_blank');
                    }
                });
            } else {
                throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'ຜິດພາດ',
                text: 'ບໍ່ສາມາດສ້າງອໍເດີໄດ້',
                confirmButtonText: 'ຕົກລົງ'
            });
        });
    }

    saveQuote() {
        if (this.cart.length === 0) {
            Swal.fire({
                icon: 'warning',
                title: 'ແຈ້ງເຕືອນ',
                text: 'ກະລຸນາເລືອກສິນຄ້າກ່ອນ',
                confirmButtonText: 'ຕົກລົງ'
            });
            return;
        }

        Swal.fire({
            title: 'ບັນທຶກໃບສະເຫນີ',
            text: 'ຕ້ອງການບັນທຶກເປັນໃບສະເຫນີລາຄາບໍ່?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'ບັນທຶກ',
            cancelButtonText: 'ຍົກເລີກ'
        }).then((result) => {
            if (result.isConfirmed) {
                // Save as quotation logic here
                const quoteId = 'QUO' + Date.now();
                
                Swal.fire({
                    icon: 'success',
                    title: 'ສຳເລັດ!',
                    text: `ບັນທຶກໃບສະເຫນີເລກທີ ${quoteId} ແລ້ວ`,
                    showCancelButton: true,
                    confirmButtonText: 'ພິມໃບສະເຫນີ',
                    cancelButtonText: 'ຕົກລົງ'
                }).then((printResult) => {
                    if (printResult.isConfirmed) {
                        window.open(`print_quotation.php?quote_id=${quoteId}`, '_blank');
                    }
                });
            }
        });
    }

    showAddToCartAnimation() {
        // Simple animation feedback
        const cartIcon = document.querySelector('.cart-header i');
        if (cartIcon) {
            cartIcon.style.transform = 'scale(1.3)';
            cartIcon.style.color = '#28a745';
            setTimeout(() => {
                cartIcon.style.transform = 'scale(1)';
                cartIcon.style.color = '';
            }, 300);
        }
    }

    formatNumber(num) {
        // ตรวจสอบว่าเป็นเงินดอลลาร์หรือไม่
        if (this.selectedCurrency === 'CUR_3') { // USD
            return new Intl.NumberFormat('lo-LA', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(num);
        } else {
            // สกุลเงินอื่นไม่แสดงทศนิยม
            return new Intl.NumberFormat('lo-LA', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
            }).format(Math.round(num));
        }
    }
}

// Global functions for onclick events
function addToCart(productId, productName, productPrice, unitId, unitName, remarkId, detailProductId, detailText) {
    posSystem.addToCart(productId, productName, productPrice, unitId, unitName, remarkId, detailProductId, detailText);
}

function clearCart() {
    posSystem.clearCart();
}

function confirmOrder() {
    posSystem.confirmOrder();
}

function saveQuote() {
    posSystem.saveQuote();
}

function closeModal() {
    posSystem.closeModal();
}

function showNewCustomerForm() {
    posSystem.showNewCustomerForm();
}

function addNewCustomer() {
    posSystem.addNewCustomer();
}

function submitOrder() {
    posSystem.submitOrder();
}

function updateTotal() {
    posSystem.updateTotal();
}

function toggleVAT() {
    posSystem.toggleVAT();
}

// Initialize POS System
let posSystem;

function initPOS() {
    posSystem = new POSSystem();
}