    // home.js - Sokxay Bill Dashboard JavaScript - Sidebar Layout

    document.addEventListener('DOMContentLoaded', function() {
        // เริ่มต้น animations
        initAnimations();

        // แสดงเวลาปัจจุบัน
        updateClock();
        setInterval(updateClock, 1000);

        // เริ่มต้น dashboard specific features
        initDashboardFeatures();

        // เรียกอัปเดตแจ้งเตือนและจำนวนแบบ real-time
        fetchAndUpdateNotifications();
        setInterval(fetchAndUpdateNotifications, 15000); // ทุก 15 วินาที
    });
    // ฟังก์ชันดึงข้อมูลแจ้งเตือนและจำนวนแบบ real-time
    let latestNotifications = null; // store last response for mark/read actions
    // Ensure markSingleNotification is globally available for click handlers
    window.markSingleNotification = function(orderId, callback) {
        if (!orderId) {
            if (typeof callback === 'function') callback();
            return;
        }
        fetch('api/mark_notifications_read.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ order_ids: [orderId] })
        }).then(response => {
            if (!response.ok) {
                return response.text().then(text => { console.error('mark single API error', response.status, text); });
            }
            return response.json();
        }).then(json => {
            if (json && json.success) {
                // refresh lists and then callback
                fetchAndUpdateNotifications();
                if (typeof callback === 'function') callback();
            } else {
                if (typeof callback === 'function') callback();
            }
        }).catch(err => {
            console.error('markSingleNotification error:', err);
            if (typeof callback === 'function') callback();
        });
    };

    

    function fetchAndUpdateNotifications() {
        fetch('api/get_notifications.php')
            .then(response => {
                if (!response.ok) {
                    return response.text().then(text => {
                        console.error('Notification API error:', response.status, text);
                        throw new Error('server_error');
                    });
                }
                return response.text();
            })
            .then(text => {
                if (!text) throw new Error('empty_response');
                let data;
                try {
                    data = JSON.parse(text);
                } catch (err) {
                    console.error('Failed to parse JSON from get_notifications:', err, text);
                    throw err;
                }
                if (data && !data.error) {
                    // store for later use (marking read)
                    latestNotifications = data;
                    // update UI
                    updateNotificationCount(data.unread_count || 0);
                    updateNotificationList(data.unread || [], data.recent_read || []);
                } else if (data && data.error) {
                    console.error('get_notifications returned error:', data);
                }
            })
            .catch(err => {
                console.error('fetchAndUpdateNotifications error:', err);
            });
    }

    // อัปเดตเลขแจ้งเตือน (เพิ่ม badge หรือเลขบนไอคอนกระดิ่ง)
    function updateNotificationCount(count) {
        let bell = document.querySelector('.notifications .fa-bell');
        if (!bell) return;
        let badge = bell.parentElement.querySelector('.notification-badge');
        if (!badge) {
            badge = document.createElement('span');
            badge.className = 'notification-badge';
            bell.parentElement.appendChild(badge);
        }
        badge.textContent = count > 0 ? count : '';
        badge.style.display = count > 0 ? 'inline-block' : 'none';
    }

    // อัปเดตรายการแจ้งเตือนใน dropdown
    function updateNotificationList(unread = [], recentRead = []) {
        const list = document.querySelector('.notification-list');
        if (!list) return;
        // ensure arrays
        if (!Array.isArray(unread)) {
            console.warn('updateNotificationList: unread is not an array', unread);
            unread = [];
        }
        if (!Array.isArray(recentRead)) {
            console.warn('updateNotificationList: recentRead is not an array', recentRead);
            recentRead = [];
        }
        list.innerHTML = '';

        // Unread section
        if (unread && unread.length > 0) {
            const header = document.createElement('div');
            header.className = 'notification-section-header';
            header.style.display = 'flex';
            header.style.justifyContent = 'space-between';
            header.style.alignItems = 'center';
            header.innerHTML = `<strong>ແຈ້ງເຕືອນໃໝ່ (${unread.length})</strong>`;
            // add "Mark all as read" button
            const markAllBtn = document.createElement('button');
            markAllBtn.type = 'button';
            markAllBtn.textContent = 'Mark all as read';
            markAllBtn.style.fontSize = '0.85rem';
            markAllBtn.style.padding = '6px 8px';
            markAllBtn.style.border = 'none';
            markAllBtn.style.background = '#2ecc71';
            markAllBtn.style.color = '#fff';
            markAllBtn.style.borderRadius = '4px';
            markAllBtn.style.cursor = 'pointer';
            markAllBtn.addEventListener('click', function(ev) {
                ev.stopPropagation();
                // mark unread items as read
                markNotificationsRead();
            });
            header.appendChild(markAllBtn);
            list.appendChild(header);

            unread.forEach(order => {
                const isInvoice = (order.type && order.type.toString().toLowerCase().includes('invoice'));
                const linkPage = isInvoice ? 'print_invoice.php' : 'print_quotation.php';
                const docType = isInvoice ? '(ໃບເກັບເງິນ)' : '(ໃບສະເໜີ)';
                // Use read_id if available (for separating invoice/quotation read status), else fallback to order_id
                const readId = order.read_id || order.order_id;
                
                // Determine display ID
                const displayId = isInvoice 
                    ? (order.invoice_no || order.order_id) 
                    : (order.quotation_no || order.order_id);

                const a = document.createElement('a');
                a.href = linkPage + '?order_id=' + encodeURIComponent(order.order_id);
                a.dataset.readId = readId; // Store for mark-all fallback
                a.className = 'notification-item approved unread';
                a.style.textDecoration = 'none';
                a.style.color = 'inherit';
                a.style.display = 'flex';

                const isRejected = (order.type && order.type.toString().toLowerCase().includes('reject')) || !!order.message;

                if (isRejected) {
                    // Rejection-style notification (red icon, show message)
                    a.innerHTML = `
                        <i class="fas fa-times-circle" style="color:#e74c3c; margin-right:10px; align-self:center;"></i>
                        <div class="notification-content">
                            <strong style="color:#e74c3c">✗ ປະຕິເສດ ${docType} - ${displayId}</strong>
                            <p>${order.message ? escapeHtml(order.message) : (order.section || 'ບໍ່ມີຫົວຂໍ້')} | ${order.cus_name || ''}</p>
                            <span style="font-size: 0.75rem; color: #999;">${formatDate(order.order_date)}</span>
                        </div>
                    `;
                } else {
                    // Approved-style notification (green dot)
                    a.innerHTML = `
                        <span style="width:12px;height:12px;border-radius:50%;background:#2ecc71;display:inline-block;margin-right:10px;flex:0 0 12px;align-self:center;"></span>
                        <i class="fas fa-check-circle"></i>
                        <div class="notification-content">
                            <strong>✓ ອະນຸມັດແລ້ວ ${docType} - ${displayId}</strong>
                            <p>${order.section || 'ບໍ່ມີຫົວຂໍ້'} | ${order.cus_name}</p>
                            <span style="font-size: 0.75rem; color: #999;">${formatDate(order.order_date)}</span>
                        </div>
                    `;
                }

                // intercept click to mark single item as read before navigating
                a.addEventListener('click', function(evt) {
                    evt.preventDefault();
                    const targetHref = a.getAttribute('href');
                    markSingleNotification(readId, function() {
                        // after marked, navigate
                        window.location.href = targetHref;
                    });
                });
                list.appendChild(a);
            });
        }

        // Recent read section
        if (recentRead && recentRead.length > 0) {
            const header2 = document.createElement('div');
            header2.className = 'notification-section-header';
            header2.style.marginTop = '8px';
            header2.innerHTML = `<strong>ໄດ້ອ່ານແລ້ວ</strong>`;
            list.appendChild(header2);

            recentRead.forEach(order => {
                const isInvoice = (order.type && order.type.toString().toLowerCase().includes('invoice'));
                const isRejected = (order.type && order.type.toString().toLowerCase().includes('rejected'));
                const linkPage = isInvoice ? 'print_invoice.php' : 'print_quotation.php';
                const docType = isInvoice ? '(ໃບເກັບເງິນ)' : '(ໃບສະເໜີ)';
                
                // Determine display ID
                const displayId = isInvoice 
                    ? (order.invoice_no || order.order_id) 
                    : (order.quotation_no || order.order_id);

                // Determine status text and icon
                const statusIcon = isRejected ? '✗' : '✓';
                const statusText = isRejected ? 'ປະຕິເສດ' : 'ອະນຸມັດແລ້ວ';
                const statusColor = isRejected ? 'color:#e74c3c' : '';

                const a = document.createElement('a');
                a.href = linkPage + '?order_id=' + encodeURIComponent(order.order_id);
                a.className = 'notification-item read';
                a.style.textDecoration = 'none';
                a.style.color = 'inherit';
                a.style.display = 'flex';
                a.innerHTML = `
                    <i class="fas fa-check-circle" style="opacity:0.5"></i>
                    <div class="notification-content" style="opacity:0.6">
                        <strong style="${statusColor}">${statusIcon} ${statusText} ${docType} - ${displayId}</strong>
                        <p>${order.section || 'ບໍ່ມີຫົວຂໍ້'} | ${order.cus_name}</p>
                        <span style="font-size: 0.75rem; color: #999;">${formatDate(order.order_date)}</span>
                    </div>
                `;
                list.appendChild(a);
            });
        }

        if ((unread && unread.length === 0) && (!recentRead || recentRead.length === 0)) {
            const div = document.createElement('div');
            div.className = 'notification-item';
            div.innerHTML = '<p style="text-align: center; color: #999;">ບໍ່ມີການແຈ້ງເຕືອນ</p>';
            list.appendChild(div);
        }
    }

    // ฟอร์แมตวันที่ (yyyy-mm-dd => dd/mm/yyyy)
    function formatDate(dateStr) {
        if (!dateStr) return '';
        const d = new Date(dateStr);
        if (isNaN(d)) return dateStr;
        const day = d.getDate().toString().padStart(2, '0');
        const month = (d.getMonth() + 1).toString().padStart(2, '0');
        const year = d.getFullYear();
        return `${day}/${month}/${year}`;
    }

    // escape HTML to avoid injection in notification messages
    function escapeHtml(str) {
        if (!str && str !== 0) return '';
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    function initAnimations() {
        // Animation สำหรับ stats cards
        const statCards = document.querySelectorAll('.stat-card');
        statCards.forEach((card, index) => {
            card.style.opacity = '0';
            card.style.transform = 'translateY(20px)';
            
            setTimeout(() => {
                card.style.transition = 'all 0.5s ease';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, 200 + (index * 100));
        });

        // Animation สำหรับ action cards
        const actionCards = document.querySelectorAll('.action-card');
        actionCards.forEach((card, index) => {
            card.style.opacity = '0';
            card.style.transform = 'translateY(20px)';
            
            setTimeout(() => {
                card.style.transition = 'all 0.5s ease';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, 600 + (index * 100));
        });

        // Animation สำหรับ activity items
        const activityItems = document.querySelectorAll('.activity-item');
        activityItems.forEach((item, index) => {
            item.style.opacity = '0';
            item.style.transform = 'translateX(20px)';
            
            setTimeout(() => {
                item.style.transition = 'all 0.5s ease';
                item.style.opacity = '1';
                item.style.transform = 'translateX(0)';
            }, 1000 + (index * 100));
        });
    }

    function initDashboardFeatures() {
        // Notification dropdown toggle
        const notifications = document.querySelector('.notifications');
        const notificationDropdown = document.querySelector('.notification-dropdown');
        
        if (notifications && notificationDropdown) {
            notifications.addEventListener('click', function(e) {
                e.stopPropagation();
                const isOpen = notificationDropdown.style.display === 'block';
                notificationDropdown.style.display = isOpen ? 'none' : 'block';
                // NOTE: Do NOT auto-mark on bell click. User must explicitly mark or click items.
            });

            // Close dropdown when clicking outside
            document.addEventListener('click', function(e) {
                if (!notifications.contains(e.target)) {
                    notificationDropdown.style.display = 'none';
                }
            });
        }
    // เรียก API mark_notifications_read.php เพื่อ mark แจ้งเตือนว่าอ่านแล้ว
    function markNotificationsRead() {
        // Prefer to use latestNotifications.unread if available
        let orderIds = [];
        if (latestNotifications && Array.isArray(latestNotifications.unread)) {
            orderIds = latestNotifications.unread.map(o => o.read_id || o.order_id).filter(Boolean);
        }

        // Fallback to DOM parsing if needed
        if (orderIds.length === 0) {
            const items = document.querySelectorAll('.notification-list .notification-item.approved');
            orderIds = Array.from(items).map(a => {
                // Use data-read-id if available
                if (a.dataset.readId) return a.dataset.readId;
                
                const href = a.getAttribute('href');
                const match = href && href.match(/order_id=([^&]+)/);
                return match ? decodeURIComponent(match[1]) : null;
            }).filter(Boolean);
        }
        if (orderIds.length === 0) return;
        fetch('api/mark_notifications_read.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ order_ids: orderIds })
        }).then(response => {
            if (!response.ok) {
                return response.text().then(text => { console.error('mark read API error', response.status, text); });
            }
            return response.json();
        }).then(json => {
            if (json && json.success) {
                // update UI: reset badge and refresh lists so unread move to recent_read
                resetNotificationBadge();
                // Refresh from server to get latest read/unread lists
                fetchAndUpdateNotifications();
            }
        }).catch(err => {
            console.error('markNotificationsRead error:', err);
        });
    }

    
    // รีเซ็ต badge แจ้งเตือน (ซ่อน badge และ set เป็น 0)
    function resetNotificationBadge() {
        let bell = document.querySelector('.notifications .fa-bell');
        if (!bell) return;
        let badge = bell.parentElement.querySelector('.notification-badge');
        if (badge) {
            badge.textContent = '';
            badge.style.display = 'none';
        }
    }

        // Animate stats numbers when page loads
        setTimeout(() => {
            animateStatNumbers();
        }, 1000);
    }

    function updateClock() {
        const now = new Date();
        const options = { 
            weekday: 'short',
            day: 'numeric',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        };
        
        const clockElement = document.getElementById('current-time');
        if (clockElement) {
            // Format: "ພຸດ, 13 ພຶ 2568, 14:30"
            const dateStr = now.toLocaleDateString('lo-LA', options);
            clockElement.textContent = dateStr;
        }
    }

    // Counter animation for stats
    function animateValue(element, start, end, duration) {
        const range = end - start;
        const startTime = performance.now();
        
        function step(currentTime) {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Easing function (easeOutQuart)
            const easeProgress = 1 - Math.pow(1 - progress, 4);
            
            const current = Math.floor(start + (range * easeProgress));
            element.textContent = current.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(step);
            }
        }
        
        requestAnimationFrame(step);
    }

    // Animate stats numbers
    function animateStatNumbers() {
        const statNumbers = document.querySelectorAll('.stat-number');
        statNumbers.forEach((stat, index) => {
            const text = stat.textContent;
            if (text.includes('₭')) {
                // For money amounts
                animateValue(stat, 0, 45890000, 2000);
                setTimeout(() => {
                    stat.textContent = '₭ ' + stat.textContent.replace(/\B(?=(\d{3})+(?!\d))/g, ',');
                }, 2000);
            } else if (text.includes('%')) {
                // For percentages
                setTimeout(() => {
                    stat.textContent = '+12.5%';
                }, index * 200);
            } else {
                // For regular numbers
                const targetValue = index === 0 ? 24 : 156;
                animateValue(stat, 0, targetValue, 1500);
            }
        });
    }