<?php
session_start();

// ตรวจสอบการล็อกอิน
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

// เชื่อมต่อฐานข้อมูล
require_once 'includes/db_con.php';

// ดึงข้อมูล orders ทั้งหมด
$orders_query = "SELECT o.*, c.cus_name, cur.currencys_symbol, u.full_name as user_name
                 FROM orders o
                 LEFT JOIN customers c ON o.cus_id = c.cus_id
                 LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id
                 LEFT JOIN user u ON o.user_id = u.id
                 ORDER BY o.order_date DESC, o.id DESC";
$orders_result = $conn->query($orders_query);
?>

<!DOCTYPE html>
<html lang="lo">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ປະຫວັດ Orders - Sokxay Bill</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="/sokxay/wayha.png">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Custom CSS -->
    <link rel="stylesheet" href="css/home.css">
    
    <style>
        .orders-container {
            padding: 2rem;
        }

        .page-header {
            margin-bottom: 2rem;
        }

        .page-header h1 {
            font-size: 1.75rem;
            color: var(--text-dark);
            margin-bottom: 0.5rem;
        }

        .page-header p {
            color: var(--text-muted);
            font-size: 0.95rem;
        }

        .filters-bar {
            background: var(--bg-white);
            padding: 1.5rem;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            margin-bottom: 1.5rem;
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
            align-items: center;
        }

        .filter-group {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .filter-group label {
            font-weight: 500;
            color: var(--text-dark);
            font-size: 0.9rem;
        }

        .filter-group select,
        .filter-group input {
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: 6px;
            font-size: 0.9rem;
            min-width: 150px;
        }

        .orders-table {
            background: var(--bg-white);
            border-radius: 8px;
            border: 1px solid var(--border-color);
            overflow: hidden;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        table thead {
            background: var(--secondary-color);
        }

        table th {
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: var(--text-dark);
            font-size: 0.9rem;
            border-bottom: 2px solid var(--border-color);
        }

        table td {
            padding: 1rem;
            border-bottom: 1px solid var(--border-color);
            font-size: 0.9rem;
        }

        table tbody tr {
            transition: background 0.2s ease;
        }

        table tbody tr:hover {
            background: var(--secondary-color);
        }

        .status-badge {
            padding: 0.35rem 0.75rem;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: 500;
            display: inline-block;
        }

        .status-badge.approved {
            background: #d4edda;
            color: #155724;
        }

        .status-badge.pending {
            background: #fff3cd;
            color: #856404;
        }

        .status-badge.rejected {
            background: #f8d7da;
            color: #721c24;
        }

        .status-badge.none {
            background: #e9ecef;
            color: #6c757d;
        }

        /* Modal for print options */
        .print-modal {
            position: fixed;
            inset: 0;
            display: none;
            align-items: center;
            justify-content: center;
            background: rgba(0,0,0,0.5);
            z-index: 9999;
        }

        .print-modal.active {
            display: flex;
        }

        .print-modal .modal-card {
            background: #fff;
            border-radius: 8px;
            padding: 1.25rem;
            width: 320px;
            max-width: 90%;
            box-shadow: 0 8px 24px rgba(0,0,0,0.15);
            text-align: center;
        }

        .print-modal h3 { margin: 0 0 0.75rem 0; }

        .print-modal .modal-actions { display:flex; gap:0.5rem; justify-content:center; margin-top:0.75rem; }

        .print-modal .btn { padding: 0.5rem 0.75rem; border-radius:6px; border:none; cursor:pointer; }

        .print-modal .btn-quote { background: var(--primary-color); color:#fff; }
        .print-modal .btn-invoice { background: #17a2b8; color:#fff; }
        .print-modal .btn-cancel { background: #6c757d; color:#fff; }

        /* Modal for editing order */
        .edit-modal {
            position: fixed;
            inset: 0;
            display: none;
            align-items: center;
            justify-content: center;
            background: rgba(0,0,0,0.5);
            z-index: 9999;
            overflow-y: auto;
            transition: opacity 0.2s ease;
            opacity: 0;
        }

        .edit-modal.active {
            display: flex;
            opacity: 1;
        }

        .edit-modal .modal-content {
            background: #fff;
            border-radius: 12px;
            width: 90%;
            max-width: 1000px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            margin: 2rem;
        }

        .edit-modal .modal-header {
            padding: 1.5rem;
            border-bottom: 2px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: var(--secondary-color);
            border-radius: 12px 12px 0 0;
        }

        .edit-modal .modal-header h2 {
            margin: 0;
            color: var(--text-dark);
            font-size: 1.5rem;
        }

        .edit-modal .modal-body {
            padding: 1.5rem;
        }

        .edit-modal .form-section {
            margin-bottom: 1.5rem;
        }

        .edit-modal .form-section h3 {
            margin: 0 0 1rem 0;
            color: var(--primary-color);
            font-size: 1.1rem;
            border-bottom: 2px solid var(--primary-color);
            padding-bottom: 0.5rem;
        }

        .edit-modal .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1rem;
        }

        .edit-modal .form-group {
            display: flex;
            flex-direction: column;
        }

        .edit-modal .form-group label {
            font-weight: 500;
            margin-bottom: 0.5rem;
            color: var(--text-dark);
            font-size: 0.9rem;
        }

        .edit-modal .form-group input,
        .edit-modal .form-group select,
        .edit-modal .form-group textarea {
            padding: 0.65rem;
            border: 1px solid var(--border-color);
            border-radius: 6px;
            font-size: 0.95rem;
        }

        .edit-modal .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }

        .edit-modal .products-section {
            margin-top: 1.5rem;
        }

        .edit-modal .products-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 1rem;
        }

        .edit-modal .products-table th {
            background: var(--secondary-color);
            padding: 0.75rem;
            text-align: left;
            font-weight: 600;
            border: 1px solid var(--border-color);
            font-size: 0.9rem;
        }

        .edit-modal .products-table td {
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            font-size: 0.9rem;
        }

        .edit-modal .products-table input,
        .edit-modal .products-table select {
            width: 100%;
            padding: 0.5rem;
            border: 1px solid var(--border-color);
            border-radius: 4px;
            font-size: 0.9rem;
        }

        .edit-modal .btn-add-product {
            background: #28a745;
            color: white;
            padding: 0.6rem 1.2rem;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95rem;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s ease;
        }

        .edit-modal .btn-add-product:hover {
            background: #218838;
        }

        .edit-modal .btn-remove-product {
            background: var(--danger-color);
            color: white;
            padding: 0.4rem 0.8rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.85rem;
        }

        .edit-modal .btn-remove-product:hover {
            background: #c82333;
        }

        .edit-modal .modal-footer {
            padding: 1.5rem;
            border-top: 2px solid var(--border-color);
            display: flex;
            justify-content: flex-end;
            gap: 1rem;
            background: var(--secondary-color);
            border-radius: 0 0 12px 12px;
        }

        .edit-modal .btn-save {
            background: var(--primary-color);
            color: white;
            padding: 0.65rem 1.5rem;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 500;
        }

        .edit-modal .btn-save:hover {
            background: #1a4a86;
        }

        .edit-modal .btn-close {
            background: #6c757d;
            color: white;
            padding: 0.65rem 1.5rem;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1rem;
        }

        .edit-modal .btn-close:hover {
            background: #5a6268;
        }

        .edit-modal .close-icon {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-muted);
            transition: color 0.2s ease;
        }

        .edit-modal .close-icon:hover {
            color: var(--danger-color);
        }

        .edit-modal .summary-box {
            background: var(--secondary-color);
            padding: 1rem;
            border-radius: 8px;
            margin-top: 1rem;
        }

        .edit-modal .summary-row {
            display: flex;
            justify-content: space-between;
            padding: 0.5rem 0;
            font-size: 0.95rem;
        }

        .edit-modal .summary-row.total {
            border-top: 2px solid var(--primary-color);
            margin-top: 0.5rem;
            padding-top: 0.75rem;
            font-weight: 600;
            font-size: 1.1rem;
            color: var(--primary-color);
        }

        .action-buttons {
            display: flex;
            gap: 0.5rem;
        }

        .btn-action {
            padding: 0.4rem 0.8rem;
            border: none;
            border-radius: 6px;
            font-size: 0.85rem;
            cursor: pointer;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            transition: all 0.2s ease;
        }

        .btn-view {
            background: var(--primary-color);
            color: white;
        }

        .btn-view:hover {
            background: #1a4a86;
        }

        .btn-delete {
            background: var(--danger-color);
            color: white;
        }

        .btn-delete:hover {
            background: #c82333;
        }

        .total-amount {
            font-weight: 600;
            color: var(--primary-color);
        }

        .empty-state {
            text-align: center;
            padding: 3rem;
            color: var(--text-muted);
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        /* Override for SweetAlert z-index */
        .swal2-container {
            z-index: 10000 !important;
        }

        @media (max-width: 768px) {
            .orders-table {
                overflow-x: auto;
            }

            table {
                min-width: 900px;
            }
        }
    </style>
</head>
<body>
    <div class="app-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="main-content">
            <div class="orders-container">
                <div class="page-header">
                    <h1><i class="fas fa-history"></i> ປະຫວັດ Orders ທັງໝົດ</h1>
                    <p>ລາຍການໃບສະເໜີລາຄາທັງໝົດໃນລະບົບ</p>
                </div>

                <div class="filters-bar">
                    <div class="filter-group">
                        <label><i class="fas fa-filter"></i> ສະຖານະໃບສະເໜີລາຄາ:</label>
                        <select id="statusFilter" onchange="filterOrders()">
                            <option value="all">ທັງໝົດ</option>
                            <option value="approved">ອະນຸມັດແລ້ວ</option>
                            <option value="pending">ລໍຖ້າອະນຸມັດ</option>
                            <option value="rejected">ປະຕິເສດ</option>
                            <option value="none">ຍັງບໍ່ມີສະຖານະ</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label><i class="fas fa-filter"></i> ສະຖານະໃບເກັບເງິນ:</label>
                        <select id="invoiceStatusFilter" onchange="filterOrders()">
                            <option value="all">ທັງໝົດ</option>
                            <option value="approved">ອະນຸມັດແລ້ວ</option>
                            <option value="pending">ລໍຖ້າອະນຸມັດ</option>
                            <option value="rejected">ປະຕິເສດ</option>
                            <option value="none">ຍັງບໍ່ມີສະຖານະ</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label><i class="fas fa-search"></i> ຄົ້ນຫາ:</label>
                        <input type="text" id="searchInput" placeholder="ຄົ້ນຫາ Order ID, ລູກຄ້າ..." onkeyup="filterOrders()">
                    </div>
                </div>

                <div class="orders-table">
                    <?php if ($orders_result && $orders_result->num_rows > 0): ?>
                        <table id="ordersTable">
                            <thead>
                                <tr>
                                    <th>Order ID</th>
                                    <th>ວັນທີ</th>
                                    <th>ຫົວຂໍ້</th>
                                    <th>ລູກຄ້າ</th>
                                    <th>ຍອດລວມ</th>
                                    <th>ສະຖານະໃບສະເໜີລາຄາ</th>
                                    <th>ສະຖານະໃບເກັບເງິນ</th>
                                    <th>ຜູ້ສ້າງ</th>
                                    <th>ຈັດການ</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($order = $orders_result->fetch_assoc()): ?>
                                    <tr data-status="<?php echo $order['approval_status'] ?: 'none'; ?>" data-invoice-status="<?php echo $order['invoice_approval_status'] ?: 'none'; ?>">
                                        <td><strong><?php echo htmlspecialchars($order['order_id']); ?></strong></td>
                                        <td><?php echo date('d/m/Y', strtotime($order['order_date'])); ?></td>
                                        <td><?php echo htmlspecialchars($order['section'] ?: '-'); ?></td>
                                        <td><?php echo htmlspecialchars($order['cus_name']); ?></td>
                                        <td class="total-amount">
                                            <?php echo htmlspecialchars($order['currencys_symbol']); ?> 
                                            <?php echo number_format($order['order_total'], 2); ?>
                                        </td>
                                        <td>
                                            <?php if ($order['approval_status'] === 'approved'): ?>
                                                <span class="status-badge approved">✓ ອະນຸມັດແລ້ວ</span>
                                            <?php elseif ($order['approval_status'] === 'pending'): ?>
                                                <span class="status-badge pending">⏳ ລໍຖ້າອະນຸມັດ</span>
                                            <?php elseif ($order['approval_status'] === 'rejected'): ?>
                                                <span class="status-badge rejected">✕ ປະຕິເສດ</span>
                                            <?php else: ?>
                                                <span class="status-badge none">- ຍັງບໍ່ມີສະຖານະ</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($order['invoice_approval_status'] === 'approved'): ?>
                                                <span class="status-badge approved">✓ ອະນຸມັດແລ້ວ</span>
                                            <?php elseif ($order['invoice_approval_status'] === 'pending'): ?>
                                                <span class="status-badge pending">⏳ ລໍຖ້າອະນຸມັດ</span>
                                            <?php elseif ($order['invoice_approval_status'] === 'rejected'): ?>
                                                <span class="status-badge rejected">✕ ປະຕິເສດ</span>
                                            <?php else: ?>
                                                <span class="status-badge none">- ຍັງບໍ່ມີສະຖານະ</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($order['user_name']); ?></td>
                                        <td>
                                            <div class="action-buttons">
                                                <button class="btn-action btn-view" onclick="showPrintOptions('<?php echo htmlspecialchars($order['order_id'], ENT_QUOTES); ?>')">
                                                    <i class="fas fa-eye"></i> ເບິ່ງ
                                                </button>
                                                <button class="btn-action" style="background: #ffc107; color: #000;" onclick="editOrder('<?php echo htmlspecialchars($order['order_id'], ENT_QUOTES); ?>')">
                                                    <i class="fas fa-edit"></i> ແກ້ໄຂ
                                                </button>
                                                <button class="btn-action btn-delete" onclick="deleteOrder('<?php echo $order['order_id']; ?>')">
                                                    <i class="fas fa-trash"></i> ລຶບ
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>    
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-inbox"></i>
                            <p>ຍັງບໍ່ມີ Orders ໃນລະບົບ</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="js/sidebar.js"></script>
    
    <script>
        // Print options modal logic
        function showPrintOptions(orderId) {
            const modal = document.getElementById('printOptionsModal');
            modal.dataset.orderId = orderId;
            modal.classList.add('active');
            modal.setAttribute('aria-hidden', 'false');
        }

        function closePrintOptions() {
            const modal = document.getElementById('printOptionsModal');
            modal.classList.remove('active');
            modal.setAttribute('aria-hidden', 'true');
            delete modal.dataset.orderId;
        }

        function openPrint(type) {
            const modal = document.getElementById('printOptionsModal');
            const orderId = modal.dataset.orderId;
            if (!orderId) return closePrintOptions();

            let url = '';
            if (type === 'quotation') {
                url = `print_quotation.php?order_id=${encodeURIComponent(orderId)}`;
            } else if (type === 'invoice') {
                url = `print_invoice.php?order_id=${encodeURIComponent(orderId)}`;
            }

            if (url) {
                window.open(url, '_blank');
            }
            closePrintOptions();
        }

        function filterOrders() {
            const statusFilter = document.getElementById('statusFilter').value;
            const invoiceStatusFilter = document.getElementById('invoiceStatusFilter').value;
            const searchInput = document.getElementById('searchInput').value.toLowerCase();
            const table = document.getElementById('ordersTable');
            const rows = table.getElementsByTagName('tr');

            for (let i = 1; i < rows.length; i++) {
                const row = rows[i];
                const status = row.getAttribute('data-status');
                const invoiceStatus = row.getAttribute('data-invoice-status');
                const text = row.textContent.toLowerCase();

                let showStatus = statusFilter === 'all' || status === statusFilter;
                let showInvoiceStatus = invoiceStatusFilter === 'all' || invoiceStatus === invoiceStatusFilter;
                let showSearch = searchInput === '' || text.includes(searchInput);

                if (showStatus && showInvoiceStatus && showSearch) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            }
        }

        function deleteOrder(orderId) {
            Swal.fire({
                title: 'ຢືນຢັນການລຶບ',
                text: `ທ່ານຕ້ອງການລຶບ Order ${orderId} ແທ້ບໍ?`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc3545',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'ລຶບ',
                cancelButtonText: 'ຍົກເລີກ'
            }).then((result) => {
                if (result.isConfirmed) {
                    // TODO: Implement delete functionality
                    Swal.fire(
                        'ລຶບສຳເລັດ!',
                        'Order ຖືກລຶບອອກຈາກລະບົບແລ້ວ',
                        'success'
                    );
                }
            });
        }

        // Edit Order Modal Functions
        let productsData = [];
        let customersData = [];
        let currenciesData = [];
        let ratesData = [];
        let unitsData = [];
        let typesData = [];
        let editProductRowCounter = 0;

        async function loadDropdownData() {
            try {
                // Load products
                const productsRes = await fetch('api/get_products.php');
                if (productsRes.ok) {
                    const productsText = await productsRes.text();
                    try {
                        const parsedData = JSON.parse(productsText);
                        // API returns {success: true, products: [...], types: [...]}
                        // We need to extract the 'products' array
                        if (parsedData.success && Array.isArray(parsedData.products)) {
                            productsData = parsedData.products;
                        } else if (Array.isArray(parsedData)) {
                            productsData = parsedData;
                        } else {
                            productsData = [];
                        }
                    } catch (e) {
                        productsData = [];
                    }
                } else {
                    productsData = [];
                }

                // Load customers
                const customersRes = await fetch('get_customers.php');
                if (customersRes.ok) {
                    customersData = await customersRes.json();
                    if (!Array.isArray(customersData)) customersData = [];
                } else {
                    customersData = [];
                }

                // Load currencies
                const currenciesRes = await fetch('get_all_currencies.php');
                if (currenciesRes.ok) {
                    currenciesData = await currenciesRes.json();
                    if (!Array.isArray(currenciesData)) currenciesData = [];
                } else {
                    currenciesData = [];
                }

                // Load rates
                const ratesRes = await fetch('get_all_rates.php');
                if (ratesRes.ok) {
                    ratesData = await ratesRes.json();
                    if (!Array.isArray(ratesData)) ratesData = [];
                } else {
                    ratesData = [];
                }

                // Load units
                const unitsRes = await fetch('get_all_units.php');
                if (unitsRes.ok) {
                    unitsData = await unitsRes.json();
                    if (!Array.isArray(unitsData)) unitsData = [];
                } else {
                    unitsData = [];
                }

                // Load types
                const typesRes = await fetch('get_all_types.php');
                if (typesRes.ok) {
                    typesData = await typesRes.json();
                    if (!Array.isArray(typesData)) typesData = [];
                } else {
                    typesData = [];
                }
            } catch (error) {
                // Initialize as empty arrays on error
                if (!Array.isArray(productsData)) productsData = [];
                if (!Array.isArray(customersData)) customersData = [];
                if (!Array.isArray(currenciesData)) currenciesData = [];
                if (!Array.isArray(ratesData)) ratesData = [];
                if (!Array.isArray(unitsData)) unitsData = [];
                if (!Array.isArray(typesData)) typesData = [];
            }
        }

        async function editOrder(orderId) {
            // Load dropdown data if not loaded
            if (productsData.length === 0) {
                await loadDropdownData();
            }

            // Fetch order details
            try {
                const response = await fetch(`get_order_details.php?order_id=${encodeURIComponent(orderId)}`);
                const orderData = await response.json();

                if (orderData.success) {
                    populateEditForm(orderData.order, orderData.items);
                    const modal = document.getElementById('editOrderModal');
                    modal.classList.add('active');
                    modal.setAttribute('aria-hidden', 'false');
                } else {
                    Swal.fire('ຜິດພາດ!', orderData.message || 'ບໍ່ສາມາດໂຫລດຂໍ້ມູນໄດ້', 'error');
                }
            } catch (error) {
                console.error('Error fetching order details:', error);
                Swal.fire('ຜິດພາດ!', 'ເກີດຂໍ້ຜິດພາດໃນການໂຫລດຂໍ້ມູນ', 'error');
            }
        }

        function populateEditForm(order, items) {
            
            // Populate basic info
            document.getElementById('edit_order_id').value = order.order_id;
            document.getElementById('edit_display_order_id').value = order.order_id;
            document.getElementById('edit_order_date').value = order.order_date;
            document.getElementById('edit_section').value = order.section || '';
            
            // ตั้งค่าส่วนลด (จำนวนเงิน)
            const discountValue = parseFloat(order.order_discount) || 0;
            document.getElementById('edit_order_discount').value = discountValue;
            
            // ตั้งค่า VAT
            const taxPercent = parseFloat(order.order_tax_percent) || 0;
            document.getElementById('edit_order_tax_percent').value = taxPercent;
            document.getElementById('edit_vat_toggle').checked = taxPercent > 0;

            // Populate customers dropdown
            const cusSelect = document.getElementById('edit_cus_id');
            cusSelect.innerHTML = '<option value="ເລືອກລູກຄ້າ</option>';
            if (Array.isArray(customersData)) {
                customersData.forEach(cus => {
                    const option = document.createElement('option');
                    option.value = cus.cus_id;
                    option.textContent = cus.cus_name;
                    if (cus.cus_id == order.cus_id) {
                        option.selected = true;
                    }
                    cusSelect.appendChild(option);
                });
            }

            // Populate currencies dropdown
            const currSelect = document.getElementById('edit_currencys_id');
            currSelect.innerHTML = '<option value="ເລືອກສະກຸນເງິນ</option>';
            if (Array.isArray(currenciesData)) {
                currenciesData.forEach(curr => {
                    const option = document.createElement('option');
                    option.value = curr.currencys_id;
                    option.textContent = `${curr.currencys_name} (${curr.currencys_symbol})`;
                    if (curr.currencys_id == order.currencys_id) {
                        option.selected = true;
                    }
                    currSelect.appendChild(option);
                });
                // เพิ่ม event ให้เปลี่ยน rate ตามสกุลเงิน
                currSelect.onchange = function() {
                    setRateDisplayByCurrency(this.value, null);
                    updateProductPricesByCurrency(this.value);
                };
            }
            // แสดงค่า rate ทันทีเมื่อเปิด modal
            setRateDisplayByCurrency(order.currencys_id, order.rate_id);
            // อัปเดตราคาสินค้าตามสกุลเงินเมื่อเปิด modal
            updateProductPricesByCurrency(order.currencys_id, true);
            // ฟังก์ชันแปลงราคาสินค้าทั้งหมดในตารางตามอัตราแลกเปลี่ยนของสกุลเงินที่เลือก
            function updateProductPricesByCurrency(currencys_id, isInit = false) {
                // หาอัตราแลกเปลี่ยนของสกุลเงินที่เลือก
                let foundRate = null;
                if (Array.isArray(ratesData)) {
                    const filtered = ratesData.filter(r => String(r.currencys_id) === String(currencys_id));
                    if (filtered.length > 0) {
                        foundRate = filtered[0];
                    }
                }
                if (!foundRate) return;

                const rateValue = parseFloat(foundRate.rate_amount);
                if (!rateValue || isNaN(rateValue)) return;

                // สมมติราคาสินค้าใน productsData เป็นราคาสกุลเงินหลัก (เช่น LAK)
                // ถ้าเปลี่ยนสกุลเงิน ให้แปลงราคาสินค้าในแต่ละแถว
                const rows = document.querySelectorAll('#editProductsTableBody tr');
                rows.forEach(row => {
                    const productSelect = row.querySelector('.product-select');
                    const priceInput = row.querySelector('.price-input');
                    if (!productSelect || !priceInput) return;
                    const proId = productSelect.value;
                    // หา product object
                    const product = productsData.find(p => p.pro_id == proId);
                    if (!product) return;

                    // ราคาต้นฉบับ (ราคาสกุลเงินหลัก)
                    let basePrice = parseFloat(product.pro_price) || 0;

                    // ถ้าเป็นการเปิด modal ครั้งแรก (isInit) และสกุลเงินตรงกับสกุลเงินหลัก ไม่ต้องแปลง
                    if (isInit && String(currencys_id) === String(product.currencys_id || '1')) {
                        priceInput.value = basePrice;
                    } else {
                        // แปลงราคาตามอัตราแลกเปลี่ยน
                        priceInput.value = (basePrice / rateValue).toFixed(2);
                    }
                    // อัปเดตยอดรวมแถว
                    calculateRowTotal(priceInput);
                });
                // อัปเดตยอดรวมทั้งหมด
                calculateEditTotal();
            }
            // Populate rates dropdown
            const rateSelect = document.getElementById('edit_rate_id');
            rateSelect.innerHTML = '<option value="ເລືອກອັດຕາແລກປ່ຽນ</option>';
            if (Array.isArray(ratesData)) {
                let rateFound = false;
                ratesData.forEach(rate => {
                    const option = document.createElement('option');
                    option.value = rate.rate_id;
                    // แสดง ชื่อสกุลเงิน และอัตรา
                    const displayText = rate.currencys_name 
                        ? `${rate.currencys_name} - ${parseFloat(rate.rate_amount).toLocaleString()} ${rate.currencys_symbol || ''}`
                        : `Rate ${rate.rate_id} - ${parseFloat(rate.rate_amount).toLocaleString()}`;
                    option.textContent = displayText;
                    // เปรียบเทียบแบบ string และ trim ค่าว่าง (ไม่เลือกถ้า rate_id เป็น 0 หรือว่าง)
                    const orderRateId = String(order.rate_id || '').trim();
                    const currentRateId = String(rate.rate_id || '').trim();
                    if (orderRateId && orderRateId !== '0' && currentRateId && orderRateId === currentRateId) {
                        option.selected = true;
                        rateFound = true;
                    }
                    rateSelect.appendChild(option);
                });
            }

            // Populate products
            const tbody = document.getElementById('editProductsTableBody');
            tbody.innerHTML = '';
            editProductRowCounter = 0;
            if (Array.isArray(items)) {
                items.forEach((item) => {
                    addProductRow(item);
                });
            }

            calculateEditTotal();
        }

        function addProductRow(itemData = null) {
            const tbody = document.getElementById('editProductsTableBody');
            const rowId = `product_row_${editProductRowCounter++}`;
            
            // ดึงข้อมูลสินค้าจาก productsData ถ้ามี itemData
            let selectedProduct = null;
            if (itemData && itemData.pro_id) {
                selectedProduct = productsData.find(p => p.pro_id == itemData.pro_id);
            }
            const row = document.createElement('tr');
            row.id = rowId;
            // ใช้ types_id และ unit_id จาก itemData ถ้ามี (จาก orders_detail) ถ้าไม่มีใช้จาก selectedProduct หรือ product_types_id/product_unit_id
            const selectedTypesId = itemData?.types_id || itemData?.product_types_id || selectedProduct?.types_id;
            const selectedUnitId = itemData?.unit_id || itemData?.product_unit_id || selectedProduct?.unit_id;
            // สร้าง datalist สำหรับ products
            let productDatalistId = 'product_datalist';
            if (!document.getElementById(productDatalistId)) {
                const datalist = document.createElement('datalist');
                datalist.id = productDatalistId;
                if (Array.isArray(productsData)) {
                    productsData.forEach(p => {
                        const option = document.createElement('option');
                        option.value = p.pro_name;
                        option.setAttribute('data-pro-id', p.pro_id);
                        datalist.appendChild(option);
                    });
                }
                document.body.appendChild(datalist);
            }

            // หาค่าเริ่มต้นของสินค้า
            let selectedProductName = '';
            let selectedProductId = '';
            if (itemData && itemData.pro_id && Array.isArray(productsData)) {
                const found = productsData.find(p => p.pro_id == itemData.pro_id);
                if (found) {
                    selectedProductName = found.pro_name;
                    selectedProductId = found.pro_id;
                }
            }
            
            // หาชื่อประเภทและหน่วยจาก id
            let typeName = '';
            if (selectedTypesId && Array.isArray(typesData)) {
                const t = typesData.find(tt => tt.types_id == selectedTypesId);
                if (t) typeName = t.types_name;
            }
            let unitName = '';
            if (selectedUnitId && Array.isArray(unitsData)) {
                const u = unitsData.find(uu => uu.unit_id == selectedUnitId);
                if (u) unitName = u.unit_name;
            }

            row.innerHTML = `
                <td style="position:relative;">
                    <input type="text" class="product-input" autocomplete="off" value="${selectedProductName}" placeholder="ค้นหาสินค้า..." required oninput="onProductInput(this)" onfocus="showProductDropdown(this)" onblur="hideProductDropdown(this)">
                    <input type="hidden" class="product-select" value="${selectedProductId}">
                    <div class="product-dropdown" style="display:none; position:absolute; left:0; right:0; top:100%; z-index:11000; background:#fff; border:1px solid #ccc; border-radius:0 0 8px 8px; max-height:200px; overflow-y:auto; box-shadow:0 4px 16px rgba(0,0,0,0.12);"></div>
                </td>
                <td>
                    <input type="text" class="type-select" value="${typeName}" readonly style="background:#f5f5f5;">
                </td>
                <td>
                    <input type="number" class="qty-input" min="1" step="1" value="${itemData ? itemData.order_qty : 1}" onchange="calculateRowTotal(this)" required>
                </td>
                <td>
                    <input type="text" class="unit-select" value="${unitName}" readonly style="background:#f5f5f5;">
                </td>
                <td>
                    <input type="number" class="price-input" min="0" step="0.01" value="${itemData ? itemData.pro_price : 0}" onchange="calculateRowTotal(this)" required>
                </td>
                <td>
                    <input type="number" class="total-input" value="${itemData ? itemData.total : 0}" disabled>
                </td>
                <td>
                    <button type="button" class="btn-remove-product" onclick="removeProductRow('${rowId}')">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;
            
            tbody.appendChild(row);
            if (itemData) {
                calculateRowTotal(row.querySelector('.qty-input'));
            }
        }

        function onProductChange(select) {
            // ไม่ใช้แล้ว (onProductChange) ให้ใช้ onProductInput แทน
        }

        // ฟังก์ชันดึงอัตราแลกเปลี่ยนปัจจุบัน
        function getCurrentRate() {
            const currencys_id = document.getElementById('edit_currencys_id')?.value;
            if (!currencys_id || !Array.isArray(ratesData)) return 1;
            const filtered = ratesData.filter(r => String(r.currencys_id) === String(currencys_id));
            if (filtered.length > 0) {
                return parseFloat(filtered[0].rate_amount) || 1;
            }
            return 1;
        }

        // ฟังก์ชันแปลงราคาตามอัตราแลกเปลี่ยน
        function convertPrice(basePrice) {
            const rate = getCurrentRate();
            if (rate === 1) return basePrice;
            return basePrice / rate;
        }

        // ฟังก์ชันใหม่: เมื่อเลือกหรือค้นหาสินค้า
        function onProductInput(input) {
            const row = input.closest('tr');
            const hiddenInput = row.querySelector('.product-select');
            const dropdown = row.querySelector('.product-dropdown');
            let val = input.value.trim().toLowerCase();
            let filtered = [];
            if (Array.isArray(productsData)) {
                if (val === '') {
                    filtered = productsData;
                } else {
                    filtered = productsData.filter(p => p.pro_name && p.pro_name.toLowerCase().includes(val));
                }
            }
            // Render dropdown
            if (filtered.length > 0) {
                dropdown.innerHTML = filtered.map((p, idx) => {
                    // หาชื่อประเภทและหน่วย
                    let typeName = 'ບໍ່ມີ';
                    if (p.types_id && Array.isArray(typesData)) {
                        const t = typesData.find(tt => tt.types_id == p.types_id);
                        if (t) typeName = t.types_name;
                    }
                    let unitName = 'ບໍ່ມີ';
                    if (p.unit_id && Array.isArray(unitsData)) {
                        const u = unitsData.find(uu => uu.unit_id == p.unit_id);
                        if (u) unitName = u.unit_name;
                    }
                    // แปลงราคาตามอัตราแลกเปลี่ยน
                    const convertedPrice = convertPrice(parseFloat(p.pro_price || 0));
                    const price = convertedPrice.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                    
                    return `
                    <div class="product-option" tabindex="0" style="display:flex;align-items:center;padding:10px 12px;cursor:pointer;${idx===0?'background:#e8f0fe;':''}" onclick="selectProductOption(this, '${p.pro_id}')">
                        <span style="font-size:1rem;">${p.pro_name}</span>
                        <div class="product-tooltip">
                            <div class="tooltip-title">${p.pro_name}</div>
                            <div class="tooltip-row"><span class="tooltip-label">ປະເພດ:</span><span class="tooltip-value">${typeName}</span></div>
                            <div class="tooltip-row"><span class="tooltip-label">ລາຍລະອຽດ:</span><span class="tooltip-value">${p.pro_detail || '-'}</span></div>
                            <div class="tooltip-row"><span class="tooltip-label">ລາຄາ:</span><span class="tooltip-value">${price}</span></div>
                        </div>
                    </div>
                `}).join('');
                dropdown.style.display = 'block';
            } else {
                dropdown.innerHTML = '<div style="padding:12px;color:#888;">ไม่พบสินค้า</div>';
                dropdown.style.display = 'block';
            }
            // Clear selection if not exact match
            let selectedProduct = null;
            if (Array.isArray(productsData)) {
                selectedProduct = productsData.find(p => p.pro_name === input.value);
            }
            if (selectedProduct) {
                hiddenInput.value = selectedProduct.pro_id;
                // แปลงราคาตามอัตราแลกเปลี่ยน
                const convertedPrice = convertPrice(parseFloat(selectedProduct.pro_price || 0));
                row.querySelector('.price-input').value = convertedPrice.toFixed(2);
                const typeSelect = row.querySelector('.type-select');
                if (selectedProduct.types_id && Array.isArray(typesData)) {
                    const t = typesData.find(tt => tt.types_id == selectedProduct.types_id);
                    typeSelect.value = t ? t.types_name : '';
                }
                const unitSelect = row.querySelector('.unit-select');
                if (selectedProduct.unit_id && Array.isArray(unitsData)) {
                    const u = unitsData.find(uu => uu.unit_id == selectedProduct.unit_id);
                    unitSelect.value = u ? u.unit_name : '';
                }
            } else {
                hiddenInput.value = '';
                row.querySelector('.price-input').value = '';
                row.querySelector('.type-select').value = '';
                row.querySelector('.unit-select').value = '';
            }
            calculateRowTotal(row.querySelector('.qty-input'));
        }

        // แสดง dropdown เมื่อ focus
        function showProductDropdown(input) {
            setTimeout(() => {
                const row = input.closest('tr');
                const dropdown = row.querySelector('.product-dropdown');
                if (!dropdown) return;
                let val = input.value.trim().toLowerCase();
                let filtered = [];
                if (Array.isArray(productsData)) {
                    if (val === '') {
                        filtered = productsData;
                    } else {
                        filtered = productsData.filter(p => p.pro_name && p.pro_name.toLowerCase().includes(val));
                    }
                }
                if (filtered.length > 0) {
                    dropdown.innerHTML = filtered.map((p, idx) => {
                        // หาชื่อประเภทและหน่วย
                        let typeName = 'ບໍ່ມີ';
                        if (p.types_id && Array.isArray(typesData)) {
                            const t = typesData.find(tt => tt.types_id == p.types_id);
                            if (t) typeName = t.types_name;
                        }
                        let unitName = 'ບໍ່ມີ';
                        if (p.unit_id && Array.isArray(unitsData)) {
                            const u = unitsData.find(uu => uu.unit_id == p.unit_id);
                            if (u) unitName = u.unit_name;
                        }
                        // แปลงราคาตามอัตราแลกเปลี่ยน
                        const convertedPrice = convertPrice(parseFloat(p.pro_price || 0));
                        const price = convertedPrice.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                        
                        return `
                        <div class="product-option" tabindex="0" style="display:flex;align-items:center;padding:10px 12px;cursor:pointer;${idx===0?'background:#e8f0fe;':''}" onclick="selectProductOption(this, '${p.pro_id}')">
                            <span style="font-size:1rem;">${p.pro_name}</span>
                            <div class="product-tooltip">
                                <div class="tooltip-title">${p.pro_name}</div>
                                <div class="tooltip-row"><span class="tooltip-label">ປະເພດ:</span><span class="tooltip-value">${typeName}</span></div>
                                <div class="tooltip-row"><span class="tooltip-label">ລາຍລະອຽດ:</span><span class="tooltip-value">${p.pro_detail || '-'}</span></div>
                                <div class="tooltip-row"><span class="tooltip-label">ລາຄາ:</span><span class="tooltip-value">${price}</span></div>
                            </div>
                        </div>
                    `}).join('');
                    dropdown.style.display = 'block';
                } else {
                    dropdown.innerHTML = '<div style="padding:12px;color:#888;">ไม่พบสินค้า</div>';
                    dropdown.style.display = 'block';
                }
            }, 80);
        }
        // ซ่อน dropdown เมื่อ blur
        function hideProductDropdown(input) {
            setTimeout(() => {
                const row = input.closest('tr');
                const dropdown = row.querySelector('.product-dropdown');
                if (dropdown) dropdown.style.display = 'none';
            }, 180);
        }
        // เลือกสินค้าใน dropdown
        function selectProductOption(div, proId) {
            const row = div.closest('tr');
            const input = row.querySelector('.product-input');
            const hiddenInput = row.querySelector('.product-select');
            const dropdown = row.querySelector('.product-dropdown');
            const product = productsData.find(p => p.pro_id == proId);
            if (product) {
                input.value = product.pro_name;
                hiddenInput.value = product.pro_id;
                // แปลงราคาตามอัตราแลกเปลี่ยน
                const convertedPrice = convertPrice(parseFloat(product.pro_price || 0));
                row.querySelector('.price-input').value = convertedPrice.toFixed(2);
                const typeSelect = row.querySelector('.type-select');
                if (product.types_id && Array.isArray(typesData)) {
                    const t = typesData.find(tt => tt.types_id == product.types_id);
                    typeSelect.value = t ? t.types_name : '';
                }
                const unitSelect = row.querySelector('.unit-select');
                if (product.unit_id && Array.isArray(unitsData)) {
                    const u = unitsData.find(uu => uu.unit_id == product.unit_id);
                    unitSelect.value = u ? u.unit_name : '';
                }
                dropdown.style.display = 'none';
                input.blur();
            }
            calculateRowTotal(row.querySelector('.qty-input'));
        }

        // Position tooltip on mouseover
        document.addEventListener('mouseover', function(e) {
            const option = e.target.closest('.product-option');
            if (option) {
                const tooltip = option.querySelector('.product-tooltip');
                if (tooltip) {
                    const rect = option.getBoundingClientRect();
                    const tooltipWidth = 320;
                    const tooltipHeight = 180;
                    
                    // ตรวจสอบว่าควรแสดงทางซ้ายหรือขวา
                    let left = rect.right + 10;
                    if (left + tooltipWidth > window.innerWidth) {
                        left = rect.left - tooltipWidth - 10;
                    }
                    
                    // ตรวจสอบว่าควรแสดงบนหรือล่าง
                    let top = rect.top;
                    if (top + tooltipHeight > window.innerHeight) {
                        top = window.innerHeight - tooltipHeight - 20;
                    }
                    if (top < 10) top = 10;
                    
                    tooltip.style.left = left + 'px';
                    tooltip.style.top = top + 'px';
                }
            }
        });
    // CSS for product dropdown
    const style = document.createElement('style');
    style.innerHTML = `
    .product-dropdown { z-index: 11000 !important; max-height: 200px !important; overflow-y: auto !important; }
    .product-dropdown::-webkit-scrollbar { width: 8px; background: #f5f5f5; }
    .product-dropdown::-webkit-scrollbar-thumb { background: #e0e0e0; border-radius: 4px; }
    .product-option { position: relative; }
    .product-option:hover, .product-option:focus { background: #e8f0fe !important; }
    .product-tooltip {
        display: none;
        position: fixed;
        background: #1a1a2e;
        color: #fff;
        padding: 14px 18px;
        border-radius: 10px;
        font-size: 0.9rem;
        min-width: 280px;
        max-width: 340px;
        box-shadow: 0 8px 32px rgba(0,0,0,0.35);
        z-index: 99999;
        white-space: normal;
        line-height: 1.6;
        border: 1px solid rgba(255,255,255,0.1);
        pointer-events: none;
    }
    .product-option:hover .product-tooltip { display: block; }
    .product-tooltip .tooltip-title { font-weight: 700; font-size: 1.05rem; margin-bottom: 10px; color: #4fc3f7; border-bottom: 2px solid #4fc3f7; padding-bottom: 8px; }
    .product-tooltip .tooltip-row { display: flex; justify-content: space-between; padding: 5px 0; border-bottom: 1px solid rgba(255,255,255,0.08); gap: 15px; }
    .product-tooltip .tooltip-row:last-child { border-bottom: none; }
    .product-tooltip .tooltip-label { color: #aaa; font-size: 0.85rem; }
    .product-tooltip .tooltip-value { color: #fff; font-weight: 600; text-align: right; }
    `;
    document.head.appendChild(style);

        function calculateRowTotal(input) {
            const row = input.closest('tr');
            const qty = parseFloat(row.querySelector('.qty-input').value) || 0;
            const price = parseFloat(row.querySelector('.price-input').value) || 0;
            const total = qty * price;
            
            row.querySelector('.total-input').value = total.toFixed(2);
            calculateEditTotal();
        }

        function removeProductRow(rowId) {
            document.getElementById(rowId).remove();
            calculateEditTotal();
        }

        function calculateEditTotal() {
            const rows = document.querySelectorAll('#editProductsTableBody tr');
            let subtotal = 0;
            
            rows.forEach(row => {
                const total = parseFloat(row.querySelector('.total-input').value) || 0;
                subtotal += total;
            });
            
            // ส่วนลดต้องเป็นจำนวนเงิน ไม่ใช่เปอร์เซ็นต์
            const discountAmount = parseFloat(document.getElementById('edit_order_discount').value) || 0;
            const taxPercent = parseFloat(document.getElementById('edit_order_tax_percent').value) || 0;
            
            const afterDiscount = subtotal - discountAmount;
            const taxAmount = afterDiscount * (taxPercent / 100);
            const total = afterDiscount + taxAmount;
            
            document.getElementById('edit_subtotal_display').textContent = subtotal.toFixed(2);
            document.getElementById('edit_discount_display').textContent = discountAmount.toFixed(2);
            document.getElementById('edit_tax_display').textContent = taxAmount.toFixed(2);
            document.getElementById('edit_total_display').textContent = total.toFixed(2);
        }

        function closeEditModal() {
            const modal = document.getElementById('editOrderModal');
            modal.classList.remove('active');
            modal.setAttribute('aria-hidden', 'true');
        }

        function toggleVAT() {
            const vatToggle = document.getElementById('edit_vat_toggle');
            const taxInput = document.getElementById('edit_order_tax_percent');
            
            if (vatToggle.checked) {
                taxInput.value = 10;
            } else {
                taxInput.value = 0;
            }
            
            calculateEditTotal();
        }

        async function saveEditOrder() {
            const form = document.getElementById('editOrderForm');
            if (!form.checkValidity()) {
                Swal.fire('ແຈ້ງເຕືອນ!', 'ກະລຸນາປ້ອນຂໍ້ມູນໃຫ້ຄົບຖ້ວນ', 'warning');
                return;
            }


            // แสดง dialog ให้เลือกประเภทการแก้ไข (ใหม่)
            const result = await Swal.fire({
                title: 'ຢືນຢັນການບັນທຶກ',
                text: 'ທ່ານຈະບັນທຶກການແກ້ໄຂຂອງ \nໃບສະເໜີລາຄາ ຫຼື ໃບເກັບເງິນ ?',
                icon: 'question',
                showCancelButton: true,
                showDenyButton: true,
                confirmButtonText: 'ແກ້ໄຂໃບສະເໜີລາຄາ',
                denyButtonText: 'ແກ້ໄຂໃບເກັບເງິນ',
                cancelButtonText: 'ຍົກເລີກ',
                allowOutsideClick: false,
                allowEscapeKey: false
            });
            if (result.isConfirmed) {
                // แก้ไขใบเสนอราคา
            } else if (result.isDenied) {
                // แก้ไขใบเก็บเงิน
            } else {
                // cancel
                return;
            }


            // Collect form data
            const formData = new FormData();
            formData.append('order_id', document.getElementById('edit_order_id').value);
            formData.append('order_date', document.getElementById('edit_order_date').value);
            formData.append('section', document.getElementById('edit_section').value);
            formData.append('cus_id', document.getElementById('edit_cus_id').value);
            formData.append('currencys_id', document.getElementById('edit_currencys_id').value);
            formData.append('rate_id', document.getElementById('edit_rate_id').value);
            formData.append('order_discount', document.getElementById('edit_order_discount').value);
            formData.append('order_tax_percent', document.getElementById('edit_order_tax_percent').value);

            // ส่งประเภทการแก้ไขไป backend
            if (result.isConfirmed) {
                // confirmButton: ແກ້ໄຂໃບສະເໜີລາຄາ
                formData.append('edit_type', 'quotation');
            } else if (result.isDenied) {
                // denyButton: ແກ້ໄຂໃບເກັບເງິນ
                formData.append('edit_type', 'invoice');
            }

            // Collect products
            const products = [];
            const rows = document.querySelectorAll('#editProductsTableBody tr');
            rows.forEach(row => {
                products.push({
                    pro_id: row.querySelector('.product-select').value,
                    types_id: row.querySelector('.type-select').value,
                    quantity: row.querySelector('.qty-input').value,
                    unit_id: row.querySelector('.unit-select').value,
                    price: row.querySelector('.price-input').value,
                    total: row.querySelector('.total-input').value
                });
            });
            formData.append('products', JSON.stringify(products));

            // Calculate totals
            const subtotal = parseFloat(document.getElementById('edit_subtotal_display').textContent);
            const discountAmount = parseFloat(document.getElementById('edit_discount_display').textContent);
            const taxAmount = parseFloat(document.getElementById('edit_tax_display').textContent);
            const total = parseFloat(document.getElementById('edit_total_display').textContent);
            
            formData.append('order_subtotal', subtotal);
            formData.append('order_tax_amount', taxAmount);
            formData.append('order_total', total);

            try {
                const response = await fetch('update_order.php', {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();

                if (result.success) {
                    // ปิด modal ก่อน
                    closeEditModal();
                    // รอให้ modal ปิดสนิท แล้วค่อยแสดง alert
                    setTimeout(() => {
                        Swal.fire({
                            icon: 'success',
                            title: 'ສຳເລັດ!',
                            text: 'ແກ້ໄຂ Order ສຳເລັດແລ້ວ',
                            confirmButtonText: 'ຕົກລົງ'
                        }).then(() => {
                            location.reload();
                        });
                    }, 150);  // 150ms พอให้ browser render display: none
                } else {
                    // ปิด modal ก่อน
                    closeEditModal();
                    // รอให้ modal ปิดสนิท แล้วค่อยแสดง alert
                    setTimeout(() => {
                        Swal.fire('ຜິດພາດ!', result.message || 'ບໍ່ສາມາດແກ້ໄຂໄດ້', 'error');
                    }, 150);
                }
            } catch (error) {
                console.error('Error saving order:', error);
                // ปิด modal ก่อน
                closeEditModal();
                // รอให้ modal ปิดสนิท แล้วค่อยแสดง alert
                setTimeout(() => {
                    Swal.fire('ຜິດພາດ!', 'ເກີດຂໍ້ຜິດພາດໃນການບັນທຶກຂໍ້ມູນ', 'error');
                }, 150);
            }
        }
        function setRateDisplayByCurrency(currencys_id, order_rate_id) {
    let foundRate = null;
    if (Array.isArray(ratesData)) {
        const filtered = ratesData.filter(r => String(r.currencys_id) === String(currencys_id));
        if (filtered.length > 0) {
            foundRate = filtered[0];
            if (order_rate_id) {
                const match = filtered.find(r => String(r.rate_id) === String(order_rate_id));
                if (match) foundRate = match;
            }
        }
    }
    const rateDisplay = document.getElementById('edit_rate_display');
    const rateIdInput = document.getElementById('edit_rate_id');
    if (foundRate) {
        rateDisplay.value = `${foundRate.rate_amount} ${foundRate.currencys_symbol || ''}`;
        rateIdInput.value = foundRate.rate_id;
    } else {
        rateDisplay.value = '';
        rateIdInput.value = '';
    }
}
// (ย้ายไปไว้หลัง populate currencies dropdown แล้ว)
    </script>
                    
                    <!-- Print Options Modal -->
                    <div id="printOptionsModal" class="print-modal" role="dialog" aria-modal="true" aria-hidden="true">
                        <div class="modal-card" role="document">
                            <h3>ເລືອກໄຟລ໌ທີ່ຕ້ອງການ</h3>
                            <p>ເລືອກໃບສະເໜີລາຄາ ຫຼື ໃບເກັບເງິນ</p>
                            <div class="modal-actions">
                                <button class="btn btn-quote" onclick="openPrint('quotation')">ໃບສະເໜີລາຄາ</button>
                                <button class="btn btn-invoice" onclick="openPrint('invoice')">ໃບເກັບເງິນ</button>
                            </div>
                            <div style="margin-top:0.75rem;">
                                <button class="btn btn-cancel" onclick="closePrintOptions()">ຍົກເລີກ</button>
                            </div>
                        </div>
                    </div>

                    <!-- Edit Order Modal -->
                    <div id="editOrderModal" class="edit-modal" role="dialog" aria-modal="true" aria-hidden="true">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h2><i class="fas fa-edit"></i> ແກ້ໄຂ Order</h2>
                                <button class="close-icon" onclick="closeEditModal()">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form id="editOrderForm">
                                    <input type="hidden" id="edit_order_id" name="order_id">
                                    
                                    <!-- ข้อมูลพื้นฐาน -->
                                    <div class="form-section">
                                        <h3><i class="fas fa-info-circle"></i> ຂໍ້ມູນພື້ນຖານ</h3>
                                        <div class="form-grid">
                                            <div class="form-group">
                                                <label>Order ID</label>
                                                <input type="text" id="edit_display_order_id" disabled>
                                            </div>
                                            <div class="form-group">
                                                <label>ວັນທີ</label>
                                                <input type="date" id="edit_order_date" name="order_date" required>
                                            </div>
                                            <div class="form-group">
                                                <label>ລູກຄ້າ</label>
                                                <select id="edit_cus_id" name="cus_id" required>
                                                    <option value="">ເລືອກລູກຄ້າ</option>
                                                </select>
                                            </div>
                                            <div class="form-group">
                                                <label>ສະກຸນເງິນ</label>
                                                <select id="edit_currencys_id" name="currencys_id" required>
                                                    <option value="">ເລືອກສະກຸນເງິນ</option>
                                                </select>
                                            </div>
                                            <div class="form-group">
                                                <label>ອັດຕາແລກປ່ຽນ</label>
                                                <input type="text" id="edit_rate_display" name="rate_display" readonly style="background:#f5f5f5;">
                                                <input type="hidden" id="edit_rate_id" name="rate_id">
                                            </div>
                                                                                    </div>
                                        <div class="form-group" style="margin-top: 1rem;">
                                            <label>ຫົວຂໍ້</label>
                                            <input type="text" id="edit_section" name="section" placeholder="ຫົວຂໍ້ (Optional)">
                                        </div>
                                    </div>

                                    <!-- รายการสินค้า -->
                                    <div class="form-section products-section">
                                        <h3><i class="fas fa-box"></i> ລາຍການສິນຄ້າ</h3>
                                        <div style="overflow-x: auto;">
                                            <table class="products-table" id="editProductsTable">
                                                <thead>
                                                    <tr>
                                                        <th style="width: 30%;">ສິນຄ້າ</th>
                                                        <th style="width: 15%;">ປະເພດ</th>
                                                        <th style="width: 10%;">ຈຳນວນ</th>
                                                        <th style="width: 10%;">ຫົວໜ່ວຍ</th>
                                                        <th style="width: 12%;">ລາຄາ</th>
                                                        <th style="width: 12%;">ລວມ</th>
                                                        <th style="width: 8%;">ລຶບ</th>
                                                    </tr>
                                                </thead>
                                                <tbody id="editProductsTableBody">
                                                    <!-- Products will be added here dynamically -->
                                                </tbody>
                                            </table>
                                        </div>
                                        <button type="button" class="btn-add-product" onclick="addProductRow()">
                                            <i class="fas fa-plus"></i> ເພີ່ມສິນຄ້າ
                                        </button>
                                    </div>

                                    <!-- สรุปยอด -->
                                    <div class="form-section">
                                        <h3><i class="fas fa-calculator"></i> ສະຫຼຸບຍອດ</h3>
                                        <div class="form-grid">
                                            <div class="form-group">
                                                <label>ສ່ວນຫຼຸດ (ຈຳນວນເງິນ)</label>
                                                <input type="number" id="edit_order_discount" name="order_discount" min="0" step="0.01" value="0" onchange="calculateEditTotal()">
                                            </div>
                                            <div class="form-group">
                                                <label style="display: flex; align-items: center; gap: 0.5rem;">
                                                    <input type="checkbox" id="edit_vat_toggle" onchange="toggleVAT()" style="width: auto; margin: 0;">
                                                    <span>VAT 10%</span>
                                                </label>
                                                <input type="hidden" id="edit_order_tax_percent" name="order_tax_percent" value="0">
                                                <small style="color: var(--text-muted); margin-top: 0.25rem;">ຕິກເພື່ອເພີ່ມ VAT 10%</small>
                                            </div>
                                        </div>
                                        <div class="summary-box">
                                            <div class="summary-row">
                                                <span>ລວມຍ່ອຍ:</span>
                                                <span id="edit_subtotal_display">0.00</span>
                                            </div>
                                            <div class="summary-row">
                                                <span>ສ່ວນຫຼຸດ:</span>
                                                <span id="edit_discount_display">0.00</span>
                                            </div>
                                            <div class="summary-row">
                                                <span>VAT 10%:</span>
                                                <span id="edit_tax_display">0.00</span>
                                            </div>
                                            <div class="summary-row total">
                                                <span>ລວມທັງໝົດ:</span>
                                                <span id="edit_total_display">0.00</span>
                                            </div>
                                        </div>
                                    </div>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn-close" onclick="closeEditModal()">ຍົກເລີກ</button>
                                <button type="button" class="btn-save" onclick="saveEditOrder()">
                                    <i class="fas fa-save"></i> ບັນທຶກການແກ້ໄຂ
                                </button>
                            </div>
                        </div>
                    </div>
</body>
</html>