<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

session_start();

// ตรวจสอบการล็อกอิน
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'ไม่ได้รับอนุญาต']);
    exit();
}

// เชื่อมต่อฐานข้อมูล
require_once '../includes/db_con.php';

try {
    // รับข้อมูล JSON
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('ข้อมูลไม่ถูกต้อง');
    }
    
    $customer_id = $input['customer_id'] ?? '';
    $section = $input['section'] ?? '';
    $order_items = $input['order_items'] ?? [];
    $subtotal = $input['subtotal_lak'] ?? 0;
    $discount = $input['discount_lak'] ?? 0;
    $vat_amount = $input['vat_amount_lak'] ?? 0;
    $total_amount = $input['total_amount_lak'] ?? 0;
    $currency_id = $input['currency_id'] ?? 'CUR_1';
    $currency_rate = $input['currency_rate'] ?? 1;
    // รับ rate_id จาก frontend (ถ้ามี)
    $rate_id = (!empty($input['rate_id']) && $input['rate_id'] !== '0') ? $input['rate_id'] : null;
    // รับ remark_id จาก frontend
    $remark_id = (!empty($input['remark_id'])) ? $input['remark_id'] : null;
    // รับ detail_product_id จาก frontend
    $detail_product_id = (!empty($input['detail_product_id'])) ? $input['detail_product_id'] : null;
    
    if (empty($customer_id) || empty($order_items)) {
        throw new Exception('กรุณาใส่ข้อมูลที่จำเป็น');
    }
    
    // ถ้าไม่มี rate_id จาก frontend ให้ดึงจากฐานข้อมูล (เป็น fallback)
    if ($rate_id === null && $currency_id !== 'CUR_1') {
        $rate_stmt = $conn->prepare("SELECT rate_id FROM rates WHERE currencys_id = ? ORDER BY id DESC LIMIT 1");
        $rate_stmt->bind_param("s", $currency_id);
        $rate_stmt->execute();
        $rate_result = $rate_stmt->get_result();
        if ($rate_result->num_rows > 0) {
            $rate_row = $rate_result->fetch_assoc();
            $rate_id = $rate_row['rate_id'];
        }
        $rate_stmt->close();
    }
    // ถ้าเป็นเงินกีบ (CUR_1) ให้บังคับ NULL
    if ($currency_id === 'CUR_1') {
        $rate_id = null;
    }
    
    // คำนวณ order_tax_percent (10% หาก VAT > 0, ไม่งั้น 0%)
    $tax_percent = ($vat_amount > 0) ? 10.00 : 0.00;
    
    // แปลงยอดเงินเป็นสกุลเงินที่เลือก (ไม่ใช่ LAK อีกต่อไป)
    $order_discount_display = $discount / $currency_rate; // แปลงส่วนลดเป็นสกุลเงินที่เลือก
    $order_subtotal_display = $subtotal / $currency_rate; // แปลงยอดรวมเป็นสกุลเงินที่เลือก
    $order_tax_amount_display = $vat_amount / $currency_rate; // แปลง VAT เป็นสกุลเงินที่เลือก
    $order_total_display = $total_amount / $currency_rate; // แปลงยอดรวมสุดท้ายเป็นสกุลเงินที่เลือก
    
    // เริ่ม Transaction
    $conn->begin_transaction();
    
    try {
        // Generate Quotation Number (QTxxx/WST26)
        $qt_prefix = 'QT';
        $qt_suffix = '/WST26';
        $qt_query = "SELECT quotation_no FROM orders WHERE quotation_no LIKE CONCAT(?, '%', ?) ORDER BY id DESC LIMIT 1";
        $qt_stmt_check = $conn->prepare($qt_query);
        $qt_stmt_check->bind_param('ss', $qt_prefix, $qt_suffix);
        $qt_stmt_check->execute();
        $qt_res = $qt_stmt_check->get_result();
        
        $next_num = 1;
        if ($qt_res->num_rows > 0) {
            $last_qt = $qt_res->fetch_assoc()['quotation_no'];
            $pattern = '/^' . preg_quote($qt_prefix, '/') . '(\d+)' . preg_quote($qt_suffix, '/') . '$/';
            if (preg_match($pattern, $last_qt, $matches)) {
                $next_num = intval($matches[1]) + 1;
            }
        }
        $quotation_no = $qt_prefix . str_pad($next_num, 3, '0', STR_PAD_LEFT) . $qt_suffix;
        $qt_stmt_check->close();

        // Generate Invoice Number (INVxxx/WST26) immediately
        $inv_prefix = 'INV';
        $inv_suffix = '/WST26';
        $inv_query = "SELECT invoice_no FROM orders WHERE invoice_no LIKE CONCAT(?, '%', ?) ORDER BY id DESC LIMIT 1";
        $inv_stmt_check = $conn->prepare($inv_query);
        $inv_stmt_check->bind_param('ss', $inv_prefix, $inv_suffix);
        $inv_stmt_check->execute();
        $inv_res = $inv_stmt_check->get_result();
        
        $next_inv_num = 1;
        if ($inv_res->num_rows > 0) {
            $last_inv = $inv_res->fetch_assoc()['invoice_no'];
            $pattern = '/^' . preg_quote($inv_prefix, '/') . '(\d+)' . preg_quote($inv_suffix, '/') . '$/';
            if (preg_match($pattern, $last_inv, $matches)) {
                $next_inv_num = intval($matches[1]) + 1;
            }
        }
        $invoice_no = $inv_prefix . str_pad($next_inv_num, 3, '0', STR_PAD_LEFT) . $inv_suffix;
        $inv_stmt_check->close();

        // เพิ่มข้อมูลในตาราง orders (ใช้ trigger สำหรับ order_id)
        $order_stmt = $conn->prepare("
            INSERT INTO orders (
                quotation_no,
                invoice_no,
                order_date, 
                section,
                order_discount,
                order_subtotal,
                order_tax_percent,
                order_tax_amount, 
                order_total, 
                cus_id,
                currencys_id,
                rate_id,
                user_id,
                remark_id,
                detail_product_id
            ) VALUES (?, ?, CURDATE(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $user_id = $_SESSION['user_id'];
        // ถ้า rate_id เป็น null ให้ bind เป็น null (i.e. type s, value null)
        if ($rate_id === null) {
            $order_stmt->bind_param(
                "sssdddddssisss", 
                $quotation_no,
                $invoice_no,
                $section,
                $order_discount_display,
                $order_subtotal_display, 
                $tax_percent,
                $order_tax_amount_display, 
                $order_total_display, 
                $customer_id,
                $currency_id,
                $rate_id,
                $user_id,
                $remark_id,
                $detail_product_id
            );
        } else {
            $order_stmt->bind_param(
                "sssdddddssssss", 
                $quotation_no,
                $invoice_no,
                $section,
                $order_discount_display,
                $order_subtotal_display, 
                $tax_percent,
                $order_tax_amount_display, 
                $order_total_display, 
                $customer_id,
                $currency_id,
                $rate_id,
                $user_id,
                $remark_id,
                $detail_product_id
            );
        }
        
        if (!$order_stmt->execute()) {
            throw new Exception('ไม่สามารถสร้างออเดอร์ได้');
        }
        
        // ดึง order_id ที่ถูกสร้างโดย trigger
        $get_order_id = $conn->prepare("SELECT order_id FROM orders WHERE id = ?");
        $order_insert_id = $conn->insert_id;
        $get_order_id->bind_param("i", $order_insert_id);
        $get_order_id->execute();
        $order_result = $get_order_id->get_result();
        $order_row = $order_result->fetch_assoc();
        $generated_order_id = $order_row['order_id'];
        $get_order_id->close();
        
        // เพิ่มรายละเอียดสินค้าในตาราง orders_detail (ใช้ trigger สำหรับ order_detail_id)
        $detail_stmt = $conn->prepare("
            INSERT INTO orders_detail (
                order_id, 
                pro_id, 
                order_qty,  
                pro_price,      
                total
            ) VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($order_items as $item) {
            $pro_id = $item['id'];
            $quantity = $item['quantity'];
            $unit_price_lak = $item['price_lak'] ?? $item['basePriceLAK']; // ราคาต่อชิ้นเป็น LAK
            
            // แปลงราคาเป็นสกุลเงินที่เลือก
            $unit_price_display = $unit_price_lak / $currency_rate;
            $total_price = $unit_price_display * $quantity;
            
            $detail_stmt->bind_param(
                "ssidd", 
                $generated_order_id, 
                $pro_id, 
                $quantity, 
                $unit_price_display, 
                $total_price
            );
            
            if (!$detail_stmt->execute()) {
                throw new Exception('ไม่สามารถบันทึกรายละเอียดสินค้าได้');
            }
        }
        
        // Commit Transaction
        $conn->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'สร้างออเดอร์สำเร็จ',
            'order_id' => $generated_order_id
        ]);
        
        $order_stmt->close();
        $detail_stmt->close();
        
    } catch (Exception $e) {
        // Rollback Transaction
        $conn->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} finally {
    $conn->close();
}
?>