<?php
// file: add_products.php
session_start();

// ตรวจสอบการล็อกอิน
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

// เชื่อมต่อฐานข้อมูล
require_once 'includes/db_con.php';

// ข้อมูลผู้ใช้
$username = isset($_SESSION['username']) ? $_SESSION['username'] : 'ผู้ใช้';

// จัดการการส่งฟอร์ม
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // การลบข้อมูล
    if (isset($_POST['action']) && $_POST['action'] == 'delete') {
        $id = intval($_POST['id']);
        
        if (empty($id)) {
            echo json_encode(['success' => false, 'message' => 'ລະຫັດສິນຄ້າໄມ່ຖືກຕ້ອງ']);
            exit();
        }
        
        try {
            // ดึงข้อมูลรูปภาพก่อนลบ
            $img_query = $conn->prepare("SELECT pro_img FROM products WHERE id = ?");
            $img_query->bind_param("i", $id);
            $img_query->execute();
            $img_result = $img_query->get_result();
            $img_data = $img_result->fetch_assoc();
            $img_query->close();
            
            // ลบข้อมูล
            $delete_stmt = $conn->prepare("DELETE FROM products WHERE id = ?");
            $delete_stmt->bind_param("i", $id);
            
            if ($delete_stmt->execute()) {
                // ลบไฟล์รูปภาพถ้ามี
                if (!empty($img_data['pro_img']) && file_exists('img/products/' . $img_data['pro_img'])) {
                    unlink('img/products/' . $img_data['pro_img']);
                }
                
                echo json_encode(['success' => true, 'message' => 'ລຶບສິນຄ້າສຳເລັດແລ້ວ']);
            } else {
                echo json_encode(['success' => false, 'message' => 'ເກີດຂໍ້ຜິດພາດໃນການລຶບ']);
            }
            $delete_stmt->close();
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'ເກີດຂໍ້ຜິດພາດ: ' . $e->getMessage()]);
        }
        exit();
    }
    
    $action = trim($_POST['action'] ?? 'add');
    $pro_name = trim($_POST['pro_name']);
    $pro_price = trim($_POST['pro_price']);
    
    // Convert empty strings to NULL for optional fields
    $types_id = !empty($_POST['types_id']) ? trim($_POST['types_id']) : null;
    $unit_id = !empty($_POST['unit_id']) ? trim($_POST['unit_id']) : null;
    $remark_id = !empty($_POST['remark_id']) ? trim($_POST['remark_id']) : null;
    $detail_product_id = !empty($_POST['detail_product_id']) ? trim($_POST['detail_product_id']) : null;
    
    // จัดการการอัพโหลดรูปภาพ
    $pro_img = '';
    $upload_dir = 'img/products/';
    
    // สร้างโฟลเดอร์หากไม่มี
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0777, true);
    }
    
    $current_img = '';
    
    if ($action === 'edit') {
        $id = intval($_POST['product_id']);
        // ดึงรูปปัจจุบัน
        $current_img_query = $conn->prepare("SELECT pro_img FROM products WHERE id = ?");
        $current_img_query->bind_param("i", $id);
        $current_img_query->execute();
        $current_img_result = $current_img_query->get_result();
        if ($current_img_row = $current_img_result->fetch_assoc()) {
            $current_img = $current_img_row['pro_img'];
        }
        $current_img_query->close();
        $pro_img = $current_img; // ค่าเริ่มต้น
    }
    
    if (isset($_FILES['pro_img']) && $_FILES['pro_img']['error'] == 0) {
        $file_tmp = $_FILES['pro_img']['tmp_name'];
        $file_name = $_FILES['pro_img']['name'];
        $file_size = $_FILES['pro_img']['size'];
        $file_type = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        // ตรวจสอบประเภทไฟล์
        $allowed_types = array('jpg', 'jpeg', 'png', 'gif', 'webp');
        if (in_array($file_type, $allowed_types)) {
            // ตรวจสอบขนาดไฟล์ (ไม่เกิน 5MB)
            if ($file_size <= 5242880) {
                // สร้างชื่อไฟล์ใหม่เพื่อป้องกันการซ้ำ
                $new_filename = 'product_' . uniqid() . '.' . $file_type;
                $target_file = $upload_dir . $new_filename;
                
                if (move_uploaded_file($file_tmp, $target_file)) {
                    $pro_img = $new_filename;
                    // ลบรูปเก่าถ้า edit และมีรูปใหม่
                    if ($action === 'edit' && !empty($current_img) && $current_img !== $pro_img && file_exists($upload_dir . $current_img)) {
                        unlink($upload_dir . $current_img);
                    }
                }
            }
        }
    }
    
    if (empty($pro_name) || empty($pro_price)) {
        $message = 'ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ';
        $message_type = 'error';
    } elseif (!is_numeric($pro_price) || $pro_price <= 0) {
        $message = 'ກະລຸນາໃສ່ລາຄາທີ່ຖືກຕ້ອງ';
        $message_type = 'error'; 
    } else {
        try {
            if ($action === 'add') {
                // ตรวจสอบว่ามีสินค้านี้แล้วหรือไม่ (ตามชื่อเท่านั้น)
                $check_stmt = $conn->prepare("SELECT id FROM products WHERE pro_name = ?");
                $check_stmt->bind_param("s", $pro_name);
                $check_stmt->execute();
                
                if ($check_stmt->get_result()->num_rows > 0) {
                    $message = 'ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ';
                    $message_type = 'error';
                } else {
                    // เพิ่มสินค้าใหม่ (pro_id จะถูก auto-generate โดย trigger)
                    $insert_stmt = $conn->prepare("INSERT INTO products (pro_name, pro_price, pro_img, types_id, unit_id, remark_id, detail_product_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $insert_stmt->bind_param("sdsssss", $pro_name, $pro_price, $pro_img, $types_id, $unit_id, $remark_id, $detail_product_id);
                    
                    if ($insert_stmt->execute()) {
                        $message = 'ເພີ່ມສິນຄ້າສຳເລັດແລ້ວ';
                        $message_type = 'success';
                        
                        // ล้างข้อมูลในฟอร์ม
                        $_POST = [];
                    } else {
                        $message = 'ເກີດຂໍ້ຜິດພາດໃນການເພີ່ມສິນຄ້າ';
                        $message_type = 'error';
                    }
                    $insert_stmt->close();
                }
                $check_stmt->close();
            } elseif ($action === 'edit') {
                // ตรวจสอบ duplicate name exclude self
                $check_stmt = $conn->prepare("SELECT id FROM products WHERE pro_name = ? AND id != ?");
                $check_stmt->bind_param("si", $pro_name, $id);
                $check_stmt->execute();
                
                if ($check_stmt->get_result()->num_rows > 0) {
                    $message = 'ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ';
                    $message_type = 'error';
                } else {
                    // อัพเดทสินค้า
                    $update_stmt = $conn->prepare("UPDATE products SET pro_name = ?, pro_price = ?, pro_img = ?, types_id = ?, unit_id = ?, remark_id = ?, detail_product_id = ? WHERE id = ?");
                    $update_stmt->bind_param("sdsssssi", $pro_name, $pro_price, $pro_img, $types_id, $unit_id, $remark_id, $detail_product_id, $id);
                    
                    if ($update_stmt->execute()) {
                        $message = 'ແກ້ໄຂສິນຄ້າສຳເລັດແລ້ວ';
                        $message_type = 'success';
                        
                        // ล้างข้อมูลในฟอร์ม
                        $_POST = [];
                    } else {
                        $message = 'ເກີດຂໍ້ຜິດພາດໃນການແກ້ໄຂສິນຄ້າ';
                        $message_type = 'error';
                    }
                    $update_stmt->close();
                }
                $check_stmt->close();
            }
        } catch (Exception $e) {
            $message = 'ເກີດຂໍ້ຜິດພາດ: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// ดึงข้อมูลประเภทสินค้า
$types_query = "SELECT * FROM types_product ORDER BY types_name";
$types_result = $conn->query($types_query);

// ดึงข้อมูลหน่วย (units)
$units_query = "SELECT * FROM units ORDER BY unit_name";
$units_result = $conn->query($units_query);

// ดึงข้อมูลหมายเหตุ (remarks)
$remarks_query = "SELECT * FROM remark ORDER BY remark_text";
$remarks_result = $conn->query($remarks_query);

// ดึงข้อมูล detail_product
$details_query = "SELECT * FROM detail_product ORDER BY detail_text";
$details_result = $conn->query($details_query);

// ดึงข้อมูลสินค้าทั้งหมด
$products_query = "
    SELECT p.*, t.types_name, u.unit_name, r.remark_text, d.detail_text
    FROM products p
    LEFT JOIN types_product t ON p.types_id = t.types_id
    LEFT JOIN units u ON p.unit_id = u.unit_id
    LEFT JOIN remark r ON p.remark_id = r.remark_id
    LEFT JOIN detail_product d ON p.detail_product_id = d.detail_product_id
    ORDER BY p.pro_name
";
$products_result = $conn->query($products_query);
?>

<!DOCTYPE html>
<html lang="lo">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sokxay Bill - ເພີ່ມສິນຄ້າ</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="/sokxay/wayha.png">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Custom CSS -->
    <link rel="stylesheet" href="css/home.css">
    <link rel="stylesheet" href="css/forms.css">
    
    <style>
        /* Modal Styles */
        .modal {
            display: none !important;
            position: fixed !important;
            z-index: 9999 !important;
            left: 0 !important;
            top: 0 !important;
            width: 100% !important;
            height: 100% !important;
            background-color: rgba(0, 0, 0, 0.5) !important;
            overflow: auto !important;
        }
        
        .modal.show {
            display: block !important;
        }
        
        .modal-content {
            background-color: #fff;
            margin: 5% auto;
            padding: 0;
            border-radius: 8px;
            width: 90%;
            max-width: 700px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from { opacity: 0; transform: translateY(-50px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .modal-header {
            background: var(--primary-color);
            color: white;
            padding: 20px;
            border-radius: 8px 8px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h2 {
            margin: 0;
            font-size: 20px;
        }
        
        .close {
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            opacity: 0.7;
            transition: opacity 0.3s;
        }
        
        .close:hover {
            opacity: 1;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .modal-footer {
            background: #f8f9fa;
            padding: 15px 20px;
            border-radius: 0 0 8px 8px;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }

        /* Fix SweetAlert2 z-index to be above custom modal */
        .swal2-container {
            z-index: 10000 !important;
        }
        
        /* Fix SweetAlert2 input width */
        .swal2-popup .swal2-input {
            box-sizing: border-box !important;
            width: 80% !important;
           
            margin: 15px auto !important;
            height: 42px !important;
        }
    </style>
</head>
<body>
    <div class="app-container">
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content Area -->
        <main class="main-content">
            <!-- Header -->
            <header class="content-header">
                <div class="header-left">
                    <h1 class="page-title">ສິນຄ້າ</h1>
                    <p class="page-subtitle">ຈັດການສິນຄ້າໃນລະບົບ</p>
                </div>
                
                <div class="header-right">
                    <div class="stats-summary">
                        <div class="stat-item">
                            <i class="fas fa-box"></i>
                            <div>
                                <span class="stat-number"><?php echo $products_result->num_rows; ?></span>
                                <span class="stat-label">ລາຍການ</span>
                            </div>
                        </div>
                    </div>
                    <button class="btn-primary" onclick="openAddModal()">
                        <i class="fas fa-plus"></i>
                        ເພີ່ມສິນຄ້າ
                    </button>
                </div>
            </header>

            <!-- Content Body -->
            <div class="content-body">
                <div class="list-container">
                    <!-- Products List -->
                    <div class="list-card">
                        <div class="list-header">
                            <h3><i class="fas fa-list"></i> ລາຍການສິນຄ້າ</h3>
                            <div class="list-controls">
                                <div class="search-box small">
                                    <i class="fas fa-search"></i>
                                    <input type="text" id="productSearch" placeholder="ຄົ້ນຫາສິນຄ້າ...">
                                </div>
                                <select id="categoryFilter" class="filter-select">
                                    <option value="">ທຸກປະເພດ</option>
                                    <?php 
                                    // Reset result pointer
                                    $types_result->data_seek(0);
                                    while ($type = $types_result->fetch_assoc()): 
                                    ?>
                                        <option value="<?php echo $type['types_id']; ?>">
                                            <?php echo htmlspecialchars($type['types_name']); ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="table-container">
                            <?php if ($products_result->num_rows > 0): ?>
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th>ລະຫັດ</th>
                                            <th>ຮູບພາບ</th>
                                            <th>ຊື່ສິນຄ້າ</th>
                                            <th>ປະເພດ</th>
                                            <th>ຫົວໜ່ວຍ</th>
                                            <th>ລາຍລະອຽດ</th>
                                            <th>ໝາຍເຫດ</th>
                                            <th>ລາຄາ</th>
                                            <th>ການກະທຳ</th>
                                        </tr>
                                    </thead>
                                    <tbody id="productsTableBody">
                                        <?php while ($product = $products_result->fetch_assoc()): ?>
                                            <tr data-category="<?php echo $product['types_id']; ?>" data-unit="<?php echo $product['unit_id']; ?>" data-name="<?php echo strtolower($product['pro_name']); ?>">
                                                <td>
                                                    <span class="product-code"><?php echo htmlspecialchars($product['pro_id']); ?></span>
                                                </td>
                                                <td>
                                                    <?php if (!empty($product['pro_img']) && file_exists('img/products/' . $product['pro_img'])): ?>
                                                        <img src="img/products/<?php echo htmlspecialchars($product['pro_img']); ?>" alt="Product Image" class="product-image">
                                                    <?php else: ?>
                                                        <div class="no-image">
                                                            <i class="fas fa-image"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="product-info">
                                                        <span class="product-name"><?php echo htmlspecialchars($product['pro_name']); ?></span>
                                                    </div>
                                                </td>
                                                <td>
                                                    <?php if ($product['types_name']): ?>
                                                        <span class="category-badge"><?php echo htmlspecialchars($product['types_name']); ?></span>
                                                    <?php else: ?>
                                                        <span class="no-category">ບໍ່ມີປະເພດ</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (!empty($product['unit_name'])): ?>
                                                        <span class="unit-badge"><?php echo htmlspecialchars($product['unit_name']); ?></span>
                                                    <?php else: ?>
                                                        <span class="no-unit">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (!empty($product['detail_text'])): ?>
                                                        <span class="detail-text"><?php echo htmlspecialchars($product['detail_text']); ?></span>
                                                    <?php else: ?>
                                                        <span class="no-detail">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (!empty($product['remark_text'])): ?>
                                                        <span class="remark-text"><?php echo htmlspecialchars($product['remark_text']); ?></span>
                                                    <?php else: ?>
                                                        <span class="no-remark">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="product-price">₭ <?php echo number_format($product['pro_price'], 2); ?></span>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <button class="btn-action btn-edit" onclick="editProduct(<?php echo $product['id']; ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn-action btn-delete" onclick="deleteProduct(<?php echo $product['id']; ?>)">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <div class="no-data">
                                    <i class="fas fa-box-open"></i>
                                    <p>ຍັງບໍ່ມີສິນຄ້າໃນລະບົບ</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Add Product Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-box"></i> ເພີ່ມສິນຄ້າໃໝ່</h2>
                <span class="close" onclick="closeAddModal()">&times;</span>
            </div>
            
            <div class="modal-body">
                <form id="addForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="add">
                    <div class="form-group full-width">
                        <label for="modal_pro_name"><i class="fas fa-tag"></i> ຊື່ສິນຄ້າ <span class="required">*</span></label>
                        <input type="text" id="modal_pro_name" name="pro_name" required placeholder="ໃສ່ຊື່ສິນຄ້າ">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="modal_pro_price">
                                <i class="fas fa-dollar-sign"></i>
                                ລາຄາ (ກີບ) <span class="required">*</span>
                            </label>
                            <div class="price-input-group">
                                <span class="currency-symbol">₭</span>
                                <input type="number" 
                                       id="modal_pro_price" 
                                       name="pro_price" 
                                       step="0.01"
                                       min="0"
                                       placeholder="0.00"
                                       required>
                            </div>
                            <small>ໃສ່ລາຄາຂາຍເປັນກີບລາວ</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="modal_types_id">
                                <i class="fas fa-tags"></i>
                                ປະເພດສິນຄ້າ
                            </label>
                            <select id="modal_types_id" name="types_id">
                                <option value="">-- ເລືອກປະເພດ --</option>
                                <?php 
                                // Reset result pointer for modal
                                $types_result->data_seek(0);
                                while ($type = $types_result->fetch_assoc()): 
                                ?>
                                    <option value="<?php echo $type['types_id']; ?>">
                                        <?php echo htmlspecialchars($type['types_name']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກປະເພດສິນຄ້າ ຫຼື <a href="#" onclick="addNewAttribute('add_type', 'modal_types_id', 'ເພີ່ມປະເພດໃໝ່'); return false;">ເພີ່ມປະເພດໃໝ່</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="modal_unit_id">
                                <i class="fas fa-balance-scale"></i>
                                ຫົວໜ່ວຍ
                            </label>
                            <select id="modal_unit_id" name="unit_id">
                                <option value="">-- ເລືອກຫົວໜ່ວຍ --</option>
                                <?php
                                // Reset units pointer for modal
                                if ($units_result) $units_result->data_seek(0);
                                while ($u = $units_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $u['unit_id']; ?>"><?php echo htmlspecialchars($u['unit_name']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກຫົວໜ່ວຍ ຫຼື <a href="#" onclick="addNewAttribute('add_unit', 'modal_unit_id', 'ເພີ່ມຫົວໜ່ວຍ'); return false;">ເພີ່ມຫົວໜ່ວຍ</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="modal_remark_id">
                                <i class="fas fa-sticky-note"></i>
                                ໝາຍເຫດ
                            </label>
                            <select id="modal_remark_id" name="remark_id">
                                <option value="">-- ເລືອກໝາຍເຫດ --</option>
                                <?php
                                if ($remarks_result) $remarks_result->data_seek(0);
                                while ($r = $remarks_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $r['remark_id']; ?>"><?php echo htmlspecialchars($r['remark_text']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກໝາຍເຫດ ຫຼື <a href="#" onclick="addNewAttribute('add_remark', 'modal_remark_id', 'ເພີ່ມໝາຍເຫດ'); return false;">ເພີ່ມໝາຍເຫດ</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="modal_detail_product_id">
                                <i class="fas fa-info-circle"></i>
                                ລາຍລະອຽດ
                            </label>
                            <select id="modal_detail_product_id" name="detail_product_id">
                                <option value="">-- ເລືອກລາຍລະອຽດ --</option>
                                <?php
                                if ($details_result) $details_result->data_seek(0);
                                while ($d = $details_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $d['detail_product_id']; ?>"><?php echo htmlspecialchars($d['detail_text']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກລາຍລະອຽດ ຫຼື <a href="#" onclick="addNewAttribute('add_detail', 'modal_detail_product_id', 'ເພີ່ມລາຍລະອຽດ'); return false;">ເພີ່ມລາຍລະອຽດ</a></small>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group full-width">
                            <label for="modal_pro_img"><i class="fas fa-image"></i> ຮູບພາບສິນຄ້າ</label>
                            <div class="image-upload-container">
                                <input type="file" id="modal_pro_img" name="pro_img" accept="image/*" onchange="previewImage(this, 'add')">
                                <div class="image-preview" id="addImagePreview" style="display: none;">
                                    <img id="addPreviewImg" src="" alt="Preview">
                                    <button type="button" class="remove-image" onclick="removeImage('add')">&times;</button>
                                </div>
                                <div class="upload-placeholder" id="addUploadPlaceholder">
                                    <i class="fas fa-cloud-upload-alt"></i>
                                    <span>ຄລິກເພື່ອເລືອກຮູບພາບ</span>
                                    <small>JPG, PNG, GIF, WEBP (ສູງສຸດ 5MB)</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group full-width">
                            <div style="background: #e8f4fd; border: 1px solid #bee5eb; padding: 15px; border-radius: 8px;">
                                <h4 style="margin: 0 0 10px 0; color: #0c5460; font-size: 14px;">
                                    <i class="fas fa-info-circle"></i> ຂໍ້ມູນການໃຊ້ງານ
                                </h4>
                                <ul style="margin: 0; padding-left: 20px; color: #0c5460; font-size: 13px;">
                                    <li>ລະຫັດສິນຄ້າ (Product ID) ຈະຖືກສ້າງອັດຕະໂນມັດ</li>
                                    <li>ຮູບແບບ: PRO_1, PRO_2, PRO_3, ...</li>
                                    <li>ບໍ່ຈຳເປັນຕ້ອງໃສ່ລະຫັດດ້ວຍຕົນເອງ</li>
                                    <li>ໃສ່ແຕ່ຊື່ສິນຄ້າແລະຂໍ້ມູນອື່ນໆ ທີ່ຕ້ອງການ</li>
                                </ul>
                            </div>
                        </div>
                    </div>

                    <!-- Product Preview in Modal -->
                    <div class="form-row">
                        <div class="form-group full-width">
                            <div class="product-preview" id="addModalProductPreview" style="display: none;">
                                <h4><i class="fas fa-eye"></i> ຕົວຢ່າງສິນຄ້າ</h4>
                                <div class="preview-card">
                                    <div class="preview-icon">
                                        <i class="fas fa-box"></i>
                                    </div>
                                    <div class="preview-info">
                                        <h3 id="addModalPreviewName">ຊື່ສິນຄ້າ</h3>
                                        <div id="addModalPreviewCategory" class="preview-category">ປະເພດສິນຄ້າ</div>
                                        <div id="addModalPreviewPrice" class="preview-price">₭ 0</div>
                                        <div id="addModalPreviewCode" class="preview-code">ລະຫັດສິນຄ້າ</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeAddModal()">
                    <i class="fas fa-times"></i>
                    ຍົກເລີກ
                </button>
                <button type="button" class="btn-primary" onclick="submitAddForm()">
                    <i class="fas fa-save"></i>
                    ບັນທຶກສິນຄ້າ
                </button>
            </div>
        </div>
    </div>

    <!-- Edit Product Modal -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-edit"></i> ແກ້ໄຂສິນຄ້າ</h2>
                <span class="close" onclick="closeEditModal()">&times;</span>
            </div>
            
            <div class="modal-body">
                <form id="editForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="product_id" id="edit_product_id">
                    <div class="form-group full-width">
                        <label for="edit_pro_name"><i class="fas fa-tag"></i> ຊື່ສິນຄ້າ <span class="required">*</span></label>
                        <input type="text" id="edit_pro_name" name="pro_name" required placeholder="ໃສ່ຊື່ສິນຄ້າ">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_pro_price">
                                <i class="fas fa-dollar-sign"></i>
                                ລາຄາ (ກີບ) <span class="required">*</span>
                            </label>
                            <div class="price-input-group">
                                <span class="currency-symbol">₭</span>
                                <input type="number" 
                                       id="edit_pro_price" 
                                       name="pro_price" 
                                       step="0.01"
                                       min="0"
                                       placeholder="0.00"
                                       required>
                            </div>
                            <small>ໃສ່ລາຄາຂາຍເປັນກີບລາວ</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="edit_types_id">
                                <i class="fas fa-tags"></i>
                                ປະເພດສິນຄ້າ
                            </label>
                            <select id="edit_types_id" name="types_id">
                                <option value="">-- ເລືອກປະເພດ --</option>
                                <?php 
                                // Reset result pointer for modal
                                $types_result->data_seek(0);
                                while ($type = $types_result->fetch_assoc()): 
                                ?>
                                    <option value="<?php echo $type['types_id']; ?>">
                                        <?php echo htmlspecialchars($type['types_name']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກປະເພດສິນຄ້າ ຫຼື <a href="#" onclick="addNewAttribute('add_type', 'edit_types_id', 'ເພີ່ມປະເພດໃໝ່'); return false;">ເພີ່ມປະເພດໃໝ່</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="edit_unit_id">
                                <i class="fas fa-balance-scale"></i>
                                ຫົວໜ່ວຍ
                            </label>
                            <select id="edit_unit_id" name="unit_id">
                                <option value="">-- ເລືອກຫົວໜ່ວຍ --</option>
                                <?php
                                if ($units_result) $units_result->data_seek(0);
                                while ($u = $units_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $u['unit_id']; ?>"><?php echo htmlspecialchars($u['unit_name']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກຫົວໜ່ວຍ ຫຼື <a href="#" onclick="addNewAttribute('add_unit', 'edit_unit_id', 'ເພີ່ມຫົວໜ່ວຍ'); return false;">ເພີ່ມຫົວໜ່ວຍ</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="edit_remark_id">
                                <i class="fas fa-sticky-note"></i>
                                ໝາຍເຫດ
                            </label>
                            <select id="edit_remark_id" name="remark_id">
                                <option value="">-- ເລືອກໝາຍເຫດ --</option>
                                <?php
                                if ($remarks_result) $remarks_result->data_seek(0);
                                while ($r = $remarks_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $r['remark_id']; ?>"><?php echo htmlspecialchars($r['remark_text']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກໝາຍເຫດ ຫຼື <a href="#" onclick="addNewAttribute('add_remark', 'edit_remark_id', 'ເພີ່ມໝາຍເຫດ'); return false;">ເພີ່ມໝາຍເຫດ</a></small>
                        </div>
                        
                        <div class="form-group">
                            <label for="edit_detail_product_id">
                                <i class="fas fa-info-circle"></i>
                                ລາຍລະອຽດ
                            </label>
                            <select id="edit_detail_product_id" name="detail_product_id">
                                <option value="">-- ເລືອກລາຍລະອຽດ --</option>
                                <?php
                                if ($details_result) $details_result->data_seek(0);
                                while ($d = $details_result->fetch_assoc()):
                                ?>
                                    <option value="<?php echo $d['detail_product_id']; ?>"><?php echo htmlspecialchars($d['detail_text']); ?></option>
                                <?php endwhile; ?>
                            </select>
                            <small>ເລືອກລາຍລະອຽດ ຫຼື <a href="#" onclick="addNewAttribute('add_detail', 'edit_detail_product_id', 'ເພີ່ມລາຍລະອຽດ'); return false;">ເພີ່ມລາຍລະອຽດ</a></small>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group full-width">
                            <label for="edit_pro_img"><i class="fas fa-image"></i> ຮູບພາບສິນຄ້າ</label>
                            <div class="image-upload-container">
                                <input type="file" id="edit_pro_img" name="pro_img" accept="image/*" onchange="previewImage(this, 'edit')">
                                <div class="image-preview" id="editImagePreview" style="display: none;">
                                    <img id="editPreviewImg" src="" alt="Preview">
                                    <button type="button" class="remove-image" onclick="removeImage('edit')">&times;</button>
                                </div>
                                <div class="upload-placeholder" id="editUploadPlaceholder">
                                    <i class="fas fa-cloud-upload-alt"></i>
                                    <span>ຄລິກເພື່ອເລືອກຮູບພາບໃໝ່ (ເລືອກເພື່ອປ່ຽນ)</span>
                                    <small>JPG, PNG, GIF, WEBP (ສູງສຸດ 5MB)</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Product Preview in Edit Modal -->
                    <div class="form-row">
                        <div class="form-group full-width">
                            <div class="product-preview" id="editModalProductPreview" style="display: none;">
                                <h4><i class="fas fa-eye"></i> ຕົວຢ່າງສິນຄ້າ</h4>
                                <div class="preview-card">
                                    <div class="preview-icon">
                                        <i class="fas fa-box"></i>
                                    </div>
                                    <div class="preview-info">
                                        <h3 id="editModalPreviewName">ຊື່ສິນຄ້າ</h3>
                                        <div id="editModalPreviewCategory" class="preview-category">ປະເພດສິນຄ້າ</div>
                                        <div id="editModalPreviewPrice" class="preview-price">₭ 0</div>
                                        <div id="editModalPreviewCode" class="preview-code">ລະຫັດສິນຄ້າ</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeEditModal()">
                    <i class="fas fa-times"></i>
                    ຍົກເລີກ
                </button>
                <button type="button" class="btn-primary" onclick="submitEditForm()">
                    <i class="fas fa-save"></i>
                    ອັບເດດສິນຄ້າ
                </button>
            </div>
        </div>
    </div>

    <!-- Sweet Alert -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <!-- Sidebar JavaScript -->
    <script src="js/sidebar.js"></script>
    
    <script>
        // Function to add new attributes (Type, Unit, Remark, Detail)
        function addNewAttribute(action, selectId, title) {
            Swal.fire({
                title: title,
                input: 'text',
                inputLabel: 'ກະລຸນາໃສ່ຂໍ້ມູນ',
                showCancelButton: true,
                confirmButtonText: 'ບັນທຶກ',
                cancelButtonText: 'ຍົກເລີກ',
                showLoaderOnConfirm: true,
                preConfirm: (value) => {
                    if (!value) {
                        Swal.showValidationMessage('ກະລຸນາໃສ່ຂໍ້ມູນ');
                        return false;
                    }
                    return fetch('api/add_product_attributes.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            action: action,
                            value: value
                        })
                    })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error(response.statusText)
                        }
                        return response.json()
                    })
                    .catch(error => {
                        Swal.showValidationMessage(
                            `Request failed: ${error}`
                        )
                    })
                },
                allowOutsideClick: () => !Swal.isLoading()
            }).then((result) => {
                if (result.isConfirmed) {
                    if (result.value.success) {
                        // Add new option to select
                        const select = document.getElementById(selectId);
                        if (select) {
                            const option = new Option(result.value.name, result.value.id);
                            option.selected = true;
                            select.add(option);
                        }
                        
                        Swal.fire({
                            icon: 'success',
                            title: 'ສຳເລັດ',
                            text: result.value.message,
                            timer: 1500,
                            showConfirmButton: false
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'ຜິດພາດ',
                            text: result.value.message
                        });
                    }
                }
            })
        }

        // Modal Functions for Add
        function openAddModal() {
            console.log('Opening add modal...');
            const modal = document.getElementById('addModal');
            const form = document.getElementById('addForm');
            const preview = document.getElementById('addModalProductPreview');
            
            if (modal) {
                modal.style.display = 'block';
                modal.style.visibility = 'visible';
                modal.classList.add('show');
                console.log('Add modal displayed');
            } else {
                console.error('Add modal element not found');
                alert('Add modal element not found! Check if modal HTML exists.');
            }
            
            if (form) {
                form.reset();
            }
            if (preview) {
                preview.style.display = 'none';
            }
            removeImage('add');
        }
        
        function closeAddModal() {
            console.log('Closing add modal...');
            const modal = document.getElementById('addModal');
            if (modal) {
                modal.style.display = 'none';
                modal.style.visibility = 'hidden';
                modal.classList.remove('show');
            }
            removeImage('add');
        }
        
        function submitAddForm() {
            const form = document.getElementById('addForm');
            const formData = new FormData(form);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(html => {
                if (html.includes('ເພີ່ມສິນຄ້າສຳເລັດແລ້ວ') || html.includes('ແກ້ໄຂສິນຄ້າສຳເລັດແລ້ວ')) {
                    let successText = html.includes('ເພີ່ມສິນຄ້າສຳເລັດແລ້ວ') ? 'ເພີ່ມສິນຄ້າໃໝ່ສຳເລັດແລ້ວ' : 'ແກ້ໄຂສິນຄ້າສຳເລັດແລ້ວ';
                    closeAddModal();
                    Swal.fire({
                        icon: 'success',
                        title: 'ສຳເລັດ!',
                        text: successText,
                        timer: 2000,
                        showConfirmButton: false
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    let errorMessage = 'ເກີດຂໍ້ຜິດພາດ';
                    if (html.includes('ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ')) {
                        errorMessage = 'ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ';
                    } else if (html.includes('ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ')) {
                        errorMessage = 'ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ';
                    } else if (html.includes('ກະລຸນາໃສ່ລາຄາທີ່ຖືກຕ້ອງ')) {
                        errorMessage = 'ກະລຸນາໃສ່ລາຄາທີ່ຖືກຕ້ອງ';
                    }
                    
                    Swal.fire({
                        icon: 'error',
                        title: 'ຜິດພາດ!',
                        text: errorMessage
                    });
                }
            })
            .catch(error => {
                console.error('Error:', error);
                Swal.fire({
                    icon: 'error',
                    title: 'ຜິດພາດ!',
                    text: 'ເກີດຂໍ້ຜິດພາດໃນການເຊື່ອມຕໍ່'
                });
            });
        }

        // Modal Functions for Edit
        function openEditModal() {
            console.log('Opening edit modal...');
            const modal = document.getElementById('editModal');
            if (modal) {
                modal.style.display = 'block';
                modal.style.visibility = 'visible';
                modal.classList.add('show');
                console.log('Edit modal displayed');
            } else {
                console.error('Edit modal element not found');
            }
        }
        
        function closeEditModal() {
            console.log('Closing edit modal...');
            const modal = document.getElementById('editModal');
            if (modal) {
                modal.style.display = 'none';
                modal.style.visibility = 'hidden';
                modal.classList.remove('show');
            }
            document.getElementById('editForm').reset();
            document.getElementById('editModalProductPreview').style.display = 'none';
            removeImage('edit');
        }
        
        function submitEditForm() {
            const form = document.getElementById('editForm');
            const formData = new FormData(form);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(html => {
                if (html.includes('ເພີ່ມສິນຄ້າສຳເລັດແລ້ວ') || html.includes('ແກ້ໄຂສິນຄ້າສຳເລັດແລ້ວ')) {
                    let successText = html.includes('ເພີ່ມສິນຄ້າສຳເລັດແລ້ວ') ? 'ເພີ່ມສິນຄ້າໃໝ່ສຳເລັດແລ້ວ' : 'ແກ້ໄຂສິນຄ້າສຳເລັດແລ້ວ';
                    closeEditModal();
                    Swal.fire({
                        icon: 'success',
                        title: 'ສຳເລັດ!',
                        text: successText,
                        timer: 2000,
                        showConfirmButton: false
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    let errorMessage = 'ເກີດຂໍ້ຜິດພາດ';
                    if (html.includes('ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ')) {
                        errorMessage = 'ມີສິນຄ້ານີ້ໃນລະບົບແລ້ວ';
                    } else if (html.includes('ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ')) {
                        errorMessage = 'ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ';
                    } else if (html.includes('ກະລຸນາໃສ່ລາຄາທີ່ຖືກຕ້ອງ')) {
                        errorMessage = 'ກະລຸນາໃສ່ລາຄາທີ່ຖືກຕ້ອງ';
                    }
                    
                    Swal.fire({
                        icon: 'error',
                        title: 'ຜິດພາດ!',
                        text: errorMessage
                    });
                }
            })
            .catch(error => {
                console.error('Error:', error);
                Swal.fire({
                    icon: 'error',
                    title: 'ຜິດພາດ!',
                    text: 'ເກີດຂໍ້ຜິດພາດໃນການເຊື່ອມຕໍ່'
                });
            });
        }

        // Edit Product Function
        function editProduct(id) {
            fetch(`get_product_data.php?action=get_product&id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        Swal.fire({
                            icon: 'error',
                            title: 'ຜິດພາດ!',
                            text: data.error
                        });
                        return;
                    }
                    // Populate form
                    document.getElementById('edit_pro_name').value = data.pro_name || '';
                    document.getElementById('edit_pro_price').value = data.pro_price || '';
                    document.getElementById('edit_types_id').value = data.types_id || '';
                    document.getElementById('edit_unit_id').value = data.unit_id || '';
                    document.getElementById('edit_remark_id').value = data.remark_id || '';
                    document.getElementById('edit_detail_product_id').value = data.detail_product_id || '';
                    document.getElementById('edit_product_id').value = data.id;
                    
                    // Handle image preview
                    const editImagePreview = document.getElementById('editImagePreview');
                    const editUploadPlaceholder = document.getElementById('editUploadPlaceholder');
                    const editPreviewImg = document.getElementById('editPreviewImg');
                    if (data.pro_img) {
                        editPreviewImg.src = `img/products/${data.pro_img}`;
                        editImagePreview.style.display = 'block';
                        editUploadPlaceholder.style.display = 'none';
                    } else {
                        editImagePreview.style.display = 'none';
                        editUploadPlaceholder.style.display = 'flex';
                    }
                    
                    // Update preview
                    updateEditPreview();
                    
                    // Open modal
                    openEditModal();
                })
                .catch(error => {
                    console.error('Error fetching product:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'ຜິດພາດ!',
                        text: 'ເກີດຂໍ້ຜິດພາດໃນການດຶງຂໍ້ມູນ'
                    });
                });
        }

        // Close modals when clicking outside
        window.addEventListener('click', function(event) {
            const addModal = document.getElementById('addModal');
            const editModal = document.getElementById('editModal');
            if (event.target === addModal) {
                closeAddModal();
            } else if (event.target === editModal) {
                closeEditModal();
            }
        });

        // Preview Image Function (generic for add/edit)
        function previewImage(input, modalType) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                const prefix = modalType === 'edit' ? 'edit' : 'add';
                
                reader.onload = function(e) {
                    document.getElementById(`${prefix}PreviewImg`).src = e.target.result;
                    document.getElementById(`${prefix}ImagePreview`).style.display = 'block';
                    document.getElementById(`${prefix}UploadPlaceholder`).style.display = 'none';
                    if (modalType === 'edit') updateEditPreview();
                };
                
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Remove Image Function (generic)
        function removeImage(modalType) {
            const prefix = modalType === 'edit' ? 'edit' : 'add';
            const fileInput = document.getElementById(`${prefix}_pro_img`);
            const imagePreview = document.getElementById(`${prefix}ImagePreview`);
            const uploadPlaceholder = document.getElementById(`${prefix}UploadPlaceholder`);
            
            if (fileInput) fileInput.value = '';
            if (imagePreview) imagePreview.style.display = 'none';
            if (uploadPlaceholder) uploadPlaceholder.style.display = 'flex';
            
            // For edit, show current image if exists, but since remove, hide preview
            if (modalType === 'edit') {
                const currentImg = document.getElementById('edit_pro_img').dataset.current || '';
                if (!currentImg) {
                    document.getElementById('editUploadPlaceholder').style.display = 'flex';
                }
            }
        }
        
        // Update Add Preview
        function updateAddPreview() {
            const name = document.getElementById('modal_pro_name').value;
            const price = document.getElementById('modal_pro_price').value;
            const typeSelect = document.getElementById('modal_types_id');
            const category = typeSelect ? typeSelect.options[typeSelect.selectedIndex].text : '';
            const unitSelect = document.getElementById('modal_unit_id');
            const unitText = unitSelect && unitSelect.selectedIndex>=0 ? unitSelect.options[unitSelect.selectedIndex].text : '';
            
            const preview = document.getElementById('addModalProductPreview');
            if (preview) {
                if (name || price) {
                    preview.style.display = 'block';
                    document.getElementById('addModalPreviewName').textContent = name || 'ບໍ່ມີຊື່';
                    document.getElementById('addModalPreviewPrice').textContent = price ? `₭ ${price}` : '₭ 0';
                    document.getElementById('addModalPreviewCode').textContent = 'AUTO';
                    const catText = category === '-- ເລືອກປະເພດ --' ? 'ບໍ່ມີປະເພດ' : category;
                    document.getElementById('addModalPreviewCategory').textContent = unitText ? (catText + ' • ' + unitText) : catText;
                } else {
                    preview.style.display = 'none';
                }
            }
        }

        // Update Edit Preview
        function updateEditPreview() {
            const name = document.getElementById('edit_pro_name').value;
            const price = document.getElementById('edit_pro_price').value;
            const typeSelect = document.getElementById('edit_types_id');
            const category = typeSelect ? typeSelect.options[typeSelect.selectedIndex].text : '';
            const unitSelect = document.getElementById('edit_unit_id');
            const unitText = unitSelect && unitSelect.selectedIndex>=0 ? unitSelect.options[unitSelect.selectedIndex].text : '';
            
            const preview = document.getElementById('editModalProductPreview');
            if (preview) {
                if (name || price) {
                    preview.style.display = 'block';
                    document.getElementById('editModalPreviewName').textContent = name || 'ບໍ່ມີຊື່';
                    document.getElementById('editModalPreviewPrice').textContent = price ? `₭ ${price}` : '₭ 0';
                    document.getElementById('editModalPreviewCode').textContent = 'AUTO';
                    const catText = category === '-- ເລືອກປະເພດ --' ? 'ບໍ່ມີປະເພດ' : category;
                    document.getElementById('editModalPreviewCategory').textContent = unitText ? (catText + ' • ' + unitText) : catText;
                } else {
                    preview.style.display = 'none';
                }
            }
        }

        // Add event listeners
        document.addEventListener('DOMContentLoaded', function() {
            // Add modal listeners
            const addInputs = ['modal_pro_name', 'modal_pro_price', 'modal_types_id', 'modal_unit_id'];
            addInputs.forEach(id => {
                const element = document.getElementById(id);
                if (element) {
                    element.addEventListener('input', updateAddPreview);
                    element.addEventListener('change', updateAddPreview);
                }
            });
            
            // Edit modal listeners
            const editInputs = ['edit_pro_name', 'edit_pro_price', 'edit_types_id', 'edit_unit_id'];
            editInputs.forEach(id => {
                const element = document.getElementById(id);
                if (element) {
                    element.addEventListener('input', updateEditPreview);
                    element.addEventListener('change', updateEditPreview);
                }
            });
            
            // Search and filter
            const searchInput = document.getElementById('productSearch');
            const filterSelect = document.getElementById('categoryFilter');
            
            if (searchInput) {
                searchInput.addEventListener('input', filterProducts);
            }
            
            if (filterSelect) {
                filterSelect.addEventListener('change', filterProducts);
            }
        });

        // Search and Filter
        function filterProducts() {
            const searchTerm = document.getElementById('productSearch').value.toLowerCase();
            const categoryFilter = document.getElementById('categoryFilter').value;
            const rows = document.querySelectorAll('#productsTableBody tr');
            
            rows.forEach(row => {
                const name = row.dataset.name || '';
                const category = row.dataset.category || '';
                
                const matchesSearch = name.includes(searchTerm);
                const matchesCategory = !categoryFilter || category === categoryFilter;
                
                row.style.display = (matchesSearch && matchesCategory) ? '' : 'none';
            });
        }

        // Delete Product Function
        function deleteProduct(id) {
            Swal.fire({
                title: 'ຢືນຢັນການລຶບ',
                text: "ຕ້ອງການລຶບສິນຄ້ານີ້ບໍ່?",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'ລຶບ',
                cancelButtonText: 'ຍົກເລີກ'
            }).then((result) => {
                if (result.isConfirmed) {
                    // ส่งคำขอลบ
                    const formData = new FormData();
                    formData.append('action', 'delete');
                    formData.append('id', id);
                    
                    fetch(window.location.href, {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'ສຳເລັດ!',
                                text: data.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'ຜິດພາດ!',
                                text: data.message
                            });
                        }
                    })
                    .catch(error => {
                        Swal.fire({
                            icon: 'error',
                            title: 'ຜິດພາດ!',
                            text: 'ເກີດຂໍ້ຜິດພາດໃນການເຊື່ອມຕໍ່'
                        });
                    });
                }
            });
        }

        // Show success message if exists
        <?php if ($message_type === 'success'): ?>
            Swal.fire({
                icon: 'success',
                title: 'ສຳເລັດ!',
                text: '<?php echo $message; ?>',
                timer: 3000,
                showConfirmButton: false
            });
        <?php endif; ?>
    </script>

    <style>
        .input-group {
            display: flex;
            align-items: stretch;
        }

        .input-group input {
            flex: 1;
            border-radius: 6px 0 0 6px;
        }

        .btn-generate {
            background: #17a2b8;
            color: white;
            border: none;
            padding: 0 15px;
            border-radius: 0 6px 6px 0;
            cursor: pointer;
            border-left: 1px solid #dee2e6;
        }

        .btn-generate:hover {
            background: #138496;
        }

        .price-input-group {
            position: relative;
        }

        .currency-symbol {
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
            font-weight: 600;
            z-index: 1;
        }

        .price-input-group input {
            padding-left: 35px;
        }

        .product-preview {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            border: 1px solid #dee2e6;
        }

        .product-preview h4 {
            margin: 0 0 15px 0;
            color: #495057;
        }

        .preview-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            display: flex;
            align-items: center;
            gap: 15px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .preview-icon {
            font-size: 40px;
            color: var(--primary-color);
        }

        .preview-info h3 {
            margin: 0 0 5px 0;
            color: var(--text-color);
        }

        .preview-category {
            font-size: 12px;
            color: #6c757d;
            margin-bottom: 5px;
        }

        .preview-price {
            font-size: 18px;
            font-weight: 700;
            color: var(--primary-color);
            margin-bottom: 5px;
        }

        .preview-code {
            font-size: 12px;
            color: #6c757d;
            font-family: monospace;
        }

        .list-controls {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .search-box.small {
            width: 250px;
        }

        .filter-select {
            padding: 8px 12px;
            border: 1px solid #dee2e6;
            border-radius: 6px;
            background: white;
            min-width: 150px;
        }

        .product-code {
            background: #e3f2fd;
            color: #1976d2;
            padding: 4px 8px;
            border-radius: 4px;
            font-family: monospace;
            font-weight: 600;
            font-size: 12px;
        }

        .category-badge {
            background: #e8f5e8;
            color: #2e7d32;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .unit-badge {
            background: #f0f4ff;
            color: #1e3a8a;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .no-unit {
            color: #6c757d;
            font-size: 12px;
            font-style: italic;
        }

        .no-category {
            color: #6c757d;
            font-style: italic;
            font-size: 12px;
        }

        .product-price {
            font-weight: 600;
            color: var(--primary-color);
        }

        .stats-summary {
            display: flex;
            gap: 20px;
        }

        .stat-item {
            display: flex;
            align-items: center;
            gap: 10px;
            background: white;
            padding: 12px 16px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }

        .stat-item i {
            font-size: 24px;
            color: var(--primary-color);
        }

        .stat-number {
            font-size: 18px;
            font-weight: 700;
            color: var(--text-color);
            display: block;
        }

        .stat-label {
            font-size: 12px;
            color: #6c757d;
        }

        .full-width {
            flex: 1 1 100%;
        }

        @media (max-width: 768px) {
            .list-controls {
                flex-direction: column;
                align-items: stretch;
            }
            
            .search-box.small {
                width: 100%;
            }
            
            .stats-summary {
                justify-content: center;
            }
            
            .input-group {
                flex-direction: column;
            }
            
            .input-group input {
                border-radius: 6px 6px 0 0;
            }
            
            .btn-generate {
                border-radius: 0 0 6px 6px;
                border-left: none;
                border-top: 1px solid #dee2e6;
            }
        }
        
        /* Button States Fix */
        .btn-primary:active,
        .btn-primary:focus,
        .btn-primary.active {
            background: var(--primary-color) !important;
            color: white !important;
            border-color: var(--primary-color) !important;
            box-shadow: 0 0 0 3px rgba(13, 110, 253, 0.25) !important;
        }
        
        .btn-secondary:active,
        .btn-secondary:focus,
        .btn-secondary.active {
            background: #6c757d !important;
            color: white !important;
            border-color: #6c757d !important;
            box-shadow: 0 0 0 3px rgba(108, 117, 125, 0.25) !important;
        }
        
        button:focus {
            outline: none !important;
        }
        
        /* Image Upload Styles */
        .image-upload-container {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            transition: border-color 0.3s ease;
            position: relative;
        }
        
        .image-upload-container:hover {
            border-color: var(--primary-color);
        }
        
        .image-upload-container input[type="file"] {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            opacity: 0;
            cursor: pointer;
        }
        
        .upload-placeholder {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 10px;
            color: #6c757d;
        }
        
        .upload-placeholder i {
            font-size: 48px;
            color: var(--primary-color);
        }
        
        .upload-placeholder span {
            font-weight: 500;
            color: var(--text-color);
        }
        
        .upload-placeholder small {
            font-size: 12px;
            color: #6c757d;
        }
        
        .image-preview {
            position: relative;
            display: inline-block;
        }
        
        .image-preview img {
            max-width: 200px;
            max-height: 200px;
            border-radius: 8px;
            border: 2px solid #dee2e6;
        }
        
        .remove-image {
            position: absolute;
            top: -10px;
            right: -10px;
            background: #dc3545;
            color: white;
            border: none;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            font-size: 16px;
            line-height: 1;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .remove-image:hover {
            background: #c82333;
        }
        
        .product-image {
            width: 60px;
            height: 60px;
            max-width: 60px;
            max-height: 60px;
            display: block;
            border-radius: 8px;
            object-fit: cover;
            border: 2px solid #dee2e6;
        }
        
        .no-image {
            width: 60px;
            height: 60px;
            background: #f8f9fa;
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #6c757d;
            font-size: 24px;
        }
        
        @media (max-width: 768px) {
            .image-upload-container {
                padding: 15px;
            }
            
            .upload-placeholder i {
                font-size: 36px;
            }
            
            .image-preview img {
                max-width: 150px;
                max-height: 150px;
            }
        }
    </style>
</body>
</html>