<?php
session_start();

// ตรวจสอบการล็อกอิน
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

// เชื่อมต่อฐานข้อมูล
require_once 'includes/db_con.php';

// ข้อมูลผู้ใช้
$username = isset($_SESSION['username']) ? $_SESSION['username'] : 'ผู้ใช้';

// จัดการการส่งฟอร์ม
$message = '';
$message_type = '';

// จัดการการลบข้อมูล
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'delete') {
    $currency_id = intval($_POST['currency_id']);
    
    if (empty($currency_id)) {
        echo json_encode(['success' => false, 'message' => 'ບໍ່ພົບລະຫັດສະກຸນເງິນ']);
        exit();
    }
    
    try {
        // ตรวจสอบว่ามีการใช้งานสกุลเงินนี้ในระบบหรือไม่
        $check_usage_stmt = $conn->prepare("SELECT COUNT(*) as count FROM orders WHERE currencys_id = (SELECT currencys_id FROM currencys WHERE id = ?)");
        $check_usage_stmt->bind_param("i", $currency_id);
        $check_usage_stmt->execute();
        $usage_result = $check_usage_stmt->get_result();
        $usage_count = $usage_result->fetch_assoc()['count'];
        $check_usage_stmt->close();
        
        if ($usage_count > 0) {
            echo json_encode(['success' => false, 'message' => 'ບໍ່ສາມາດລຶບໄດ້ເພາະມີການໃຊ້ງານສະກຸນເງິນນີ້ໃນລະບົບ']);
            exit();
        }
        
        // ลบสกุลเงิน
        $delete_stmt = $conn->prepare("DELETE FROM currencys WHERE id = ?");
        $delete_stmt->bind_param("i", $currency_id);
        
        if ($delete_stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'ລຶບສະກຸນເງິນສຳເລັດແລ້ວ']);
        } else {
            echo json_encode(['success' => false, 'message' => 'ເກີດຂໍ້ຜິດພາດໃນການລຶບສະກຸນເງິນ']);
        }
        $delete_stmt->close();
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'ເກີດຂໍ້ຜິດພາດ: ' . $e->getMessage()]);
    }
    exit();
}

// จัดการการแก้ไขข้อมูล
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'edit') {
    $currency_id = trim($_POST['currency_id']);
    $currencys_name = trim($_POST['currencys_name']);
    $currencys_fullname = trim($_POST['currencys_fullname']);
    $currencys_symbol = trim($_POST['currencys_symbol']);
    
    if (empty($currency_id) || empty($currencys_name) || empty($currencys_symbol)) {
        $message = 'ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ';
        $message_type = 'error';
    } else {
        try {
            // ตรวจสอบว่ามีสกุลเงินอื่นที่มีชื่อหรือสัญลักษณ์เดียวกันหรือไม่ (ยกเว้นตัวเอง)
            $check_stmt = $conn->prepare("SELECT id FROM currencys WHERE (currencys_name = ? OR currencys_symbol = ?) AND id != ?");
            $check_stmt->bind_param("ssi", $currencys_name, $currencys_symbol, $currency_id);
            $check_stmt->execute();
            
            if ($check_stmt->get_result()->num_rows > 0) {
                $message = 'ມີສະກຸນເງິນນີ້ໃນລະບົບແລ້ວ';
                $message_type = 'error';
            } else {
                // อัปเดตข้อมูลสกุลเงิน
                $update_stmt = $conn->prepare("UPDATE currencys SET currencys_name = ?, currencys_fullname = ?, currencys_symbol = ? WHERE id = ?");
                $update_stmt->bind_param("sssi", $currencys_name, $currencys_fullname, $currencys_symbol, $currency_id);
                
                if ($update_stmt->execute()) {
                    $message = 'ແກ້ໄຂສະກຸນເງິນສຳເລັດແລ້ວ';
                    $message_type = 'success';
                } else {
                    $message = 'ເກີດຂໍ້ຜິດພາດໃນການແກ້ໄຂສະກຸນເງິນ: ' . $conn->error;
                    $message_type = 'error';
                }
                $update_stmt->close();
            }
            $check_stmt->close();
        } catch (Exception $e) {
            $message = 'ເກີດຂໍ້ຜິດພາດ: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
} else if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $currencys_name = trim($_POST['currencys_name']);
    $currencys_fullname = trim($_POST['currencys_fullname']);
    $currencys_symbol = trim($_POST['currencys_symbol']);
    $base_currency = 'NO'; // ตั้งเป็น NO เสมอ
    
    if (empty($currencys_name) || empty($currencys_symbol)) {
        $message = 'ກະລຸນາໃສ່ຂໍ້ມູນໃຫ້ຄົບຖ້ວນ';
        $message_type = 'error';
    } else {
        try {
            // ตรวจสอบว่ามีสกุลเงินนี้แล้วหรือไม่
            $check_stmt = $conn->prepare("SELECT id FROM currencys WHERE currencys_name = ? OR currencys_symbol = ?");
            $check_stmt->bind_param("ss", $currencys_name, $currencys_symbol);
            $check_stmt->execute();
            
            if ($check_stmt->get_result()->num_rows > 0) {
                $message = 'ມີສະກຸນເງິນນີ້ໃນລະບົບແລ້ວ';
                $message_type = 'error';
            } else {
                // เพิ่มสกุลเงินใหม่ (currencys_id จะถูก auto-generate โดย trigger)
                $insert_stmt = $conn->prepare("INSERT INTO currencys (currencys_name, currencys_fullname, currencys_symbol, base_currency) VALUES (?, ?, ?, ?)");
                $insert_stmt->bind_param("ssss", $currencys_name, $currencys_fullname, $currencys_symbol, $base_currency);
                
                if ($insert_stmt->execute()) {
                    $message = 'ເພີ່ມສະກຸນເງິນສຳເລັດແລ້ວ';
                    $message_type = 'success';
                    
                    // ล้างข้อมูลในฟอร์ม
                    $_POST = [];
                } else {
                    $message = 'ເກີດຂໍ້ຜິດພາດໃນການເພີ່ມສະກຸນເງິນ: ' . $conn->error;
                    $message_type = 'error';
                }
                $insert_stmt->close();
            }
            $check_stmt->close();
        } catch (Exception $e) {
            $message = 'ເກີດຂໍ້ຜິດພາດ: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// ดึงข้อมูลสกุลเงินทั้งหมด
$currencys_query = "SELECT * FROM currencys ORDER BY base_currency DESC, currencys_name";
$currencys_result = $conn->query($currencys_query);

// Debug: ตรวจสอบว่ามีข้อมูลหรือไม่
if (!$currencys_result) {
    error_log("Currency query failed: " . $conn->error);
}
?>

<!DOCTYPE html>
<html lang="lo">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sokxay Bill - ເພີ່ມສະກຸນເງິນ</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="/sokxay/wayha.png">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Custom CSS -->
    <link rel="stylesheet" href="css/home.css">
    <link rel="stylesheet" href="css/forms.css">
    
    <style>
        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            overflow: auto;
        }
        
        .modal-content {
            background-color: #fff;
            margin: 5% auto;
            padding: 0;
            border-radius: 8px;
            width: 90%;
            max-width: 600px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from { opacity: 0; transform: translateY(-50px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .modal-header {
            background: var(--primary-color);
            color: white;
            padding: 20px;
            border-radius: 8px 8px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h2 {
            margin: 0;
            font-size: 20px;
        }
        
        .close {
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            opacity: 0.7;
            transition: opacity 0.3s;
        }
        
        .close:hover {
            opacity: 1;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .modal-footer {
            background: #f8f9fa;
            padding: 15px 20px;
            border-radius: 0 0 8px 8px;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        /* Button States Fix */
        .btn-primary:active,
        .btn-primary:focus,
        .btn-primary.active {
            background: var(--primary-color) !important;
            color: white !important;
            border-color: var(--primary-color) !important;
            box-shadow: 0 0 0 3px rgba(13, 110, 253, 0.25) !important;
        }
        
        .btn-secondary:active,
        .btn-secondary:focus,
        .btn-secondary.active {
            background: #6c757d !important;
            color: white !important;
            border-color: #6c757d !important;
            box-shadow: 0 0 0 3px rgba(108, 117, 125, 0.25) !important;
        }
        
        button:focus {
            outline: none !important;
        }
        
        /* Currency Display Styles */
        .currency-symbol {
            position: static !important;
            background: #e8f4fd;
            color: #0056b3 !important;
            padding: 4px 8px;
            border-radius: 4px;
            font-family: monospace;
            font-weight: 600;
            font-size: 16px;
            display: inline-block;
            transform: none !important;
            left: auto !important;
            top: auto !important;
            z-index: auto !important;
        }
        
        .currency-description {
            color: #6c757d;
            font-style: italic;
        }
    </style>
</head>
<body>
    <div class="app-container">
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content Area -->
        <main class="main-content">
            <!-- Header -->
            <header class="content-header">
                <div class="header-left">
                    <h1 class="page-title">ສະກຸນເງິນ</h1>
                    <p class="page-subtitle">ຈັດການສະກຸນເງິນໃນລະບົບ</p>
                </div>
                
                <div class="header-right">
                    <button class="btn-primary" onclick="openAddModal()">
                        <i class="fas fa-plus"></i>
                        ເພີ່ມສະກຸນເງິນ
                    </button>
                </div>
            </header>

            <!-- Content Body -->
            <div class="content-body">
                <div class="list-container">
                    <!-- Currency List -->
                    <div class="list-card">
                        <div class="list-header">
                            <h3><i class="fas fa-list"></i> ລາຍການສະກຸນເງິນ</h3>
                        </div>
                        
                        <div class="table-container">
                            <?php if ($currencys_result && $currencys_result->num_rows > 0): ?>
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th>ລະຫັດ</th>
                                            <th>ຊື່ສະກຸນເງິນ</th>
                                            <th>ຊື່ເຕັມ</th>
                                            <th>ສັນຍາລັກ</th>
                                            <th>ການກະທຳ</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($currency = $currencys_result->fetch_assoc()): ?>
                                            <tr>
                                                <td>
                                                    <span class="currency-code"><?php echo htmlspecialchars($currency['currencys_id'] ?? 'N/A'); ?></span>
                                                </td>
                                                <td><?php echo htmlspecialchars($currency['currencys_name'] ?? 'N/A'); ?></td>
                                                <td>
                                                    <span class="currency-description"><?php echo htmlspecialchars($currency['currencys_fullname'] ?? '-'); ?></span>
                                                </td>
                                                <td>
                                                    <span class="currency-symbol"><?php 
                                                        $symbol = $currency['currencys_symbol'] ?? '-';
                                                        echo $symbol !== '' ? htmlspecialchars($symbol) : '-';
                                                    ?></span>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <button class="btn-action btn-edit" onclick="editCurrency(<?php echo $currency['id']; ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn-action btn-delete" onclick="deleteCurrency(<?php echo $currency['id']; ?>)">
                                                               <i class="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>  
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <div class="no-data">
                                    <i class="fas fa-coins"></i>
                                    <p>ຍັງບໍ່ມີສະກຸນເງິນໃນລະບົບ</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Add Currency Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-coins"></i> ເພີ່ມສະກຸນເງິນໃໝ່</h2>
                <span class="close" onclick="closeAddModal()">&times;</span>
            </div>
            
            <div class="modal-body">
                <form id="addForm" method="POST" class="data-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="modal_currencys_name">
                                <i class="fas fa-tag"></i>
                                ຊື່ສະກຸນເງິນ <span class="required">*</span>
                            </label>
                            <select id="modal_currencys_name" 
                                    name="currencys_name" 
                                    onchange="updateCurrencyInfo()" 
                                    required>
                                <option value="">-- ເລືອກສະກຸນເງິນ --</option>
                                <option value="US Dollar" data-symbol="$" data-desc="ໂດລາສະຫະລັດ">US Dollar (ໂດລາສະຫະລັດ)</option>
                                <option value="Thai Baht" data-symbol="฿" data-desc="ບາດໄທ">Thai Baht (ບາດໄທ)</option>
                                <option value="Lao Kip" data-symbol="₭" data-desc="ກີບລາວ">Lao Kip (ກີບລາວ)</option>
                                <option value="Euro" data-symbol="€" data-desc="ເອີໂຣ">Euro (ເອີໂຣ)</option>
                                <option value="Japanese Yen" data-symbol="¥" data-desc="ເຢນຍີ່ປຸ່ນ">Japanese Yen (ເຢນຍີ່ປຸ່ນ)</option>
                                <option value="British Pound" data-symbol="£" data-desc="ປອນອັງກິດ">British Pound (ປອນອັງກິດ)</option>
                                <option value="Chinese Yuan" data-symbol="¥" data-desc="ຫຍວນຈີນ">Chinese Yuan (ຫຍວນຈີນ)</option>
                                <option value="Singapore Dollar" data-symbol="S$" data-desc="ໂດລາສິງກະໂປ">Singapore Dollar (ໂດລາສິງກະໂປ)</option>
                                <option value="Australian Dollar" data-symbol="A$" data-desc="ໂດລາອອສເຕຣເລຍ">Australian Dollar (ໂດລາອອສເຕຣເລຍ)</option>
                                <option value="Canadian Dollar" data-symbol="C$" data-desc="ໂດລາການາດາ">Canadian Dollar (ໂດລາການາດາ)</option>
                                <option value="Korean Won" data-symbol="₩" data-desc="ວອນເກົາຫຼີ">Korean Won (ວອນເກົາຫຼີ)</option>
                                <option value="Hong Kong Dollar" data-symbol="HK$" data-desc="ໂດລາຮ່ອງກົງ">Hong Kong Dollar (ໂດລາຮ່ອງກົງ)</option>
                                <option value="Malaysian Ringgit" data-symbol="RM" data-desc="ຣິງກິດມາເລເຊຍ">Malaysian Ringgit (ຣິງກິດມາເລເຊຍ)</option>
                                <option value="Vietnamese Dong" data-symbol="₫" data-desc="ດ່ອງຫວຽດນາມ">Vietnamese Dong (ດ່ອງຫວຽດນາມ)</option>
                                <option value="Philippine Peso" data-symbol="₱" data-desc="ເປໂຊຟິລິບປິນ">Philippine Peso (ເປໂຊຟິລິບປິນ)</option>
                                <option value="Indonesian Rupiah" data-symbol="Rp" data-desc="ຣູປີອິນໂດເນເຊຍ">Indonesian Rupiah (ຣູປີອິນໂດເນເຊຍ)</option>
                                <option value="Indian Rupee" data-symbol="₹" data-desc="ຣູປີອິນເດຍ">Indian Rupee (ຣູປີອິນເດຍ)</option>
                                <option value="Swiss Franc" data-symbol="CHF" data-desc="ຟຣັງສະວິດ">Swiss Franc (ຟຣັງສະວິດ)</option>
                                <option value="New Zealand Dollar" data-symbol="NZ$" data-desc="ໂດລານິວຊີແລນ">New Zealand Dollar (ໂດລານິວຊີແລນ)</option>
                                <option value="Norwegian Krone" data-symbol="kr" data-desc="ໂຄຣນນໍເວ">Norwegian Krone (ໂຄຣນນໍເວ)</option>
                                <option value="Swedish Krona" data-symbol="kr" data-desc="ໂຄຣນສະວີເດັນ">Swedish Krona (ໂຄຣນສະວີເດັນ)</option>
                                <option value="Danish Krone" data-symbol="kr" data-desc="ໂຄຣນເດນມາກ">Danish Krone (ໂຄຣນເດນມາກ)</option>
                                <option value="Russian Ruble" data-symbol="₽" data-desc="ຣູເບິນລັດເຊຍ">Russian Ruble (ຣູເບິນລັດເຊຍ)</option>
                                <option value="Brazilian Real" data-symbol="R$" data-desc="ເຣອລບຣາຊິນ">Brazilian Real (ເຣອລບຣາຊິນ)</option>
                                <option value="Mexican Peso" data-symbol="$" data-desc="ເປໂຊເມັກຊິໂກ">Mexican Peso (ເປໂຊເມັກຊິໂກ)</option>
                                <option value="South African Rand" data-symbol="R" data-desc="ແຣນອາຟຣິກາໃຕ້">South African Rand (ແຣນອາຟຣິກາໃຕ້)</option>
                                <option value="Turkish Lira" data-symbol="₺" data-desc="ລີຣາເທີກີ">Turkish Lira (ລີຣາເທີກີ)</option>
                                <option value="Saudi Riyal" data-symbol="﷼" data-desc="ຣິຍານຊາອຸດິອາເຣເບຍ">Saudi Riyal (ຣິຍານຊາອຸດິອາເຣເບຍ)</option>
                                <option value="UAE Dirham" data-symbol="د.إ" data-desc="ດິຮາມສາຢຸດ">UAE Dirham (ດິຮາມສາຢຸດ)</option>
                                <option value="Kuwaiti Dinar" data-symbol="د.ك" data-desc="ດີນາຄູເວດ">Kuwaiti Dinar (ດີນາຄູເວດ)</option>
                                <option value="Qatari Riyal" data-symbol="﷼" data-desc="ຣິຍານກາຕາ">Qatari Riyal (ຣິຍານກາຕາ)</option>
                            </select>
                            <small>ເລືອກສະກຸນເງິນທີ່ຕ້ອງການເພີ່ມໃສ່ລະບົບ</small>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="modal_currencys_fullname">
                                <i class="fas fa-info-circle"></i>
                                ຊື່ເຕັມສະກຸນເງິນ
                            </label>
                            <input type="text" 
                                   id="modal_currencys_fullname" 
                                   name="currencys_fullname" 
                                   placeholder="ເຊັ່ນ: ໂດລາສະຫະລັດອາເມລິກາ, ບາດໄທ">
                            <small>ຊື່ເຕັມຂອງສະກຸນເງິນ</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="modal_currencys_symbol">
                                <i class="fas fa-dollar-sign"></i>
                                ສັນຍາລັກສະກຸນເງິນ <span class="required">*</span>
                            </label>
                            <input type="text" 
                                   id="modal_currencys_symbol" 
                                   name="currencys_symbol" 
                                   placeholder="ເຊັ່ນ: $, ฿, €, ¥"
                                   maxlength="10"
                                   required>
                            <small>ສັນຍາລັກທີ່ໃຊ້ແສດງກັບຕົວເລກ ເຊັ່ນ: 2,000 $ ຫຼື 2,000 ฿</small>
                        </div>
                    </div>
                    

                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeAddModal()">
                    <i class="fas fa-times"></i>
                    ຍົກເລີກ
                </button>
                <button type="button" class="btn-primary" onclick="submitForm()">
                    <i class="fas fa-save"></i>
                    ບັນທຶກສະກຸນເງິນ
                </button>
            </div>
        </div>
    </div>

    <!-- Edit Currency Modal -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-edit"></i> ແກ້ໄຂສະກຸນເງິນ</h2>
                <span class="close" onclick="closeEditModal()">&times;</span>
            </div>
            
            <div class="modal-body">
                <form id="editForm" method="POST" class="data-form">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" id="edit_currency_id" name="currency_id">
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_currencys_name">
                                <i class="fas fa-tag"></i>
                                ຊື່ສະກຸນເງິນ <span class="required">*</span>
                            </label>
                            <select id="edit_currencys_name" 
                                    name="currencys_name" 
                                    onchange="updateEditCurrencyInfo()" 
                                    required>
                                <option value="">-- ເລືອກສະກຸນເງິນ --</option>
                                <option value="US Dollar" data-symbol="$" data-desc="ໂດລາສະຫະລັດ">US Dollar (ໂດລາສະຫະລັດ)</option>
                                <option value="Thai Baht" data-symbol="฿" data-desc="ບາດໄທ">Thai Baht (ບາດໄທ)</option>
                                <option value="Lao Kip" data-symbol="₭" data-desc="ກີບລາວ">Lao Kip (ກີບລາວ)</option>
                                <option value="Euro" data-symbol="€" data-desc="ເອີໂຣ">Euro (ເອີໂຣ)</option>
                                <option value="Japanese Yen" data-symbol="¥" data-desc="ເຢນຍີ່ປຸ່ນ">Japanese Yen (ເຢນຍີ່ປຸ່ນ)</option>
                                <option value="British Pound" data-symbol="£" data-desc="ປອນອັງກິດ">British Pound (ປອນອັງກິດ)</option>
                                <option value="Chinese Yuan" data-symbol="¥" data-desc="ຫຍວນຈີນ">Chinese Yuan (ຫຍວນຈີນ)</option>
                                <option value="Singapore Dollar" data-symbol="S$" data-desc="ໂດລາສິງກະໂປ">Singapore Dollar (ໂດລາສິງກະໂປ)</option>
                                <option value="Australian Dollar" data-symbol="A$" data-desc="ໂດລາອອສເຕຣເລຍ">Australian Dollar (ໂດລາອອສເຕຣເລຍ)</option>
                                <option value="Canadian Dollar" data-symbol="C$" data-desc="ໂດລາການາດາ">Canadian Dollar (ໂດລາການາດາ)</option>
                                <option value="Korean Won" data-symbol="₩" data-desc="ວອນເກົາຫຼີ">Korean Won (ວອນເກົາຫຼີ)</option>
                                <option value="Hong Kong Dollar" data-symbol="HK$" data-desc="ໂດລາຮ່ອງກົງ">Hong Kong Dollar (ໂດລາຮ່ອງກົງ)</option>
                                <option value="Malaysian Ringgit" data-symbol="RM" data-desc="ຣິງກິດມາເລເຊຍ">Malaysian Ringgit (ຣິງກິດມາເລເຊຍ)</option>
                                <option value="Vietnamese Dong" data-symbol="₫" data-desc="ດ່ອງຫວຽດນາມ">Vietnamese Dong (ດ່ອງຫວຽດນາມ)</option>
                                <option value="Philippine Peso" data-symbol="₱" data-desc="ເປໂຊຟິລິບປິນ">Philippine Peso (ເປໂຊຟິລິບປິນ)</option>
                                <option value="Indonesian Rupiah" data-symbol="Rp" data-desc="ຣູປີອິນໂດເນເຊຍ">Indonesian Rupiah (ຣູປີອິນໂດເນເຊຍ)</option>
                                <option value="Indian Rupee" data-symbol="₹" data-desc="ຣູປີອິນເດຍ">Indian Rupee (ຣູປີອິນເດຍ)</option>
                                <option value="Swiss Franc" data-symbol="CHF" data-desc="ຟຣັງສະວິດ">Swiss Franc (ຟຣັງສະວິດ)</option>
                                <option value="New Zealand Dollar" data-symbol="NZ$" data-desc="ໂດລານິວຊີແລນ">New Zealand Dollar (ໂດລານິວຊີແລນ)</option>
                                <option value="Norwegian Krone" data-symbol="kr" data-desc="ໂຄຣນນໍເວ">Norwegian Krone (ໂຄຣນນໍເວ)</option>
                                <option value="Swedish Krona" data-symbol="kr" data-desc="ໂຄຣນສະວີເດັນ">Swedish Krona (ໂຄຣນສະວີເດັນ)</option>
                                <option value="Danish Krone" data-symbol="kr" data-desc="ໂຄຣນເດນມາກ">Danish Krone (ໂຄຣນເດນມາກ)</option>
                                <option value="Russian Ruble" data-symbol="₽" data-desc="ຣູເບິນລັດເຊຍ">Russian Ruble (ຣູເບິນລັດເຊຍ)</option>
                                <option value="Brazilian Real" data-symbol="R$" data-desc="ເຣອລບຣາຊິນ">Brazilian Real (ເຣອລບຣາຊິນ)</option>
                                <option value="Mexican Peso" data-symbol="$" data-desc="ເປໂຊເມັກຊິໂກ">Mexican Peso (ເປໂຊເມັກຊິໂກ)</option>
                                <option value="South African Rand" data-symbol="R" data-desc="ແຣນອາຟຣິກາໃຕ້">South African Rand (ແຣນອາຟຣິກາໃຕ້)</option>
                                <option value="Turkish Lira" data-symbol="₺" data-desc="ລີຣາເທີກີ">Turkish Lira (ລີຣາເທີກີ)</option>
                                <option value="Saudi Riyal" data-symbol="﷼" data-desc="ຣິຍານຊາອຸດິອາເຣເບຍ">Saudi Riyal (ຣິຍານຊາອຸດິອາເຣເບຍ)</option>
                                <option value="UAE Dirham" data-symbol="د.إ" data-desc="ດິຮາມສາຢຸດ">UAE Dirham (ດິຮາມສາຢຸດ)</option>
                                <option value="Kuwaiti Dinar" data-symbol="د.ك" data-desc="ດີນາຄູເວດ">Kuwaiti Dinar (ດີນາຄູເວດ)</option>
                                <option value="Qatari Riyal" data-symbol="﷼" data-desc="ຣິຍານກາຕາ">Qatari Riyal (ຣິຍານກາຕາ)</option>
                            </select>
                            <small>ເລືອກສະກຸນເງິນທີ່ຕ້ອງການແກ້ໄຂ</small>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_currencys_fullname">
                                <i class="fas fa-info-circle"></i>
                                ຊື່ເຕັມສະກຸນເງິນ
                            </label>
                            <input type="text" 
                                   id="edit_currencys_fullname" 
                                   name="currencys_fullname" 
                                   placeholder="ເຊັ່ນ: ໂດລາສະຫະລັດອາເມລິກາ, ບາດໄທ">
                            <small>ຊື່ເຕັມຂອງສະກຸນເງິນ</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="edit_currencys_symbol">
                                <i class="fas fa-dollar-sign"></i>
                                ສັນຍາລັກສະກຸນເງິນ <span class="required">*</span>
                            </label>
                            <input type="text" 
                                   id="edit_currencys_symbol" 
                                   name="currencys_symbol" 
                                   placeholder="ເຊັ່ນ: $, ฿, €, ¥"
                                   maxlength="10"
                                   required>
                            <small>ສັນຍາລັກທີ່ໃຊ້ແສດງກັບຕົວເລກ ເຊັ່ນ: 2,000 $ ຫຼື 2,000 ฿</small>
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeEditModal()">
                    <i class="fas fa-times"></i>
                    ຍົກເລີກ
                </button>
                <button type="button" class="btn-primary" onclick="submitEditForm()">
                    <i class="fas fa-save"></i>
                    ບັນທຶກການແກ້ໄຂ
                </button>
            </div>
        </div>
    </div>

    <!-- Sweet Alert -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <!-- Sidebar JavaScript -->
    <script src="js/sidebar.js"></script>
    
    <script>
        // Modal Functions
        function openAddModal() {
            document.getElementById('addModal').style.display = 'block';
            document.getElementById('addForm').reset();
        }
        
        function closeAddModal() {
            document.getElementById('addModal').style.display = 'none';
        }
        
        function submitForm() {
            const form = document.getElementById('addForm');
            const formData = new FormData(form);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(html => {
                if (html.includes('ເພີ່ມສະກຸນເງິນສຳເລັດແລ້ວ')) {
                    Swal.fire({
                        icon: 'success',
                        title: 'ສຳເລັດ!',
                        text: 'ເພີ່ມສະກຸນເງິນສຳເລັດແລ້ວ',
                        timer: 2000,
                        showConfirmButton: false
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    // Extract error message
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = html;
                    const alertDiv = tempDiv.querySelector('.alert-error');
                    const errorMessage = alertDiv ? alertDiv.textContent.trim() : 'ເກີດຂໍ້ຜິດພາດ';
                    
                    Swal.fire({
                        icon: 'error',
                        title: 'ຜິດພາດ',
                        text: errorMessage
                    });
                }
            })
            .catch(error => {
                Swal.fire({
                    icon: 'error',
                    title: 'ຜິດພາດ',
                    text: 'ເກີດຂໍ້ຜິດພາດໃນການສົ່ງຂໍ້ມູນ'
                });
            });
        }
        
        // Auto update symbol and fullname when currency is selected
        function updateCurrencyInfo() {
            const select = document.getElementById('modal_currencys_name');
            const selectedOption = select.options[select.selectedIndex];
            
            if (selectedOption && selectedOption.value) {
                const symbol = selectedOption.getAttribute('data-symbol') || '';
                const fullname = selectedOption.getAttribute('data-desc') || '';
                
                document.getElementById('modal_currencys_symbol').value = symbol;
                document.getElementById('modal_currencys_fullname').value = fullname;
            }
        }

        // Close modal when clicking outside
        window.addEventListener('click', function(event) {
            const addModal = document.getElementById('addModal');
            const editModal = document.getElementById('editModal');
            
            if (event.target === addModal) {
                closeAddModal();
            }
            
            if (event.target === editModal) {
                closeEditModal();
            }
        });

        // Edit Currency Function
        function editCurrency(id) {
            // ดึงข้อมูลสกุลเงินจากฐานข้อมูล
            fetch('get_currency_data.php?id=' + id)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // เติมข้อมูลในฟอร์ม
                        document.getElementById('edit_currency_id').value = data.currency.id;
                        document.getElementById('edit_currencys_name').value = data.currency.currencys_name;
                        document.getElementById('edit_currencys_fullname').value = data.currency.currencys_fullname || '';
                        document.getElementById('edit_currencys_symbol').value = data.currency.currencys_symbol;
                        
                        // เปิด modal
                        document.getElementById('editModal').style.display = 'block';
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'ຜິດພາດ',
                            text: 'ບໍ່ສາມາດດຶງຂໍ້ມູນສະກຸນເງິນໄດ້'
                        });
                    }
                })
                .catch(error => {
                    // Fallback: ใช้ข้อมูลจากตารางที่แสดงอยู่
                    const row = document.querySelector(`button[onclick="editCurrency(${id})"]`).closest('tr');
                    const cells = row.querySelectorAll('td');
                    
                    document.getElementById('edit_currency_id').value = id;
                    document.getElementById('edit_currencys_name').value = cells[1].textContent.trim();
                    document.getElementById('edit_currencys_fullname').value = cells[2].textContent.trim();
                    document.getElementById('edit_currencys_symbol').value = cells[3].querySelector('.currency-symbol').textContent.trim();
                    
                    // เปิด modal
                    document.getElementById('editModal').style.display = 'block';
                });
        }
        
        // Close Edit Modal
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        // Submit Edit Form
        function submitEditForm() {
            const form = document.getElementById('editForm');
            const formData = new FormData(form);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(html => {
                if (html.includes('ແກ້ໄຂສະກຸນເງິນສຳເລັດແລ້ວ')) {
                    Swal.fire({
                        icon: 'success',
                        title: 'ສຳເລັດ!',
                        text: 'ແກ້ໄຂສະກຸນເງິນສຳເລັດແລ້ວ',
                        timer: 2000,
                        showConfirmButton: false
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    // Extract error message
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = html;
                    const alertDiv = tempDiv.querySelector('.alert-error');
                    const errorMessage = alertDiv ? alertDiv.textContent.trim() : 'ເກີດຂໍ້ຜິດພາດ';
                    
                    Swal.fire({
                        icon: 'error',
                        title: 'ຜິດພາດ',
                        text: errorMessage
                    });
                }
            })
            .catch(error => {
                Swal.fire({
                    icon: 'error',
                    title: 'ຜິດພາດ',
                    text: 'ເກີດຂໍ້ຜິດພາດໃນການສົ່ງຂໍ້ມູນ'
                });
            });
        }
        
        // Auto update symbol and fullname when currency is selected in edit form
        function updateEditCurrencyInfo() {
            const select = document.getElementById('edit_currencys_name');
            const selectedOption = select.options[select.selectedIndex];
            
            if (selectedOption && selectedOption.value) {
                const symbol = selectedOption.getAttribute('data-symbol') || '';
                const fullname = selectedOption.getAttribute('data-desc') || '';
                
                document.getElementById('edit_currencys_symbol').value = symbol;
                document.getElementById('edit_currencys_fullname').value = fullname;
            }
        }
        
        // Delete Currency Function
        function deleteCurrency(id) {
            Swal.fire({
                title: 'ຢືນຢັນການລຶບ',
                text: "ຕ້ອງການລຶບສະກຸນເງິນນີ້ບໍ່?",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'ລຶບ',
                cancelButtonText: 'ຍົກເລີກ'
            }).then((result) => {
                if (result.isConfirmed) {
                    // ส่งคำขอลบไปยัง server
                    const formData = new FormData();
                    formData.append('action', 'delete');
                    formData.append('currency_id', id);
                    
                    fetch(window.location.href, {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'ສຳເລັດ!',
                                text: data.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'ຜິດພາດ',
                                text: data.message
                            });
                        }
                    })
                    .catch(error => {
                        Swal.fire({
                            icon: 'error',
                            title: 'ຜິດພາດ',
                            text: 'ເກີດຂໍ້ຜິດພາດໃນການສົ່ງຂໍ້ມູນ'
                        });
                    });
                }
            });
        }

        // Show success message if exists
        <?php if ($message_type === 'success'): ?>
            Swal.fire({
                icon: 'success',
                title: 'ສຳເລັດ!',
                text: '<?php echo $message; ?>',
                timer: 3000,
                showConfirmButton: false
            });
        <?php elseif ($message_type === 'error'): ?>
            Swal.fire({
                icon: 'error',
                title: 'ຜິດພາດ',
                text: '<?php echo $message; ?>'
            });
        <?php endif; ?>
    </script>
</body>
</html>