<?php
session_start();
require_once '../includes/con_database.php';

header('Content-Type: application/json');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);

try {
    if (!isset($_SESSION['shop_id'])) {
        echo json_encode(['error' => 'No shop selected', 'transactions' => [], 'totals' => ['total_income' => 0, 'total_expense' => 0, 'total_cost' => 0, 'total_profit' => 0]]);
        exit;
    }

    $shop_id = $_SESSION['shop_id'];
    $search_term = trim($_POST['search_term'] ?? '');
    $start_date = $_POST['start_date'] ?? '';
    $end_date = $_POST['end_date'] ?? '';
    $offset = intval($_POST['offset'] ?? 0);
    $limit = intval($_POST['limit'] ?? 100);

    if ($limit < 1 || $limit > 500)
        $limit = 100;
    if ($offset < 0)
        $offset = 0;

    // Default date range if not specified
    if (empty($start_date))
        $start_date = '2000-01-01';
    else
        $start_date = date('Y-m-d', strtotime($start_date));

    if (empty($end_date))
        $end_date = '2099-12-31';
    else
        $end_date = date('Y-m-d', strtotime($end_date));

    // Get all transactions (without pagination limit for totals calculation)
    $all_transactions = getTransactions($conn, $shop_id, $search_term, $start_date, $end_date, 0, 10000);

    // Calculate totals from the transaction data that's displayed
    $totals = calculateTotalsFromTransactions($all_transactions);

    // Apply pagination for display
    $transactions = array_slice($all_transactions, $offset, $limit);

    echo json_encode(['transactions' => $transactions, 'totals' => $totals]);

} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage(), 'transactions' => [], 'totals' => ['total_income' => 0, 'total_expense' => 0, 'total_cost' => 0, 'total_profit' => 0]]);
}

// Simple function to calculate totals from transaction data
function calculateTotalsFromTransactions($transactions)
{
    $total_income = 0;
    $total_expense = 0;
    $total_cost = 0;

    foreach ($transactions as $t) {
        // Income = amount_in_lak (what was received in LAK)
        $total_income += floatval($t['amount_in_lak'] ?? 0);
        // Expense = expense_amount
        $total_expense += floatval($t['expense_amount'] ?? 0);
        // Cost = cost_amount
        $total_cost += floatval($t['cost_amount'] ?? 0);
    }

    // Net profit = Income - Cost - Expenses
    $total_profit = $total_income - $total_cost - $total_expense;

    return [
        'total_income' => $total_income,
        'total_expense' => $total_expense,
        'total_cost' => $total_cost,
        'total_profit' => $total_profit
    ];
}

// Helper function for multi-field search
function matchesSearch($search_term, $payment_status, $source_type, $details, $currency_name, $currency_rate, $or_id = '')
{
    if (empty($search_term))
        return true;

    $search_lower = mb_strtolower($search_term, 'UTF-8');

    // Status mapping for search (Lao language)
    $status_labels = [
        'paid' => 'ຊຳລະແລ້ວ',
        'partial' => 'ມັດຈຳ',
        'pending' => 'ລໍຖ້າ', // Used for both orders (ຍັງບໍ່ຈ່າຍ) and PO (ລໍຖ້າ)
        'deposit_paid' => 'ມັດຈຳ',
        'payment_received' => 'ຊຳລະເງິນ',
        'was_pending' => 'ຍັງບໍ່ຈ່າຍ',
        // Purchase order statuses
        'received' => 'ໄດ້ຮັບແລ້ວ',
        'ordered' => 'ສັ່ງແລ້ວ',
        'draft' => 'ຮ່າງ'
    ];

    $status_label = $status_labels[$payment_status] ?? $payment_status;

    // Format exchange rate for search
    $rate_text = $currency_rate > 1 ? "1 {$currency_name} = " . number_format($currency_rate, 0) . " ₭" : '';

    // Search in all fields
    $searchable_fields = [
        mb_strtolower($status_label, 'UTF-8'),           // ສະຖານະ
        mb_strtolower($source_type, 'UTF-8'),            // ແຫຼ່ງທີ່ມາ
        mb_strtolower($details, 'UTF-8'),                // ລາຍລະອຽດ
        mb_strtolower($currency_name, 'UTF-8'),          // ສະກຸນເງິນ
        mb_strtolower($rate_text, 'UTF-8'),              // ອັດຕາແລກປ່ຽນ
        mb_strtolower($or_id, 'UTF-8')                   // OR_id
    ];

    foreach ($searchable_fields as $field) {
        if (mb_strpos($field, $search_lower, 0, 'UTF-8') !== false) {
            return true;
        }
    }

    return false;
}

function getTransactions($conn, $shop_id, $search_term, $start_date, $end_date, $offset, $limit)
{
    $all = [];
    $search_pattern = '%' . $search_term . '%';

    // ==================== ORDERS ====================
    // Simple query first - just get order data with cost calculated in PHP
    $sql = "SELECT 
                o.id,
                o.OR_id,
                o.created_at as transaction_date,
                o.payment_status,
                o.order_type,
                o.total_amount,
                o.tax_amount,
                o.deposit_amount,
                o.total_amount_lak,
                o.created_at,
                o.updated_at,
                o.currencys_id,
                o.currency_rate,
                COALESCE(c.currencys_symbol, '₭') as currency_symbol,
                COALESCE(c.currencys_name, 'LAK') as currency_name
            FROM orders o
            LEFT JOIN currencys c ON o.currencys_id = c.currencys_id
            WHERE o.shop_id = ?
              AND o.deleted_at IS NULL
              AND DATE(o.created_at) BETWEEN ? AND ?
            ORDER BY o.created_at ASC";

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        throw new Exception("Orders prepare failed: " . $conn->error);
    }
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        // Calculate cost from order_items
        $cost_sql = "SELECT COALESCE(SUM(oi.quantity * p.cost_price), 0) as total_cost 
                     FROM order_items oi 
                     JOIN products p ON oi.product_id = p.PR_id 
                     WHERE oi.order_id = ?";
        $cost_stmt = $conn->prepare($cost_sql);
        $cost_stmt->bind_param("i", $row['id']);
        $cost_stmt->execute();
        $cost_result = $cost_stmt->get_result();
        $cost_row = $cost_result->fetch_assoc();
        $total_cost = floatval($cost_row['total_cost'] ?? 0);

        // Determine source and amounts based on payment status
        $source = 'other';
        $amount = 0;
        $cost_amount = 0;

        // Check if this is a paid order that was previously partial (has deposit and was updated)
        $was_partial = $row['payment_status'] == 'paid'
            && floatval($row['deposit_amount']) > 0
            && $row['created_at'] != $row['updated_at'];

        // Check if this was a pending order that is now paid (no deposit, but was updated)
        // DISABLED: Treat as single paid transaction even if updated later (per user request)
        $was_pending = false;
        /* $row['payment_status'] == 'paid'
            && floatval($row['deposit_amount']) == 0
            && $row['created_at'] != $row['updated_at']; */

        if ($row['payment_status'] == 'paid') {
            if ($was_partial) {
                // This order was partial and now paid - we'll create 2 entries
                // Entry 1: Deposit amount (using created_at)
                $deposit_amount = floatval($row['deposit_amount']);

                // LAK calculations first
                $stored_total_lak = floatval($row['total_amount_lak'] ?? 0);
                $currency_rate = floatval($row['currency_rate'] ?? 1);
                $is_foreign_currency = ($row['currencys_id'] ?? 'CUR_1') !== 'CUR_1' && $currency_rate > 1;

                // Calculate full_amount with VAT
                // For foreign currency: use total_amount_lak / currency_rate (since tax_amount is stored in LAK)
                // For LAK: stored_total_lak already includes VAT
                if ($is_foreign_currency && $currency_rate > 0 && $stored_total_lak > 0) {
                    $full_amount = $stored_total_lak / $currency_rate;
                } else {
                    $full_amount = $stored_total_lak; // LAK order - total_amount_lak is the full amount
                }
                $remaining_amount = $full_amount - $deposit_amount;

                // Proportional cost for deposit
                $deposit_cost = 0;
                if ($full_amount > 0) {
                    $deposit_cost = ($deposit_amount / $full_amount) * $total_cost;
                }

                $deposit_lak = 0;
                $remaining_lak = 0;
                if ($stored_total_lak > 0 && $full_amount > 0) {
                    $deposit_lak = ($deposit_amount / $full_amount) * $stored_total_lak;
                    $remaining_lak = $stored_total_lak - $deposit_lak;
                } else {
                    $deposit_lak = $deposit_amount * $currency_rate;
                    $remaining_lak = $remaining_amount * $currency_rate;
                }

                // Remaining cost
                $remaining_cost = $total_cost - $deposit_cost;

                // Source type
                $source_type = 'POS';
                if ($row['order_type'] == 'online')
                    $source_type = 'Online';
                elseif ($row['order_type'] == 'wholesale')
                    $source_type = 'Wholesale';

                $details = $row['OR_id'];

                // Apply search filter (multi-field)
                if (!matchesSearch($search_term, 'deposit_paid', $source_type, $details, $row['currency_name'] ?? 'LAK', $currency_rate, $row['OR_id'])) {
                    continue;
                }

                // Entry 1: DEPOSIT (created_at date)
                $all[] = [
                    'id' => $row['id'],
                    'OR_id' => $row['OR_id'],
                    'transaction_date' => $row['created_at'],
                    'source' => 'deposit',
                    'source_type' => $source_type,
                    'details' => $details,
                    'amount' => $deposit_amount,
                    'amount_in_lak' => $deposit_lak,
                    'expense_amount' => 0,
                    'cost_amount' => $deposit_cost,
                    'payment_status' => 'deposit_paid', // special status for display
                    'full_amount' => $full_amount,
                    'deposit_amount' => $deposit_amount,
                    'total_amount_lak' => $stored_total_lak,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'currencys_id' => $row['currencys_id'] ?? 'CUR_1',
                    'currency_rate' => $currency_rate,
                    'currency_symbol' => $row['currency_symbol'] ?? '₭',
                    'currency_name' => $row['currency_name'] ?? 'LAK',
                    'is_foreign_currency' => $is_foreign_currency,
                    'entry_type' => 'deposit'
                ];

                // Entry 2: REMAINING PAYMENT (updated_at date)
                $all[] = [
                    'id' => $row['id'],
                    'OR_id' => $row['OR_id'],
                    'transaction_date' => $row['updated_at'],
                    'source' => 'payment_received',
                    'source_type' => $source_type,
                    'details' => $details . ' (ຊຳລະສ່ວນທີ່ເຫຼືອ)',
                    'amount' => $remaining_amount,
                    'amount_in_lak' => $remaining_lak,
                    'expense_amount' => 0,
                    'cost_amount' => $remaining_cost,
                    'payment_status' => 'payment_received', // special status for display
                    'full_amount' => $full_amount,
                    'deposit_amount' => $deposit_amount,
                    'total_amount_lak' => $stored_total_lak,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'currencys_id' => $row['currencys_id'] ?? 'CUR_1',
                    'currency_rate' => $currency_rate,
                    'currency_symbol' => $row['currency_symbol'] ?? '₭',
                    'currency_name' => $row['currency_name'] ?? 'LAK',
                    'is_foreign_currency' => $is_foreign_currency,
                    'entry_type' => 'payment'
                ];

                continue; // Skip the normal processing below
            } elseif ($was_pending) {
                // This order was pending and now paid - we'll create 2 entries
                // Entry 1: Pending row (using created_at) - shows as "ยังบ่ອย" with dashes
                // Entry 2: Payment received (using updated_at) - shows as "ชำระเงิน" with full amount

                // Calculate full_amount with VAT
                $stored_total_lak = floatval($row['total_amount_lak'] ?? 0);
                $currency_rate = floatval($row['currency_rate'] ?? 1);
                $is_foreign_currency = ($row['currencys_id'] ?? 'CUR_1') !== 'CUR_1' && $currency_rate > 1;

                // For foreign currency: use total_amount_lak / currency_rate (since tax_amount is stored in LAK)
                // For LAK: stored_total_lak already includes VAT
                if ($is_foreign_currency && $currency_rate > 0 && $stored_total_lak > 0) {
                    $full_amount = $stored_total_lak / $currency_rate;
                } else {
                    $full_amount = $stored_total_lak; // LAK order
                }

                // Source type
                $source_type = 'POS';
                if ($row['order_type'] == 'online')
                    $source_type = 'Online';
                elseif ($row['order_type'] == 'wholesale')
                    $source_type = 'Wholesale';

                $details = $row['OR_id'];

                // Apply search filter (multi-field)
                if (!matchesSearch($search_term, 'was_pending', $source_type, $details, $row['currency_name'] ?? 'LAK', $currency_rate, $row['OR_id'])) {
                    continue;
                }

                // Entry 1: PENDING row (created_at date)
                $all[] = [
                    'id' => $row['id'],
                    'OR_id' => $row['OR_id'],
                    'transaction_date' => $row['created_at'],
                    'source' => 'pending',
                    'source_type' => $source_type,
                    'details' => $details,
                    'amount' => 0,
                    'amount_in_lak' => 0,
                    'expense_amount' => 0,
                    'cost_amount' => 0,
                    'payment_status' => 'was_pending', // special status - was pending, now paid
                    'full_amount' => $full_amount,
                    'deposit_amount' => 0,
                    'total_amount_lak' => $stored_total_lak,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'currencys_id' => $row['currencys_id'] ?? 'CUR_1',
                    'currency_rate' => $currency_rate,
                    'currency_symbol' => $row['currency_symbol'] ?? '₭',
                    'currency_name' => $row['currency_name'] ?? 'LAK',
                    'is_foreign_currency' => $is_foreign_currency,
                    'entry_type' => 'pending_original'
                ];

                // Entry 2: FULL PAYMENT (updated_at date)
                $all[] = [
                    'id' => $row['id'],
                    'OR_id' => $row['OR_id'],
                    'transaction_date' => $row['updated_at'],
                    'source' => 'payment_received',
                    'source_type' => $source_type,
                    'details' => $details . ' (ຊຳລະເງິນ)',
                    'amount' => $full_amount,
                    'amount_in_lak' => $stored_total_lak,
                    'expense_amount' => 0,
                    'cost_amount' => $total_cost,
                    'payment_status' => 'payment_received', // special status for display
                    'full_amount' => $full_amount,
                    'deposit_amount' => 0,
                    'total_amount_lak' => $stored_total_lak,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'currencys_id' => $row['currencys_id'] ?? 'CUR_1',
                    'currency_rate' => $currency_rate,
                    'currency_symbol' => $row['currency_symbol'] ?? '₭',
                    'currency_name' => $row['currency_name'] ?? 'LAK',
                    'is_foreign_currency' => $is_foreign_currency,
                    'entry_type' => 'payment'
                ];

                continue; // Skip the normal processing below
            } else {
                // Normal paid order (paid immediately, no status change)
                $source = 'orders';
                // Calculate amount with VAT
                // For foreign currency: use total_amount_lak / currency_rate
                $stored_total_lak = floatval($row['total_amount_lak'] ?? 0);
                $curr_rate = floatval($row['currency_rate'] ?? 1);
                $is_fc = ($row['currencys_id'] ?? 'CUR_1') !== 'CUR_1' && $curr_rate > 1;

                if ($is_fc && $curr_rate > 0 && $stored_total_lak > 0) {
                    // Use stored total_amount if available to avoid rounding errors
                    if (!empty($row['total_amount']) && $row['total_amount'] > 0) {
                        $amount = floatval($row['total_amount']);
                    } else {
                        $amount = $stored_total_lak / $curr_rate;
                    }
                } else {
                    $amount = $stored_total_lak; // LAK order
                }
                $cost_amount = $total_cost;
            }
        } elseif ($row['payment_status'] == 'partial') {
            $source = 'deposit';
            $amount = floatval($row['deposit_amount']);
            // Proportional cost for deposit - use total_amount_lak for accurate ratio
            $partial_total_lak = floatval($row['total_amount_lak'] ?? 0);
            $partial_curr_rate = floatval($row['currency_rate'] ?? 1);
            $partial_is_fc = ($row['currencys_id'] ?? 'CUR_1') !== 'CUR_1' && $partial_curr_rate > 1;

            if ($partial_is_fc && $partial_curr_rate > 0 && $partial_total_lak > 0) {
                $full_with_tax = $partial_total_lak / $partial_curr_rate;
            } else {
                $full_with_tax = $partial_total_lak; // LAK order
            }

            if ($full_with_tax > 0) {
                $cost_amount = ($row['deposit_amount'] / $full_with_tax) * $total_cost;
            }
        } elseif ($row['payment_status'] == 'pending') {
            $source = 'pending';
            $amount = 0;
            $cost_amount = 0;
        }

        // Source type
        $source_type = 'POS';
        if ($row['order_type'] == 'online')
            $source_type = 'Online';
        elseif ($row['order_type'] == 'wholesale')
            $source_type = 'Wholesale';

        // Apply search filter (multi-field)
        $details = $row['OR_id'];
        $currency_rate = floatval($row['currency_rate'] ?? 1);
        if (!matchesSearch($search_term, $row['payment_status'], $source_type, $details, $row['currency_name'] ?? 'LAK', $currency_rate, $row['OR_id'])) {
            continue;
        }

        // Calculate LAK amount (for foreign currency orders)
        $currency_rate = floatval($row['currency_rate'] ?? 1);
        $is_foreign_currency = ($row['currencys_id'] ?? 'CUR_1') !== 'CUR_1' && $currency_rate > 1;

        // Use stored total_amount_lak from database to avoid rounding errors
        // total_amount_lak already includes VAT so use it directly
        // For paid orders: use total_amount_lak directly
        // For partial orders: calculate proportional LAK from stored total
        $stored_total_lak = floatval($row['total_amount_lak'] ?? 0);
        // Calculate full amount with tax for ratio calculations
        // For foreign currency: use total_amount_lak / currency_rate
        if ($is_foreign_currency && $currency_rate > 0 && $stored_total_lak > 0) {
            $full_with_tax = $stored_total_lak / $currency_rate;
        } else {
            $full_with_tax = $stored_total_lak; // LAK order
        }

        if ($row['payment_status'] == 'paid') {
            $amount_in_lak = $stored_total_lak;
        } elseif ($row['payment_status'] == 'partial') {
            // For partial, calculate proportion based on deposit vs total (with tax)
            if ($full_with_tax > 0 && $stored_total_lak > 0) {
                $amount_in_lak = ($amount / $full_with_tax) * $stored_total_lak;
            } else {
                $amount_in_lak = $amount * $currency_rate;
            }
        } else {
            $amount_in_lak = 0; // pending
        }

        $all[] = [
            'id' => $row['id'],
            'OR_id' => $row['OR_id'],
            'transaction_date' => $row['transaction_date'],
            'source' => $source,
            'source_type' => $source_type,
            'details' => $details,
            'amount' => $amount,
            'amount_in_lak' => $amount_in_lak,
            'expense_amount' => 0,
            'cost_amount' => $cost_amount,
            'payment_status' => $row['payment_status'],
            'full_amount' => $full_with_tax,
            'deposit_amount' => floatval($row['deposit_amount']),
            'total_amount_lak' => floatval($row['total_amount_lak'] ?? 0),
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at'],
            'currencys_id' => $row['currencys_id'] ?? 'CUR_1',
            'currency_rate' => $currency_rate,
            'currency_symbol' => $row['currency_symbol'] ?? '₭',
            'currency_name' => $row['currency_name'] ?? 'LAK',
            'is_foreign_currency' => $is_foreign_currency
        ];
    }

    // ==================== OTHER INCOME ====================
    $sql = "SELECT 
                id,
                IN_id,
                income_date as transaction_date,
                CONCAT(IN_id, ' - ', COALESCE(description, category, 'ລາຍຮັບອື່ນໆ')) as details,
                amount,
                created_at
            FROM other_income
            WHERE shop_id = ?
              AND deleted_at IS NULL
              AND DATE(income_date) BETWEEN ? AND ?
            ORDER BY income_date ASC";

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        throw new Exception("Other income prepare failed: " . $conn->error);
    }
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        // Apply search filter (multi-field)
        if (!matchesSearch($search_term, 'paid', 'ລາຍຮັບອື່ນໆ', $row['details'], 'LAK', 1, $row['IN_id'])) {
            continue;
        }

        $all[] = [
            'id' => $row['id'],
            'OR_id' => $row['IN_id'],
            'transaction_date' => $row['transaction_date'],
            'source' => 'other_income',
            'source_type' => 'ລາຍຮັບອື່ນໆ',
            'details' => $row['details'],
            'amount' => floatval($row['amount']),
            'amount_in_lak' => floatval($row['amount']),
            'expense_amount' => 0,
            'cost_amount' => 0,
            'payment_status' => 'paid',
            'full_amount' => floatval($row['amount']),
            'deposit_amount' => 0,
            'total_amount_lak' => floatval($row['amount']),
            'created_at' => $row['created_at'],
            'updated_at' => $row['created_at'],
            'currencys_id' => 'CUR_1',
            'currency_rate' => 1,
            'currency_symbol' => '₭',
            'currency_name' => 'LAK',
            'is_foreign_currency' => false
        ];
    }

    // ==================== EXPENSES ====================
    $sql = "SELECT 
                id,
                EX_id,
                expense_date as transaction_date,
                CONCAT(EX_id, ' - ', COALESCE(description, category, 'ລາຍຈ່າຍ')) as details,
                amount,
                created_at
            FROM expenses
            WHERE shop_id = ?
              AND deleted_at IS NULL
              AND DATE(expense_date) BETWEEN ? AND ?
            ORDER BY expense_date ASC";

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        throw new Exception("Expenses prepare failed: " . $conn->error);
    }
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        // Apply search filter (multi-field)
        if (!matchesSearch($search_term, 'paid', 'ລາຍຈ່າຍ', $row['details'], 'LAK', 1, $row['EX_id'])) {
            continue;
        }

        $all[] = [
            'id' => $row['id'],
            'OR_id' => $row['EX_id'],
            'transaction_date' => $row['transaction_date'],
            'source' => 'expenses',
            'source_type' => 'ລາຍຈ່າຍ',
            'details' => $row['details'],
            'amount' => 0,
            'amount_in_lak' => 0,
            'expense_amount' => floatval($row['amount']),
            'cost_amount' => 0,
            'payment_status' => 'paid',
            'full_amount' => floatval($row['amount']),
            'deposit_amount' => 0,
            'total_amount_lak' => 0,
            'created_at' => $row['created_at'],
            'updated_at' => $row['created_at'],
            'currencys_id' => 'CUR_1',
            'currency_rate' => 1,
            'currency_symbol' => '₭',
            'currency_name' => 'LAK',
            'is_foreign_currency' => false
        ];
    }

    // ==================== PURCHASE ORDERS ====================
    $sql = "SELECT 
                id,
                PO_id,
                order_date as transaction_date,
                status,
                CONCAT(PO_id, ' - ', COALESCE(supplier_name, 'ສັ່ງຊື້ສິນຄ້າ')) as details,
                total_amount,
                created_at
            FROM purchase_orders
            WHERE shop_id = ?
              AND deleted_at IS NULL
              AND status IN ('pending', 'received', 'partial', 'ordered')
              AND DATE(order_date) BETWEEN ? AND ?
            ORDER BY order_date ASC";

    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
        $stmt->execute();
        $result = $stmt->get_result();

        while ($row = $result->fetch_assoc()) {
            // Status mapping for search
            $status_display = 'ສັ່ງຊື້';
            if ($row['status'] == 'received')
                $status_display = 'ໄດ້ຮັບແລ້ວ';
            elseif ($row['status'] == 'partial')
                $status_display = 'ໄດ້ຮັບບາງສ່ວນ';
            elseif ($row['status'] == 'ordered')
                $status_display = 'ສັ່ງແລ້ວ';
            elseif ($row['status'] == 'pending')
                $status_display = 'ລໍຖ້າ';

            // Apply search filter (multi-field)
            if (!matchesSearch($search_term, $row['status'], 'ສັ່ງຊື້ສິນຄ້າ', $row['details'], 'LAK', 1, $row['PO_id'])) {
                continue;
            }

            $all[] = [
                'id' => $row['id'],
                'OR_id' => $row['PO_id'],
                'transaction_date' => $row['transaction_date'],
                'source' => 'purchase_orders',
                'source_type' => 'ສັ່ງຊື້ສິນຄ້າ',
                'details' => $row['details'],
                'amount' => floatval($row['total_amount']),
                'amount_in_lak' => 0,
                'expense_amount' => floatval($row['total_amount']),
                'cost_amount' => 0,
                'payment_status' => $row['status'],
                'full_amount' => floatval($row['total_amount']),
                'deposit_amount' => 0,
                'total_amount_lak' => floatval($row['total_amount']),
                'created_at' => $row['created_at'],
                'updated_at' => $row['created_at'],
                'currencys_id' => 'CUR_1',
                'currency_rate' => 1,
                'currency_symbol' => '₭',
                'currency_name' => 'LAK',
                'is_foreign_currency' => false
            ];
        }
    }

    // Sort by transaction_date/created_at
    usort($all, function ($a, $b) {
        $time_a = strtotime($a['transaction_date'] ?? $a['created_at']);
        $time_b = strtotime($b['transaction_date'] ?? $b['created_at']);
        if ($time_a == $time_b) {
            return ($a['id'] ?? 0) <=> ($b['id'] ?? 0);
        }
        return $time_a - $time_b;
    });

    // Apply pagination
    return array_slice($all, $offset, $limit);
}

function calculateTotals($conn, $shop_id, $search_term, $start_date, $end_date)
{
    $total_income = 0;
    $total_cost = 0;
    $total_expense = 0;

    // Calculate from orders - handle split payment logic like getTransactions
    $sql = "SELECT 
                o.id,
                o.OR_id,
                o.payment_status,
                o.order_type,
                o.total_amount,
                o.deposit_amount,
                o.total_amount_lak,
                o.currency_rate,
                o.currencys_id,
                o.created_at,
                o.updated_at,
                COALESCE(c.currencys_name, 'LAK') as currency_name
            FROM orders o
            LEFT JOIN currencys c ON o.currencys_id = c.currencys_id
            WHERE o.shop_id = ?
              AND o.deleted_at IS NULL
              AND o.payment_status IN ('paid', 'partial')
              AND DATE(o.created_at) BETWEEN ? AND ?";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        // Get cost for this order
        $cost_sql = "SELECT COALESCE(SUM(oi.quantity * p.cost_price), 0) as total_cost 
                     FROM order_items oi 
                     JOIN products p ON oi.product_id = p.PR_id 
                     WHERE oi.order_id = ?";
        $cost_stmt = $conn->prepare($cost_sql);
        $cost_stmt->bind_param("i", $row['id']);
        $cost_stmt->execute();
        $cost_result = $cost_stmt->get_result();
        $cost_row = $cost_result->fetch_assoc();
        $order_cost = floatval($cost_row['total_cost'] ?? 0);

        // Prepare values
        $source_type = 'POS';
        if ($row['order_type'] == 'online')
            $source_type = 'Online';
        elseif ($row['order_type'] == 'wholesale')
            $source_type = 'Wholesale';

        $details = $row['OR_id'];
        $currency_rate = floatval($row['currency_rate'] ?? 1);
        $currency_name = $row['currency_name'] ?? 'LAK';
        $total_lak = floatval($row['total_amount_lak'] ?? 0);
        $total_amount = floatval($row['total_amount']);
        $deposit_amount = floatval($row['deposit_amount']);

        // Check if this is a paid order that was previously partial
        $was_partial = $row['payment_status'] == 'paid'
            && $deposit_amount > 0
            && $row['created_at'] != $row['updated_at'];

        // Check if this was a pending order that is now paid
        // DISABLED: Treat as single paid transaction even if updated later
        $was_pending = false;
        /* $row['payment_status'] == 'paid'
            && $deposit_amount == 0
            && $row['created_at'] != $row['updated_at']; */

        if ($row['payment_status'] == 'paid') {
            if ($was_partial) {
                // Split into deposit entry and payment entry
                $full_amount = $total_amount;
                $remaining_amount = $full_amount - $deposit_amount;

                // Calculate proportional values
                $deposit_cost = 0;
                $deposit_lak = 0;
                $remaining_lak = 0;
                $remaining_cost = 0;

                if ($full_amount > 0) {
                    $deposit_cost = ($deposit_amount / $full_amount) * $order_cost;
                    $remaining_cost = $order_cost - $deposit_cost;

                    if ($total_lak > 0) {
                        $deposit_lak = ($deposit_amount / $full_amount) * $total_lak;
                        $remaining_lak = $total_lak - $deposit_lak;
                    } else {
                        $deposit_lak = $deposit_amount * $currency_rate;
                        $remaining_lak = $remaining_amount * $currency_rate;
                    }
                }

                // Entry 1: DEPOSIT - check if matches search
                if (matchesSearch($search_term, 'deposit_paid', $source_type, $details, $currency_name, $currency_rate, $row['OR_id'])) {
                    $total_income += $deposit_lak;
                    $total_cost += $deposit_cost;
                }

                // Entry 2: REMAINING PAYMENT - check if matches search
                if (matchesSearch($search_term, 'payment_received', $source_type, $details . ' (ຊຳລະສ່ວນທີ່ເຫຼືອ)', $currency_name, $currency_rate, $row['OR_id'])) {
                    $total_income += $remaining_lak;
                    $total_cost += $remaining_cost;
                }
            } elseif ($was_pending) {
                // Split into pending entry (no values) and payment entry (full values)
                // Entry 1: PENDING - no income/cost (skip if doesn't match)
                // Entry 2: PAYMENT - check if matches search
                if (matchesSearch($search_term, 'payment_received', $source_type, $details . ' (ຊຳລະເງິນ)', $currency_name, $currency_rate, $row['OR_id'])) {
                    $total_income += $total_lak;
                    $total_cost += $order_cost;
                }
            } else {
                // Normal paid order - check if matches search
                if (matchesSearch($search_term, 'paid', $source_type, $details, $currency_name, $currency_rate, $row['OR_id'])) {
                    $total_income += $total_lak;
                    $total_cost += $order_cost;
                }
            }
        } elseif ($row['payment_status'] == 'partial') {
            // Current partial order - check if matches search
            if (matchesSearch($search_term, 'partial', $source_type, $details, $currency_name, $currency_rate, $row['OR_id'])) {
                // Proportional income and cost
                if ($total_amount > 0 && $total_lak > 0) {
                    $deposit_income_lak = ($deposit_amount / $total_amount) * $total_lak;
                    $total_income += $deposit_income_lak;
                    $total_cost += ($deposit_amount / $total_amount) * $order_cost;
                } else {
                    $total_income += $deposit_amount * $currency_rate;
                    if ($total_amount > 0) {
                        $total_cost += ($deposit_amount / $total_amount) * $order_cost;
                    }
                }
            }
        }
    }

    // Add other_income (with search filter)
    $sql = "SELECT id, IN_id, COALESCE(description, category, 'ລາຍຮັບອື່ນໆ') as details, amount 
            FROM other_income 
            WHERE shop_id = ? AND deleted_at IS NULL AND DATE(income_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        if (!matchesSearch($search_term, 'paid', 'ລາຍຮັບອື່ນໆ', $row['details'], 'LAK', 1, $row['IN_id'])) {
            continue;
        }
        $total_income += floatval($row['amount']);
    }

    // Calculate expenses (with search filter)
    $sql = "SELECT id, EX_id, COALESCE(description, category, 'ລາຍຈ່າຍ') as details, amount 
            FROM expenses 
            WHERE shop_id = ? AND deleted_at IS NULL AND DATE(expense_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $shop_id, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        // Apply search filter
        if (!matchesSearch($search_term, 'paid', 'ລາຍຈ່າຍ', $row['details'], 'LAK', 1, $row['EX_id'])) {
            continue;
        }
        $total_expense += floatval($row['amount']);
    }

    // Net profit = Income - Cost - Expenses
    $total_profit = $total_income - $total_cost - $total_expense;

    return [
        'total_income' => $total_income,
        'total_expense' => $total_expense,
        'total_cost' => $total_cost,
        'total_profit' => $total_profit
    ];
}
?>