<?php
// AJAX endpoint for loading sales table body
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo '<tr><td colspan="14" class="empty-state">Unauthorized</td></tr>';
    exit;
}

require_once dirname(__DIR__) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo '<tr><td colspan="14" class="empty-state">ບໍ່ພົບຂໍ້ມູນຮ້ານຄ້າ</td></tr>';
    exit;
}

// Get parameters
$search_query = isset($_GET['search']) ? trim($_GET['search']) : '';

// When searching, allow empty dates to search all dates
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : (!empty($search_query) ? '' : date('Y-m-d'));
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : (!empty($search_query) ? '' : date('Y-m-d'));
$page = isset($_GET['page']) && is_numeric($_GET['page']) && $_GET['page'] > 0 ? intval($_GET['page']) : 1;
$per_page = 100;

// Calculate offset
$offset = ($page - 1) * $per_page;
if ($offset < 0)
    $offset = 0;

// Build query
$sql = "SELECT 
            o.id,
            o.OR_id,
            o.order_type,
            o.order_date,
            o.order_status,
            o.payment_status,
            o.subtotal,
            o.discount_amount,
            o.total_amount,
            o.tax_amount,
            o.vat_enabled,
            o.currency_rate,
            o.deposit_amount,
            o.shipping_code,
            o.printed_online,
            o.printed_shop,
            o.printed_count_online,
            o.printed_count_shop,
            o.currencys_id,
            o.refund_amount,
            COALESCE(cur.currencys_symbol, '₭') as currency_symbol,
            c.name as customer_name,
            c.phone as customer_phone,
            sc.company_name as shipping_company,
            sb.branch_name as shipping_branch
        FROM orders o
        LEFT JOIN customers c ON o.customer_id = c.CU_id
        LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id
        LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id
        LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id AND cur.shop_id = o.shop_id
        WHERE o.shop_id = ? AND o.deleted_at IS NULL";

$params = [$shop_id];
$types = "i";

// Add search condition
if (!empty($search_query)) {
    $sql .= " AND (
        o.OR_id LIKE ? OR 
        o.order_code LIKE ? OR 
        o.customer_id LIKE ? OR
        c.name LIKE ? OR 
        c.phone LIKE ? OR
        sc.company_name LIKE ? OR
        sb.branch_name LIKE ?
    )";
    $search_param = "%" . $search_query . "%";
    for ($i = 0; $i < 7; $i++) {
        $params[] = $search_param;
        $types .= "s";
    }
}


// Add date filter ONLY when not searching (search should find across all dates)
if (empty($search_query)) {
    if (!empty($date_from)) {
        $sql .= " AND DATE(o.order_date) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }
    if (!empty($date_to)) {
        $sql .= " AND DATE(o.order_date) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }
}

// Order by
$sql .= " ORDER BY o.created_at ASC";

// Add pagination
$sql .= " LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = $offset;
$types .= "ii";

// Execute query
$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

// Output rows
if ($result->num_rows > 0) {
    $count = $offset + 1;
    while ($row = $result->fetch_assoc()) {
        $order_type_class = '';
        $order_type_text = '-';

        switch ($row['order_type']) {
            case 'pos':
                $order_type_class = 'type-pos';
                $order_type_text = 'ໜ້າຮ້ານ';
                break;
            case 'online':
                $order_type_class = 'type-online';
                $order_type_text = 'ອອນລາຍ';
                break;
            case 'wholesale':
                $order_type_class = 'type-wholesale';
                $order_type_text = 'ສົ່ງ';
                break;
        }

        $payment_class = '';
        $payment_text = '-';
        switch ($row['payment_status']) {
            case 'paid':
                $payment_class = 'status-paid';
                $payment_text = 'ຈ່າຍແລ້ວ';
                break;
            case 'partial':
                $payment_class = 'status-partial';
                $payment_text = 'ມັດຈຳ';
                break;
            case 'pending':
                $payment_class = 'status-pending';
                $payment_text = 'ຍັງບໍ່ຈ່າຍ';
                break;
            case 'partial_refund':
                $payment_class = 'status-refunded';
                $payment_text = 'ຄືນບາງສ່ວນ';
                break;
            case 'refunded':
                $payment_class = 'status-refunded';
                $payment_text = 'ຄືນເຕັມ';
                break;
        }

        $customer_name = htmlspecialchars($row['customer_name'] ?? '-');
        $customer_phone = htmlspecialchars($row['customer_phone'] ?? '-');
        $shipping_company = htmlspecialchars($row['shipping_company'] ?? '-');
        $shipping_branch = htmlspecialchars($row['shipping_branch'] ?? '-');

        echo '<tr data-order-id="' . htmlspecialchars($row['OR_id']) . '" data-order-type="' . htmlspecialchars($row['order_type']) . '">';
        echo '<td class="bulk-checkbox"><input type="checkbox" class="order-checkbox" data-order-id="' . htmlspecialchars($row['OR_id']) . '"></td>';
        echo '<td>' . $count . '</td>';
        echo '<td><span class="badge ' . $order_type_class . '">' . $order_type_text . '</span></td>';
        echo '<td class="order-id-cell">' . htmlspecialchars($row['OR_id']) . '</td>';
        echo '<td>' . date('d/m/Y H:i', strtotime($row['order_date'])) . '</td>';
        echo '<td title="' . $customer_name . '">' . (mb_strlen($customer_name) > 15 ? mb_substr($customer_name, 0, 13) . '…' : $customer_name) . '</td>';
        echo '<td>' . $customer_phone . '</td>';
        echo '<td title="' . $shipping_company . '">' . (mb_strlen($shipping_company) > 12 ? mb_substr($shipping_company, 0, 10) . '…' : $shipping_company) . '</td>';
        echo '<td title="' . $shipping_branch . '">' . (mb_strlen($shipping_branch) > 12 ? mb_substr($shipping_branch, 0, 10) . '…' : $shipping_branch) . '</td>';
        echo '<td><span class="badge ' . $payment_class . '">' . $payment_text . '</span></td>';

        // Print status column
        $print_status_html = '';
        $order_type = $row['order_type'];
        if ($order_type === 'online') {
            if ($row['printed_online']) {
                $print_status_html = '<span class="badge status-paid" title="ພິມແລ້ວ ' . intval($row['printed_count_online']) . ' ຄັ້ງ"><i class="fas fa-check"></i> ພິມແລ້ວ</span>';
            } else {
                $print_status_html = '<span class="badge status-pending"><i class="fas fa-times"></i> ຍັງບໍ່ພິມ</span>';
            }
        } else {
            if ($row['printed_shop']) {
                $print_status_html = '<span class="badge status-paid" title="ພິມແລ້ວ ' . intval($row['printed_count_shop']) . ' ຄັ້ງ"><i class="fas fa-check"></i> ພິມແລ້ວ</span>';
            } else {
                $print_status_html = '<span class="badge status-pending"><i class="fas fa-times"></i> ຍັງບໍ່ພິມ</span>';
            }
        }
        echo '<td>' . $print_status_html . '</td>';

        $currency_symbol = htmlspecialchars($row['currency_symbol'] ?? '₭');
        $currencys_id = $row['currencys_id'] ?? 'CUR_1';

        // Use total_amount from database directly (no recalculation)
        $display_total = floatval($row['total_amount'] ?? 0);
        $refund_amount = floatval($row['refund_amount'] ?? 0);

        // Calculate net total after refund
        $net_total = $display_total - $refund_amount;

        // Format based on currency (USD = 2 decimals, others = 0 decimals)
        $decimals = ($currencys_id === 'CUR_3') ? 2 : 0;
        $total_formatted = number_format($display_total, $decimals);

        // Show refund info if there's a refund
        if ($refund_amount > 0) {
            $refund_formatted = number_format($refund_amount, $decimals);
            $net_formatted = number_format($net_total, $decimals);
            echo '<td class="amount-cell">';
            echo '<div style="text-decoration:line-through;color:#999;font-size:12px;">' . $currency_symbol . $total_formatted . '</div>';
            echo '<div style="color:#dc3545;font-size:11px;"><i class="fas fa-undo-alt"></i> -' . $currency_symbol . $refund_formatted . '</div>';
            echo '<div style="color:#28a745;font-weight:700;">' . $currency_symbol . $net_formatted . '</div>';
            echo '</td>';
        } else {
            echo '<td class="amount-cell">' . $currency_symbol . $total_formatted . '</td>';
        }
        echo '<td>';
        echo '<div class="action-btns">';
        echo '<button class="btn-icon view" onclick="viewOrder(\'' . $row['OR_id'] . '\')" title="ເບິ່ງ"><i class="fas fa-eye"></i></button>';
        echo '<button class="btn-icon edit" onclick="editOrder(\'' . $row['OR_id'] . '\')" title="ແກ້ໄຂ"><i class="fas fa-pen"></i></button>';
        echo '<button class="btn-icon delete" onclick="confirmDeleteOrder(\'' . $row['OR_id'] . '\')" title="ລົບ"><i class="fas fa-trash"></i></button>';
        echo '</div>';
        echo '</td>';
        echo '</tr>';
        $count++;
    }
} else {
    echo '<tr><td colspan="14" class="empty-state"><i class="fas fa-inbox" style="font-size:32px;opacity:.4;margin-bottom:12px;display:block"></i>ບໍ່ພົບຂໍ້ມູນ</td></tr>';
}

$stmt->close();
?>