<?php
// AJAX endpoint for loading pagination
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    exit;
}

require_once dirname(__DIR__) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    exit;
}

// Get parameters
$search_query = isset($_GET['search']) ? trim($_GET['search']) : '';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-d');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$page = isset($_GET['page']) && is_numeric($_GET['page']) && $_GET['page'] > 0 ? intval($_GET['page']) : 1;
$per_page = 100;

// Count total rows
$count_sql = "SELECT COUNT(*) as total
                FROM orders o
                LEFT JOIN customers c ON o.customer_id = c.CU_id
                LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id
                LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id
                WHERE o.shop_id = ? AND o.deleted_at IS NULL";

$params = [$shop_id];
$types = "i";

// Add search condition
if (!empty($search_query)) {
    $count_sql .= " AND (
            o.OR_id LIKE ? OR 
            o.order_code LIKE ? OR 
            o.customer_id LIKE ? OR
            c.name LIKE ? OR 
            c.phone LIKE ? OR
            sc.company_name LIKE ? OR
            sb.branch_name LIKE ?
        )";
    $search_param = "%" . $search_query . "%";
    for ($i = 0; $i < 7; $i++) {
        $params[] = $search_param;
        $types .= "s";
    }
}

// Add date filter ONLY when not searching (search should find across all dates)
if (empty($search_query)) {
    if (!empty($date_from)) {
        $count_sql .= " AND DATE(o.order_date) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }
    if (!empty($date_to)) {
        $count_sql .= " AND DATE(o.order_date) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }
}

// Execute count query
$stmt = $conn->prepare($count_sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$total_rows = $result->fetch_assoc()['total'] ?? 0;
$stmt->close();

// Calculate total pages
$total_pages = ceil($total_rows / $per_page);
if ($page > $total_pages && $total_pages > 0) {
    $page = $total_pages;
}

// Output pagination HTML
echo '<div class="pagination-container">';
echo '<div class="pagination-info">ທັງໝົດ ' . number_format($total_rows) . ' ລາຍການ</div>';
echo '<div class="pagination-buttons">';

if ($total_pages > 1) {
    // First page button
    if ($page > 1) {
        echo '<button class="page-btn" onclick="loadTablePage(1)" title="ໜ້າທຳອິດ"><i class="fas fa-angle-double-left"></i></button>';
        echo '<button class="page-btn" onclick="loadTablePage(' . ($page - 1) . ')"><i class="fas fa-angle-left"></i></button>';
    }

    // Page numbers
    $start = max(1, $page - 2);
    $end = min($total_pages, $page + 2);

    if ($start > 1) {
        echo '<span style="padding:0 8px;color:var(--text-secondary)">...</span>';
    }

    for ($i = $start; $i <= $end; $i++) {
        $active = ($i == $page) ? ' active' : '';
        echo '<button class="page-btn' . $active . '" onclick="loadTablePage(' . $i . ')">' . $i . '</button>';
    }

    if ($end < $total_pages) {
        echo '<span style="padding:0 8px;color:var(--text-secondary)">...</span>';
    }

    // Next/Last page buttons
    if ($page < $total_pages) {
        echo '<button class="page-btn" onclick="loadTablePage(' . ($page + 1) . ')"><i class="fas fa-angle-right"></i></button>';
        echo '<button class="page-btn" onclick="loadTablePage(' . $total_pages . ')" title="ໜ້າສຸດທ້າຍ"><i class="fas fa-angle-double-right"></i></button>';
    }
}

echo '</div>';
echo '</div>';
?>