<?php
/**
 * Print Shop Orders - ใบบิลซื้อ-ขาย สำหรับขายหน้าร้าน
 * Paper size: 70mm x 100mm
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
date_default_timezone_set('Asia/Vientiane');

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo "Unauthorized";
    exit;
}

require_once dirname(__DIR__) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo "Shop not found";
    exit;
}

// Get order IDs
$order_ids = [];
if (isset($_GET['order_ids'])) {
    $order_ids = explode(',', $_GET['order_ids']);
} elseif (isset($_GET['order_id'])) {
    $order_ids = [$_GET['order_id']];
}

if (empty($order_ids)) {
    header("Location: sales_history.php");
    exit;
}

// Get shop info
$shop_sql = "SELECT s.*, u.first_name, u.last_name, u.phone as owner_phone 
             FROM shops s 
             LEFT JOIN users u ON s.owner_user_id = u.id 
             WHERE s.id = ?";
$shop_stmt = $conn->prepare($shop_sql);
$shop_stmt->bind_param("i", $shop_id);
$shop_stmt->execute();
$shop_result = $shop_stmt->get_result();
$shop = $shop_result->fetch_assoc();
$shop_stmt->close();

$shop_name = $shop['shop_name'] ?? 'ร้านค้า';
$shop_phone = $shop['phone'] ?? $shop['whatsapp'] ?? '';
$shop_facebook = $shop['facebook'] ?? '';
$shop_logo = $shop['logo'] ?? 'img/Logo.png';
?>
<!DOCTYPE html>
<html lang="lo">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ພິມໃບບິນໜ້າຮ້ານ</title>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            font-family: 'Noto Sans Lao', sans-serif !important;
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: #f5f5f5;
        }

        .print-button {
            text-align: center;
            padding: 20px;
            background: #fff;
            margin-bottom: 10px;
        }

        .print-button button {
            padding: 12px 30px;
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
        }

        .print-button button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.4);
        }

        .bill-container {
            width: 70mm;
            height: 100mm;
            margin: 10px auto;
            padding: 8px;
            background: #fff;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            position: relative;
        }

        .bill-content {
            transform-origin: top left;
        }

        .bill-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            border-bottom: 1px solid #bbb;
            padding-bottom: 8px;
            margin-bottom: 2mm;
            position: relative;
        }

        .bill-header img {
            width: 35px;
            object-fit: contain;
        }

        .bill-title-section {
            position: absolute;
            left: 0;
            right: 0;
            top: -6px;
            text-align: center;
            z-index: 2;
        }

        .bill-title {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 0.2mm;
        }

        .bill-shop {
            font-size: 8pt;
            font-weight: 500;
            margin-top: -2px;
        }

        .bill-meta {
            text-align: right;
            min-width: 32mm;
            font-size: 7pt;
        }

        .bill-meta span {
            display: block;
            margin-top: 1mm;
        }

        .bill-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 7pt;
            margin-bottom: 2mm;
        }

        .bill-table th,
        .bill-table td {
            border-bottom: none;
            padding: 0.5px 0;
            text-align: center;
        }

        .bill-table th {
            font-weight: bold;
        }

        .bill-table .item-name {
            text-align: left;
            padding-left: 2mm;
            width: 32mm;
            max-width: 32mm;
            word-break: break-all;
        }

        .bill-table .item-price {
            text-align: right;
            padding-right: 2mm;
        }

        .bill-table .item-qty {
            text-align: center;
            width: 12mm;
        }

        .bill-table .item-no {
            text-align: center;
            width: 8mm;
        }

        .empty-row td {
            border: none;
            color: #bbb;
        }

        .bill-total-row {
            font-size: 14px;
            font-weight: bold;
            text-align: right;
            margin-top: -4px;
            margin-bottom: 1mm;
            border-top: 1px solid #bbb;
            border-bottom: 1px solid #bbb;
            padding: 4px 0 2px 0;
            line-height: 1.2;
        }

        .bill-total-row .main-total {
            font-size: 13px;
            font-weight: bold;
            color: #EE4035;
        }

        .bill-total-row .sub-info {
            font-size: 9px;
            font-weight: normal;
            color: #333;
        }

        .bill-footer {
            margin-top: -5px;
            font-size: 7pt;
        }

        .bill-social {
            font-size: 7pt;
        }

        .bill-social i {
            margin-right: 2px;
        }

        .thankyou {
            margin-top: 2px;
            font-size: 5pt;
            text-align: center;
        }

        @media print {
            @page {
                size: 70mm 100mm;
                margin: 0;
            }

            body {
                width: 70mm;
                margin: 0;
                padding: 0;
                background: #fff;
            }

            .print-button {
                display: none;
            }

            .bill-container {
                width: 70mm;
                height: 100mm;
                margin: 0;
                padding: 8px;
                box-shadow: none;
                page-break-after: always;
                page-break-inside: avoid;
                overflow: hidden;
            }

            .bill-container:last-child {
                page-break-after: auto;
            }
        }

        /* SweetAlert2 styling */
        .swal2-popup,
        .swal2-title,
        .swal2-content,
        .swal2-confirm,
        .swal2-cancel {
            font-family: 'Noto Sans Lao', sans-serif !important;
        }
    </style>
</head>

<body>
    <div class="print-button">
        <button onclick="window.print()">🖨️ ພິມໃບບິນ</button>
    </div>

    <?php foreach ($order_ids as $order_id): ?>
        <?php
        $order_id = $conn->real_escape_string(trim($order_id));

        // Query order
        $sql = "SELECT o.*, 
                   c.name as customer_name, 
                   c.phone as customer_phone,
                   cur.currencys_symbol,
                   cur.currencys_name
            FROM orders o
            LEFT JOIN customers c ON o.customer_id = c.CU_id
            LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id AND cur.shop_id = o.shop_id
            WHERE o.OR_id = ? AND o.shop_id = ?";

        $stmt = $conn->prepare($sql);
        $stmt->bind_param("si", $order_id, $shop_id);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo "<div style='color:red; text-align:center; padding:20px;'>ບໍ່ພົບຂໍ້ມູນໃບບິນ: $order_id</div>";
            continue;
        }

        $order = $result->fetch_assoc();
        $stmt->close();

        // Get order items
        $items_sql = "SELECT oi.*, p.product_name 
                  FROM order_items oi 
                  LEFT JOIN products p ON oi.product_id = p.PR_id 
                  WHERE oi.order_id = ?";
        $items_stmt = $conn->prepare($items_sql);
        $items_stmt->bind_param("i", $order['id']);
        $items_stmt->execute();
        $items_result = $items_stmt->get_result();

        $items = [];
        while ($row = $items_result->fetch_assoc()) {
            $items[] = $row;
        }
        $items_stmt->close();

        $date = date('d/m/Y', strtotime($order['order_date']));
        $total = floatval($order['total_amount'] ?? 0);
        $discount = floatval($order['discount_amount'] ?? 0);
        $deposit = floatval($order['deposit_amount'] ?? 0);
        $final_total = $total;
        $cash_received = floatval($order['cash_received'] ?? 0);
        $cash_change = floatval($order['cash_change'] ?? 0);

        // Get currency symbol (default to K if not set)
        $currency_symbol = $order['currencys_symbol'] ?? 'K';
        $currency_rate = floatval($order['currency_rate'] ?? 1);

        // total_amount, deposit_amount, cash_received, cash_change are already stored in the selected currency
        // No need to divide by currency_rate - just display as-is
        $total_display = $final_total;
        $deposit_display = $deposit;
        $cash_received_display = $cash_received;
        $cash_change_display = $cash_change;

        // Use decimals for foreign currency (non-LAK)
        $decimals = ($currency_rate > 1) ? 0 : 0; // Both use 0 decimals for cleaner display
        ?>
        <div class="bill-container" id="bill-<?= $order['id'] ?>">
            <div class="bill-content">
                <div class="bill-header">
                    <div
                        style="flex: 0 0 auto; display: flex; flex-direction: column; align-items: flex-start; padding-top: 6px;">
                        <?php if ($shop_logo && file_exists($shop_logo)): ?>
                            <img src="<?= htmlspecialchars($shop_logo) ?>" alt="logo">
                        <?php else: ?>
                            <img src="img/Logo.png" alt="logo" onerror="this.style.display='none'">
                        <?php endif; ?>
                    </div>
                    <div class="bill-title-section">
                        <div class="bill-title">ໃບບິນຊື້-ຂາຍ</div>
                        <div class="bill-shop">ຮ້ານ
                            <?= htmlspecialchars($shop_name) ?>
                        </div>
                    </div>
                    <div class="bill-meta">
                        <span>ເລກທີ:
                            <?= htmlspecialchars($order['OR_id']) ?>
                        </span>
                        <span>
                            <?= $date ?>
                        </span>
                    </div>
                </div>

                <table class="bill-table">
                    <thead>
                        <tr>
                            <th class="item-no">ອັນດັບ</th>
                            <th class="item-name">ລາຍການ</th>
                            <th class="item-qty">ຈຳນວນ</th>
                            <th class="item-price">ລາຄາ</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $i = 1;
                        foreach ($items as $item):
                            ?>
                            <tr>
                                <td class="item-no">
                                    <?= $i ?>
                                </td>
                                <td class="item-name">
                                    <?= htmlspecialchars($item['product_name'] ?? '-') ?>
                                </td>
                                <td class="item-qty">
                                    <?= number_format($item['quantity'], 0) ?>
                                </td>
                                <td class="item-price">
                                    <?= number_format($item['total'] ?? 0, $decimals) ?>        <?= htmlspecialchars($currency_symbol) ?>
                                </td>
                            </tr>
                            <?php $i++; endforeach; ?>
                        <?php
                        // Add empty rows if less than 5 items (reduced to fit)
                        $itemCount = count($items);
                        if ($itemCount < 5) {
                            for ($j = $itemCount + 1; $j <= 5; $j++) {
                                ?>
                                <tr class="empty-row">
                                    <td>-</td>
                                    <td></td>
                                    <td></td>
                                    <td></td>
                                </tr>
                                <?php
                            }
                        }
                        ?>
                    </tbody>
                </table>

                <div class="bill-total-row">
                    <span class="main-total">ລາຄາລວມທີ່ຕ້ອງຈ່າຍ:
                        <?= number_format($total_display, $decimals) ?>     <?= htmlspecialchars($currency_symbol) ?>
                    </span><br>
                    <?php if ($order['payment_status'] === 'pending'): ?>
                        <span class="main-total">ຍັງບໍ່ທັນຈ່າຍ</span>
                    <?php elseif ($order['payment_status'] === 'partial'): ?>
                        <span class="main-total">ມັດຈຳ:
                            <?= number_format($deposit_display, $decimals) ?>         <?= htmlspecialchars($currency_symbol) ?>
                        </span>
                    <?php else: ?>
                        <span class="sub-info">ເງິນທີ່ໄດ້ຮັບ :
                            <?= $cash_received > 0 ? number_format($cash_received_display, $decimals) : '-' ?>
                            <?= htmlspecialchars($currency_symbol) ?>
                        </span><br>
                        <span class="sub-info">ເງິນທອນ :
                            <?= $cash_change > 0 ? number_format($cash_change_display, $decimals) : '-' ?>
                            <?= htmlspecialchars($currency_symbol) ?>
                        </span>
                    <?php endif; ?>
                </div>

                <div class="bill-footer">
                    <div class="bill-social">
                        <i class="fab fa-tiktok"></i>
                        <i class="fab fa-facebook"></i>
                        <?= htmlspecialchars($shop_facebook ?: $shop_name) ?>
                        <i class="fab fa-whatsapp"></i>
                        <?= htmlspecialchars($shop_phone) ?>
                    </div>
                    <div class="thankyou">THANK YOU ☺</div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            var orderIds = <?php echo json_encode($order_ids); ?>;
            var saleType = "shop";

            var printDialogOpened = false;
            var userPrinted = false;

            // Auto-scale function to fit content in single page
            function autoScaleBills() {
                var containers = document.querySelectorAll('.bill-container');
                containers.forEach(function (container) {
                    var content = container.querySelector('.bill-content');
                    if (!content) return;

                    // Get container dimensions (in pixels, 70mm x 100mm at 96dpi)
                    var containerWidth = container.offsetWidth - 16; // minus padding
                    var containerHeight = container.offsetHeight - 16;

                    // Get content dimensions
                    var contentWidth = content.scrollWidth;
                    var contentHeight = content.scrollHeight;

                    // Calculate scale factor
                    var scaleX = containerWidth / contentWidth;
                    var scaleY = containerHeight / contentHeight;
                    var scale = Math.min(scaleX, scaleY, 1); // Don't scale up, only down

                    if (scale < 1) {
                        content.style.transform = 'scale(' + scale + ')';
                        content.style.transformOrigin = 'top left';
                    }
                });
            }

            // Run auto-scale
            autoScaleBills();
            window.addEventListener('resize', autoScaleBills);

            setTimeout(function () {
                autoScaleBills();
                printDialogOpened = true;
                window.print();
            }, 500);

            window.onafterprint = function () {
                if (printDialogOpened) {
                    Swal.fire({
                        title: 'ຢືນຢັນການພິມ',
                        text: 'ທ່ານໄດ້ພິມໃບບິນແລ້ວບໍ່?',
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonColor: '#28a745',
                        cancelButtonColor: '#dc3545',
                        confirmButtonText: '✓ ພິມແລ້ວ',
                        cancelButtonText: '✗ ຍົກເລີກ',
                        allowOutsideClick: false,
                        allowEscapeKey: false
                    }).then((result) => {
                        if (result.isConfirmed) {
                            userPrinted = true;
                            updatePrintedStatus();
                        } else {
                            userPrinted = true;
                            window.close();
                        }
                    });
                }
            };

            function updatePrintedStatus() {
                fetch('api/update_print_status.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        order_ids: orderIds,
                        type: saleType
                    })
                })
                    .then(res => res.json())
                    .then(data => {
                        if (data.status === 'success') {
                            Swal.fire({
                                title: 'ສຳເລັດ!',
                                text: 'ອັບເດດສະຖານະການພິມແລ້ວ',
                                icon: 'success',
                                timer: 1500,
                                showConfirmButton: false
                            }).then(() => {
                                window.close();
                            });
                        } else {
                            window.close();
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        window.close();
                    });
            }

            window.onbeforeunload = function () {
                if (!userPrinted && printDialogOpened) {
                    return 'ທ່ານຍັງບໍ່ໄດ້ພິມໃບບິນ ສະຖານະການພິມຈະບໍ່ຖືກອັບເດດ';
                }
            };
        });
    </script>
</body>

</html>