<?php
/**
 * Print Online Orders - ใบฝากพัสดุ สำหรับขายออนไลน์
 * Paper size: 70mm x 100mm
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
date_default_timezone_set('Asia/Vientiane');

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo "Unauthorized";
    exit;
}

require_once dirname(__DIR__) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo "Shop not found";
    exit;
}

// Get order IDs
$order_ids = [];
if (isset($_GET['order_ids'])) {
    $order_ids = explode(',', $_GET['order_ids']);
} elseif (isset($_GET['order_id'])) {
    $order_ids = [$_GET['order_id']];
}

if (empty($order_ids)) {
    header("Location: sales_history.php");
    exit;
}

// Sanitize order IDs
$sanitized_order_ids = [];
foreach ($order_ids as $id) {
    $sanitized_order_ids[] = $conn->real_escape_string(trim($id));
}

$placeholders = implode(',', array_fill(0, count($sanitized_order_ids), '?'));
$types = str_repeat('s', count($sanitized_order_ids));

// Get shop info
$shop_sql = "SELECT s.*, u.first_name, u.last_name, u.phone as owner_phone 
             FROM shops s 
             LEFT JOIN users u ON s.owner_user_id = u.id 
             WHERE s.id = ?";
$shop_stmt = $conn->prepare($shop_sql);
$shop_stmt->bind_param("i", $shop_id);
$shop_stmt->execute();
$shop_result = $shop_stmt->get_result();
$shop = $shop_result->fetch_assoc();
$shop_stmt->close();

$shop_name = $shop['shop_name'] ?? 'ร้านค้า';
$owner_name = trim(($shop['first_name'] ?? '') . ' ' . ($shop['last_name'] ?? ''));
$shop_phone = $shop['phone'] ?? $shop['owner_phone'] ?? '';
?>
<!DOCTYPE html>
<html lang="lo">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ພິມໃບບິນອອນລາຍ</title>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@400;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        * {
            font-family: 'Noto Sans Lao', sans-serif !important;
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: #f5f5f5;
        }

        .print-button {
            text-align: center;
            padding: 20px;
            background: #fff;
            margin-bottom: 10px;
        }

        .print-button button {
            padding: 12px 30px;
            background: linear-gradient(135deg, #4a90e2, #357abd);
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
        }

        .print-button button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(74, 144, 217, 0.4);
        }

        .invoice-container {
            width: 70mm;
            height: 100mm;
            margin: 10px auto;
            padding: 3mm;
            background: #fff;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            position: relative;
        }

        .invoice-content {
            transform-origin: top left;
        }

        .bill-header {
            text-align: center;
            margin-bottom: 2mm;
        }

        .bill-header h1 {
            font-size: 18px;
            margin-bottom: 1mm;
        }

        .bill-header h2 {
            font-size: 11px;
            margin-bottom: 1mm;
            font-weight: 500;
        }

        .bill-header .meta-row {
            display: flex;
            justify-content: space-between;
            font-size: 10px;
        }

        hr {
            border: none;
            border-top: 1px dashed #999;
            margin: 2mm 0;
        }

        .sender-receiver {
            display: flex;
            justify-content: space-between;
            gap: 3mm;
        }

        .sender-receiver>div {
            flex: 1;
        }

        .sender-receiver h3 {
            font-size: 14px;
            margin-bottom: 1mm;
            text-align: center;
        }

        .sender-receiver p {
            font-size: 10px;
            margin-bottom: 0.5mm;
        }

        .transport-info {
            display: flex;
            justify-content: space-between;
            gap: 3mm;
        }

        .transport-info>div {
            flex: 1;
            text-align: center;
        }

        .transport-info h4 {
            font-size: 10px;
            margin-bottom: 1mm;
            font-weight: 600;
        }

        .transport-info p {
            font-size: 10px;
        }

        .price-info {
            display: flex;
            justify-content: space-around;
            text-align: center;
            margin: 2mm 0;
        }

        .price-info>div h4 {
            font-size: 10px;
            margin-bottom: 1mm;
            font-weight: 600;
        }

        .price-info>div p {
            font-size: 11px;
            font-weight: 600;
        }

        .items-list {
            font-size: 10px;
            padding: 1mm 0;
        }

        .items-list p {
            margin-bottom: 0.5mm;
        }

        .items-list.two-columns {
            display: flex;
            flex-wrap: wrap;
        }

        .items-list.two-columns .items-col {
            width: 50%;
            padding-right: 1mm;
        }

        .items-list.two-columns .items-col p {
            margin-bottom: 0.3mm;
            font-size: 9px;
        }

        .thank-you {
            text-align: center;
            font-size: 11px;
            margin-top: 2mm;
        }

        @media print {
            @page {
                size: 70mm 100mm;
                margin: 0;
            }

            body {
                width: 70mm;
                margin: 0;
                padding: 0;
                background: #fff;
            }

            .print-button {
                display: none;
            }

            .invoice-container {
                width: 70mm;
                height: 100mm;
                margin: 0;
                padding: 3mm;
                box-shadow: none;
                page-break-after: always;
                page-break-inside: avoid;
                overflow: hidden;
            }

            .invoice-container:last-child {
                page-break-after: auto;
            }
        }

        /* SweetAlert2 styling */
        .swal2-popup,
        .swal2-title,
        .swal2-content,
        .swal2-confirm,
        .swal2-cancel {
            font-family: 'Noto Sans Lao', sans-serif !important;
        }
    </style>
</head>

<body>
    <div class="print-button">
        <button onclick="window.print()">🖨️ ພິມໃບບິນ</button>
    </div>

    <?php
    // Query orders
    $sql = "SELECT o.*, 
                   c.name as customer_name, 
                   c.phone as customer_phone,
                   sc.company_name as shipping_company,
                   sb.branch_name as shipping_branch,
                   cur.currencys_symbol,
                   cur.currencys_name
            FROM orders o
            LEFT JOIN customers c ON o.customer_id = c.CU_id
            LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id
            LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id
            LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id AND cur.shop_id = o.shop_id
            WHERE o.OR_id IN ($placeholders) AND o.shop_id = ?";

    $params = array_merge($sanitized_order_ids, [$shop_id]);
    $types .= "i";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($order = $result->fetch_assoc()) {
        $order_id = $order['OR_id'];

        // Get order items
        $items_sql = "SELECT oi.*, p.product_name 
                      FROM order_items oi 
                      LEFT JOIN products p ON oi.product_id = p.PR_id 
                      WHERE oi.order_id = ?";
        $items_stmt = $conn->prepare($items_sql);
        $items_stmt->bind_param("i", $order['id']);
        $items_stmt->execute();
        $items_result = $items_stmt->get_result();

        $items = [];
        while ($item = $items_result->fetch_assoc()) {
            $items[] = $item;
        }
        $items_stmt->close();

        // Calculate remaining amount
        $subtotal = floatval($order['subtotal'] ?? 0);
        $discount = floatval($order['discount_amount'] ?? 0);
        $deposit = floatval($order['deposit_amount'] ?? 0);
        $remaining = $subtotal - $discount - $deposit;

        // Get currency symbol (default to ກີບ if not set)
        $currency_symbol = $order['currencys_symbol'] ?? '₭';
        $currency_rate = floatval($order['currency_rate'] ?? 1);

        // subtotal, discount, deposit are already stored in the selected currency
        // No need to divide by currency_rate - just display as-is
        $remaining_display = $remaining;
        $decimals = 0; // Use 0 decimals for cleaner display
    
        // Determine COD type from notes
        $cod_type = 'ຄ່າຝາກປາຍທາງ';
        if (strpos($order['notes'] ?? '', 'ຫັກ COD') !== false) {
            $cod_type = 'ຫັກ COD';
        }
        ?>
        <div class="invoice-container" id="invoice-<?php echo $order['id']; ?>">
            <div class="invoice-content">
                <div class="bill-header">
                    <h1>ບິນຝາກພັດສະດຸ</h1>
                    <h2>
                        <?php echo htmlspecialchars($shop_name); ?>
                    </h2>
                    <div class="meta-row">
                        <span>ເລກທີບິນ:
                            <?php echo htmlspecialchars($order['OR_id']); ?>
                        </span>
                        <span>ວັນທີ:
                            <?php echo date('d/m/Y', strtotime($order['order_date'])); ?>
                        </span>
                    </div>
                </div>

                <hr>

                <div class="sender-receiver">
                    <div class="sender">
                        <h3>ຜູ້ຝາກ</h3>
                        <p><strong>ຊື່:</strong>
                            <?php echo htmlspecialchars($owner_name ?: '-'); ?>
                        </p>
                        <p><strong>ເບີໂທ:</strong>
                            <?php echo htmlspecialchars($shop_phone ?: '-'); ?>
                        </p>
                    </div>
                    <div class="receiver">
                        <h3>ຜູ້ຮັບ</h3>
                        <p><strong>ຊື່ລູກຄ້າ:</strong>
                            <?php echo htmlspecialchars($order['customer_name'] ?? '-'); ?>
                        </p>
                        <p><strong>ເບີໂທ:</strong>
                            <?php echo htmlspecialchars($order['customer_phone'] ?? '-'); ?>
                        </p>
                    </div>
                </div>

                <hr>

                <div class="transport-info">
                    <div>
                        <h4>ບໍລິສັດຂົນສົ່ງ</h4>
                        <p>
                            <?php echo htmlspecialchars($order['shipping_company'] ?? '-'); ?>
                        </p>
                    </div>
                    <div>
                        <h4>ສາຂາຂົນສົ່ງ</h4>
                        <p>
                            <?php echo htmlspecialchars($order['shipping_branch'] ?? '-'); ?>
                        </p>
                    </div>
                </div>

                <hr>

                <div class="price-info">
                    <div>
                        <h4>ລາຄາ</h4>
                        <p>
                            <?php
                            if ($order['payment_status'] == 'paid') {
                                echo "ຈ່າຍແລ້ວ";
                            } else {
                                echo number_format($remaining_display, $decimals) . " " . htmlspecialchars($currency_symbol);
                            }
                            ?>
                        </p>
                    </div>
                    <div>
                        <h4>ຄ່າຝາກ</h4>
                        <p>
                            <?php echo htmlspecialchars($cod_type); ?>
                        </p>
                    </div>
                </div>

                <hr>

                <div class="items-list<?php echo count($items) >= 4 ? ' two-columns' : ''; ?>">
                    <?php if (count($items) > 0): ?>
                        <?php if (count($items) >= 4): ?>
                            <?php
                            $halfCount = ceil(count($items) / 2);
                            $leftItems = array_slice($items, 0, $halfCount);
                            $rightItems = array_slice($items, $halfCount);
                            ?>
                            <div class="items-col">
                                <?php foreach ($leftItems as $item): ?>
                                    <p>
                                        <?php echo htmlspecialchars($item['product_name'] ?? '-'); ?>:
                                        <?php echo number_format($item['quantity'], 0); ?>
                                    </p>
                                <?php endforeach; ?>
                            </div>
                            <div class="items-col">
                                <?php foreach ($rightItems as $item): ?>
                                    <p>
                                        <?php echo htmlspecialchars($item['product_name'] ?? '-'); ?>:
                                        <?php echo number_format($item['quantity'], 0); ?>
                                    </p>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <?php foreach ($items as $item): ?>
                                <p>
                                    <?php echo htmlspecialchars($item['product_name'] ?? '-'); ?>:
                                    <?php echo number_format($item['quantity'], 0); ?>
                                </p>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    <?php else: ?>
                        <p>ບໍ່ພົບລາຍການສິນຄ້າ</p>
                    <?php endif; ?>
                </div>

                <hr>

                <div class="thank-you">
                    <p>ຂອບໃຈທີ່ໃຊ້ບໍລິການ🤍</p>
                </div>
            </div>
        </div>
        <?php
    }

    $stmt->close();
    ?>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            var orderIds = <?php echo json_encode($order_ids); ?>;
            var saleType = "online";

            var printDialogOpened = false;
            var userPrinted = false;

            // Auto-scale function to fit content in single page
            function autoScaleInvoices() {
                var containers = document.querySelectorAll('.invoice-container');
                containers.forEach(function (container) {
                    var content = container.querySelector('.invoice-content');
                    if (!content) return;

                    // Get container dimensions (in pixels, 70mm x 100mm at 96dpi)
                    var containerWidth = container.offsetWidth - 6; // minus padding
                    var containerHeight = container.offsetHeight - 6;

                    // Get content dimensions
                    var contentWidth = content.scrollWidth;
                    var contentHeight = content.scrollHeight;

                    // Calculate scale factor
                    var scaleX = containerWidth / contentWidth;
                    var scaleY = containerHeight / contentHeight;
                    var scale = Math.min(scaleX, scaleY, 1); // Don't scale up, only down

                    if (scale < 1) {
                        content.style.transform = 'scale(' + scale + ')';
                        content.style.transformOrigin = 'top left';
                    }
                });
            }

            // Run auto-scale
            autoScaleInvoices();
            window.addEventListener('resize', autoScaleInvoices);

            setTimeout(function () {
                autoScaleInvoices();
                printDialogOpened = true;
                window.print();
            }, 500);

            window.onafterprint = function () {
                if (printDialogOpened) {
                    Swal.fire({
                        title: 'ຢືນຢັນການພິມ',
                        text: 'ທ່ານໄດ້ພິມໃບບິນແລ້ວບໍ່?',
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonColor: '#28a745',
                        cancelButtonColor: '#dc3545',
                        confirmButtonText: '✓ ພິມແລ້ວ',
                        cancelButtonText: '✗ ຍົກເລີກ',
                        allowOutsideClick: false,
                        allowEscapeKey: false
                    }).then((result) => {
                        if (result.isConfirmed) {
                            userPrinted = true;
                            updatePrintedStatus();
                        } else {
                            userPrinted = true;
                            window.close();
                        }
                    });
                }
            };

            function updatePrintedStatus() {
                fetch('api/update_print_status.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        order_ids: orderIds,
                        type: saleType
                    })
                })
                    .then(res => res.json())
                    .then(data => {
                        if (data.status === 'success') {
                            Swal.fire({
                                title: 'ສຳເລັດ!',
                                text: 'ອັບເດດສະຖານະການພິມແລ້ວ',
                                icon: 'success',
                                timer: 1500,
                                showConfirmButton: false
                            }).then(() => {
                                window.close();
                            });
                        } else {
                            window.close();
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        window.close();
                    });
            }

            window.onbeforeunload = function () {
                if (!userPrinted && printDialogOpened) {
                    return 'ທ່ານຍັງບໍ່ໄດ້ພິມໃບບິນ ສະຖານະການພິມຈະບໍ່ຖືກອັບເດດ';
                }
            };
        });
    </script>
</body>

</html>