<?php
require_once("auth.php");
require_once("db_con.php");
$formatNumberCustom_defined = false;
if (!function_exists('formatNumberCustom')) {
    function formatNumberCustom($num) {
        $formatted = number_format($num, 2);
        $formatted = rtrim(rtrim($formatted, '0'), '.');
        return $formatted;
    }
    $formatNumberCustom_defined = true;
}

// Set default date range to current day
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');


// --- Simple file-based cache (10 min) ---
$transactions = [];
$cache_dir = __DIR__ . '/cache';
if (!is_dir($cache_dir)) {
    mkdir($cache_dir, 0777, true);
}
$cache_key = md5($start_date . '_' . $end_date);
$cache_file = $cache_dir . "/summary_income_{$cache_key}.json";
$cache_lifetime = 600; // 10 นาที

if (file_exists($cache_file) && (time() - filemtime($cache_file) < $cache_lifetime)) {
    $transactions = json_decode(file_get_contents($cache_file), true);
} else {
    $start_date_formatted = date('Y-m-d', strtotime($start_date));
    $end_date_formatted = date('Y-m-d', strtotime($end_date));
    $transactions = getTransactions($conn, $start_date, $end_date);
    file_put_contents($cache_file, json_encode($transactions));
}

// Function to get all transactions (orders, other_income, expenses)
function getTransactions($conn, $start_date, $end_date) {
    $transactions = [];
    
    // Convert dates to ensure proper format for SQL comparison
    $start_date_formatted = date('Y-m-d', strtotime($start_date));
    $end_date_formatted = date('Y-m-d', strtotime($end_date));
    
    // Get orders (POS) with cost calculation for profit
    // เพิ่ม index ที่ order_date, updated_at, created_at, payment_status, order_id ในฐานข้อมูลเพื่อความเร็ว
    $sql = "SELECT SQL_NO_CACHE o.order_id as id, o.order_date as transaction_date, 
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'orders'
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຍັງບໍ່ຈ່າຍ' THEN 'pending'
                ELSE 'other'
            END as source,
            'POS' as source_type, 
            CONCAT('Order #', o.order_id) as details,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN o.deposit_amount
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (o.subtotal - COALESCE(o.discount, 0))
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
                ELSE (o.subtotal - COALESCE(o.discount, 0))
            END as amount,
            0 as expense_amount,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 
                            (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (
                                SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi FORCE INDEX (order_id) JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
                            )
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE 
                            (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi FORCE INDEX (order_id) JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 
                    (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (
                        SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi FORCE INDEX (order_id) JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
                    )
                WHEN o.payment_status = 'ຍັງບໍ່ຈ່າຍ' THEN 0
                ELSE 
                    (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi FORCE INDEX (order_id) JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
            END as cost_amount,
            o.payment_status,
            o.total_amount as full_amount,
            o.subtotal,
            COALESCE(o.discount, 0) as discount,
            o.deposit_amount,
            o.created_at,
            o.updated_at,
            CASE
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'original'
                    END
                ELSE 'original'
            END as entry_type
            FROM orders o
            WHERE (DATE(o.order_date) BETWEEN ? AND ?) OR (DATE(o.updated_at) BETWEEN ? AND ? AND o.created_at != o.updated_at)
            ORDER BY o.order_date DESC, o.updated_at DESC
            LIMIT 1000";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $start_date_formatted, $end_date_formatted, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
        // ...existing code...
        if (($row['payment_status'] == 'ຈ່າຍເເລ້ວ' || $row['payment_status'] == 'ຈ່າຍແລ້ວ') && $row['created_at'] != $row['updated_at']) {
            // ...existing code...
        }
    }
    
    // Get other income (no cost, all profit)
    $sql = "SELECT income_id as id, income_date as transaction_date, 
            'other_income' as source, 'ລາຍຮັບອື່ນໆ' as source_type, 
            details, amount, 0 as expense_amount, 0 as cost_amount,
            'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount,
            created_at, created_at as updated_at, 'income' as entry_type
            FROM other_income 
            WHERE DATE(income_date) BETWEEN ? AND ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }
    
    // Get expenses (operating expenses - not product costs)
    $sql = "SELECT expense_id as id, expense_date as transaction_date, 
            'expenses' as source, 'ລາຍຈ່າຍ' as source_type, 
            details, 0 as amount, amount as expense_amount, 0 as cost_amount,
            'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount,
            created_at, created_at as updated_at, 'expense' as entry_type
            FROM expenses 
            WHERE DATE(expense_date) BETWEEN ? AND ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }
    
    // Get purchase orders (inventory purchases - not direct costs until sold)
    $sql = "SELECT po.id, po.order_date as transaction_date,
            'purchase_orders' as source, 'ຊື້ສິນຄ້າ' as source_type,
            CONCAT('ໃບສັ່ງຊື້ #', po.order_code) as details, 0 as amount, po.total_amount as expense_amount, 0 as cost_amount,
            'ຊື້ສິນຄ້າ' as payment_status, po.total_amount as full_amount, 0 as deposit_amount,
            po.created_at, po.created_at as updated_at, 'purchase' as entry_type
            FROM purchase_orders po
            WHERE DATE(po.order_date) BETWEEN ? AND ?";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }
    
    // Sort by created_at/updated_at timestamp (older entries first, newer entries last)
    usort($transactions, function($a, $b) {
        // ສຳລັບລາຍການຊຳລະເງິນ ໃຫ້ໃຊ້ updated_at
        $time_a = ($a['source'] == 'payment_received') ? strtotime($a['updated_at']) : strtotime($a['created_at']);
        $time_b = ($b['source'] == 'payment_received') ? strtotime($b['updated_at']) : strtotime($b['created_at']);
        
        if ($time_a == $time_b) {
            // ຖ້າເວລາເທົ່າກັນ, ຈັດລຽງຕາມ ID
            return $a['id'] <=> $b['id'];
        }
        
        return $time_a - $time_b;
    });
    
    return $transactions;
}

$transactions = getTransactions($conn, $start_date, $end_date);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/home.css">
    <link rel="stylesheet" href="css/summary_income.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@100;300;400;500;700;900&display=swap" rel="stylesheet">
    <title>ສະຫຼຸບລາຍຮັບ-ຈ່າຍ - ຮ້ານ Jຕຸກຂາຍເຄື່ອງໃຊ້ໄຟຟ້າ</title>
    <link rel="icon" type="image/png" href="/test01/favicon.png">

</head> 
<body>
    <?php include_once('home.php'); ?>

    <div class="content-area">
        <div class="summary-container">
            <div class="sticky-header-wrapper">
                <div class="table-header">
                    <h1><i class="fas fa-chart-line"></i> ສະຫຼຸບລາຍຮັບ-ຈ່າຍ</h1>
                </div>
                <!-- Sticky footer for totals -->
                
                
                <div class="filter-container" style="display: flex; align-items: flex-end; justify-content: space-between; gap: 15px;">
                    <form id="filterForm" method="GET" action="" style="display: flex; gap: 15px; align-items: flex-end;">
                        <label for="start_date">ຈາກວັນທີ:</label>
                        <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date ?: date('Y-m-d')); ?>">
                        
                        <label for="end_date">ຮອດວັນທີ:</label>
                        <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date ?: date('Y-m-d')); ?>">

                        <button type="submit"><i class="fas fa-filter"></i> ກັ່ນຕອງ</button>
                        
                        <?php if(!empty($_GET['start_date']) || !empty($_GET['end_date'])): ?>
                            <a href="summary_income" class="btn reset-btn" style="margin-left: 10px; background-color: #f0f2f5; color: #333; padding: 8px 15px; border-radius: 4px; text-decoration: none;">
                                <i class="fas fa-undo"></i> ລ້າງຕົວກອງ
                            </a>
                        <?php endif; ?>
                    </form>
                    <!-- ช่องค้นหาอยู่ขวาสุด -->
                    <input
    type="text"
    id="searchInput"
    placeholder="ຄົ້ນຫາຂໍ້ມູນ..."
    style="
        padding: 10px 16px;
        border: 2px solid #EE4035;
        border-radius: 25px;
        width: 270px;
        font-size: 16px;
        transition: border-color 0.3s, box-shadow 0.3s;
        outline: none;
        box-shadow: 0 2px 8px rgba(238,64,53,0.08);
        background: #fffdfa;
        color: #333;
    "
    onfocus="this.style.borderColor='#d63429'; this.style.boxShadow='0 0 0 2px #ffd6d2';"
    onblur="this.style.borderColor='#EE4035'; this.style.boxShadow='0 2px 8px rgba(238,64,53,0.08)';"
/>
                </div>
                
                <div class="table-header-row" id="tableHeaderRow">
                    <table class="summary-table">
                        <thead>
                            <tr>
                                <th colspan="10" id="dateRangeHeader" style="background:#fffbe7; color:#b85c00; text-align:left; font-weight:normal; font-size:15px;">
                                    <span id="dateRangeText">
                                        <?php if(!empty($start_date) && !empty($end_date)): ?>
                                            ຊ່ວງວັນທີ: <?php echo date('d/m/Y', strtotime($start_date)); ?> - <?php echo date('d/m/Y', strtotime($end_date)); ?>
                                        <?php else: ?>
                                            ທຸກວັນ
                                        <?php endif; ?>
                                    </span>
                                </th>
                            </tr>
                            <tr>
                                <th>ລຳດັບ</th>
                                <th>ວັນທີ</th>
                                <th>ສະຖານະ</th>
                                <th>ແຫຼ່ງທີ່ມາ</th>
                                <th>ລາຍລະອຽດ</th>
                                <th>ລາຍຮັບ</th>
                                <th>ລາຍຈ່າຍ</th>
                                <th>ຕົ້ນທຶນສິນຄ້າ</th>
                                <th>ກຳໄລ/ເເຖວ</th>
                                <th>ກຳໄລທີ່ເເທ້ຈິງ</th>
                            </tr>
                        </thead>
                    </table>
                </div>
                    <div class="table-body-container">
                        <table class="summary-table">
                            <thead style="visibility: hidden; position: absolute;">
                                <tr>
                                    <th>ລຳດັບ</th>
                                    <th>ວັນທີ</th>
                                    <th>ສະຖານະ</th>
                                    <th>ແຫຼ່ງທີ່ມາ</th>
                                    <th>ລາຍລະອຽດ</th>
                                    <th>ລາຍຮັບ</th>
                                    <th>ລາຍຈ່າຍ</th>
                                    <th>ຕົ້ນທຶນສິນຄ້າ</th>
                                    <th>ກຳໄລ/ເເຖວ</th>
                                    <th>ກຳໄລທີ່ເເທ້ຈິງ</th>
                                </tr>
                            </thead>
                        <tbody>
                        </tbody>
                    </table>
                    <button id="loadMoreBtn" style="display:none; margin: 20px auto; padding: 10px 30px; background: #EE4035; color: #fff; border: none; border-radius: 6px; font-size: 16px; cursor: pointer;">
                        <i class="fas fa-plus"></i> ໂຫລດຂໍ້ມູນເພີ່ມເຕິມ
                    </button>
                    <div class="sticky-footer" style="position: sticky; bottom: 0; background: #f5f5f5; z-index: 100; box-shadow: 0 -2px 8px rgba(0,0,0,0.04);">
                    <table class="summary-table">
                        <tfoot>
                            <tr class="total-row">
                                <td colspan="5" id="totalRowLabel">ລວມທັງໝົດ:</td>
                                <td id="totalIncomeCell">0 ₭</td>
                                <td id="totalExpenseCell">0 ₭</td>
                                <td id="totalCostCell">0 ₭</td>
                                <td id="totalProfitCell">0 ₭</td>
                                <td id="totalBalanceCell">0 ₭</td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                </div>

            </div>
        </div>
    </div>
    <script>
// --- Pagination state ---
let currentOffset = 0;
const pageSize = 500;
let isLoading = false;
let isEndOfData = false;
let lastSearchTerm = '';
let lastStartDate = '';
let lastEndDate = '';
// --- ปรับ UI header/footer ให้แสดง "ทุกวัน" เมื่อค้นหา ---
function setAllDateHeaderFooter(isAllDate) {
    // Header
    var dateRangeText = document.getElementById('dateRangeText');
    if (dateRangeText) {
        if (isAllDate) {
            dateRangeText.textContent = 'ທຸກວັນ';
        } else {
            // คืนค่าเดิม (ช่วงวันที่)
            var start = document.getElementById('start_date').value;
            var end = document.getElementById('end_date').value;
            if (start && end) {
                // แปลงเป็น d/m/Y
                var s = start.split('-');
                var e = end.split('-');
                dateRangeText.textContent = 'ຊ່ວງວັນທີ: ' + s[2] + '/' + s[1] + '/' + s[0] + ' - ' + e[2] + '/' + e[1] + '/' + e[0];
            } else {
                dateRangeText.textContent = 'ທຸກວັນ';
            }
        }
    }
    // Footer label (optional, for clarity)
    var totalRowLabel = document.getElementById('totalRowLabel');
    if (totalRowLabel) {
        totalRowLabel.textContent = isAllDate ? 'ລວມທັງໝົດ (ທຸກວັນ):' : 'ລວມທັງໝົດ:';
    }
}
let searchTimeout;

function updateTotals() {
    // เลือกเฉพาะ tbody ของตารางข้อมูล (ไม่ใช่ header/footer)
    var tableBody = document.querySelectorAll('.table-body-container .summary-table tbody')[0];
    if (!tableBody) return;
    var rows = tableBody.querySelectorAll('tr');
    var total_income = 0, total_expense = 0, total_cost = 0, total_profit = 0;
    var calculated_balance = 0;
    var visibleRowCount = 0;
    var validRowCount = 0;
    var lastVisibleBalance = null;

    rows.forEach(function(row, index) {
        if (row.style.display === 'none') return; // ข้ามแถวที่ซ่อน
        var tds = row.querySelectorAll('td');
        var statusText = tds[2] ? tds[2].textContent.trim() : '';
        // ข้ามแถวที่เป็น "ຍັງບໍ່ຈ່າຍ" (pending) ไม่ให้รวมในยอดรวม
        if (statusText.includes('ຍັງບໍ່ຈ່າຍ')) return;
        visibleRowCount++;
        var incomeText = tds[5] ? tds[5].textContent.trim() : '';
        var expenseText = tds[6] ? tds[6].textContent.trim() : '';
        var costText = tds[7] ? tds[7].textContent.trim() : '';
        var profitText = tds[8] ? tds[8].textContent.trim() : '';
        var balanceText = tds[9] ? tds[9].textContent.trim() : '';

        // แปลงเป็นตัวเลข (ข้าม "-" และข้อความอื่นๆ)
        var income = (incomeText !== '-' && incomeText !== '') ? parseFloat(incomeText.replace(/[^0-9.-]+/g,"")) || 0 : 0;
        var expense = (expenseText !== '-' && expenseText !== '') ? parseFloat(expenseText.replace(/[^0-9.-]+/g,"")) || 0 : 0;
        var cost = (costText !== '-' && costText !== '') ? parseFloat(costText.replace(/[^0-9.-]+/g,"")) || 0 : 0;
        var profit = (profitText !== '-' && profitText !== '') ? parseFloat(profitText.replace(/[^0-9.-]+/g,"")) || 0 : 0;
        var balance = (balanceText !== '-' && balanceText !== '') ? parseFloat(balanceText.replace(/[^0-9.-]+/g,"")) || 0 : 0;

        // ตรวจสอบสถานะเหมือนกับ PHP - เฉพาะรายการที่ได้รับเงินจริงและไม่ยกเลิก
        var shouldCount = false;
        if (statusText.includes('ຊຳລະເງິນ') || // ชำระเงิน
            statusText.includes('ລາຍຮັບ') ||   // รายรับ
            statusText.includes('ມັດຈຳ')) {     // มัดจำ
            shouldCount = true;
        }
        // ไม่นับรายการที่ยกเลิก
        if (statusText.includes('ຍົກເລີກ')) {  // ยกเลิก
            shouldCount = false;
        }
        // รวมค่าเฉพาะรายการที่ควรนับ
        if (shouldCount) {
            total_income += income;
            total_cost += cost;
            total_profit += profit;
            validRowCount++;
        }
        // รายจ่ายและยอดเงินนับทุกรายการ (ยกเว้น pending)
        total_expense += expense;
        calculated_balance += (income - expense); // ไม่หักต้นทุน
        // เก็บ balance ของแถวสุดท้ายที่แสดง
        lastVisibleBalance = balanceText;
    });

    // อัปเดตแถวสรุปรวม
    var totalRow = document.querySelector('.sticky-footer .total-row');
    if (totalRow) {
        var totalCells = totalRow.querySelectorAll('td');
        if (totalCells.length >= 6) {
            // totalCells[0] = "ລວມທັງໝົດ:" (colspan=5)
            // totalCells[1] = รายรับ
            // totalCells[2] = รายจ่าย
            // totalCells[3] = ต้นทุน
            // totalCells[4] = กำไร
            // totalCells[5] = ยอดเงิน (ใหม่: ให้ตรงกับแถวสุดท้ายที่แสดง)
            totalCells[1].textContent = numberWithCommas(total_income) + " ₭"; // รายรับ
            totalCells[2].textContent = numberWithCommas(total_expense) + " ₭"; // รายจ่าย
            totalCells[3].textContent = numberWithCommas(total_cost) + " ₭"; // ต้นทุน
            totalCells[4].textContent = numberWithCommas(total_profit) + " ₭"; // กำไร
            // ถ้าไม่มีแถวแสดง ให้แสดง 0
            if (lastVisibleBalance !== null && lastVisibleBalance !== undefined && lastVisibleBalance !== "") {
                totalCells[5].textContent = lastVisibleBalance;
            } else {
                totalCells[5].textContent = "0 ₭";
            }
        }
    }
}

function numberWithCommas(x) {
    // แสดงเฉพาะทศนิยมเมื่อมีเศษ
    if (typeof x !== 'number') x = parseFloat(x);
    if (isNaN(x)) return '';
    let parts = x.toLocaleString('en-US', { minimumFractionDigits: 0, maximumFractionDigits: 2 }).split('.');
    if (parts.length === 2 && parseInt(parts[1]) !== 0) {
        // มีเศษ
        return x.toLocaleString('en-US', { minimumFractionDigits: 0, maximumFractionDigits: 2 });
    } else {
        // ไม่มีเศษ
        return x.toLocaleString('en-US', { maximumFractionDigits: 0 });
    }
}

function renderTransactionRow(transaction, index, isSearchMode = false) {
    // หากเป็นโหมดค้นหา ไม่ต้องตรวจสอบวันที่
    if (!isSearchMode) {
        const startDate = document.getElementById('start_date').value;
        const endDate = document.getElementById('end_date').value;
        const startDateFormatted = new Date(startDate).toISOString().split('T')[0];
        const endDateFormatted = new Date(endDate).toISOString().split('T')[0];
        
        // ตรวจสอบว่ารายการอยู่ในช่วงวันที่
        const transactionDate = new Date(transaction.transaction_date).toISOString().split('T')[0];
        let isInDateRange = false;
        
        if (transaction.source === 'payment_received') {
            const paymentDate = new Date(transaction.updated_at).toISOString().split('T')[0];
            isInDateRange = (paymentDate >= startDateFormatted && paymentDate <= endDateFormatted);
        } else {
            isInDateRange = (transactionDate >= startDateFormatted && transactionDate <= endDateFormatted);
        }
        
        if (!isInDateRange) return '';
    }
    
    // กำหนด CSS class แถว
    let rowClass = '';
    if (transaction.source === 'payment_received') {
        rowClass = 'payment-row';
    } else if (transaction.entry_type === 'deposit') {
        rowClass = 'deposit-row';
    } else if (transaction.payment_status === 'ຍົກເລີກ') {
        rowClass = 'cancelled-row';
    }
    
    // กำหนดสถานะ
    let status, statusClass, badgeClass;
    if (transaction.source === 'payment_received') {
        status = 'ຊຳລະເງິນ';
        statusClass = 'income';
        badgeClass = 'payment';
    } else if (transaction.source === 'expenses' || transaction.source === 'purchase_orders') {
        status = transaction.source === 'purchase_orders' ? 'ຊື້ສິນຄ້າ' : 'ລາຍຈ່າຍ';
        statusClass = 'expense';
        badgeClass = 'pending';
    } else if (transaction.source === 'pending' || transaction.payment_status === 'ຍັງບໍ່ຈ່າຍ') {
        status = 'ຍັງບໍ່ຈ່າຍ';
        statusClass = 'pending';
        badgeClass = 'pending';
    } else if (transaction.source === 'deposit' || transaction.payment_status === 'ມັດຈຳ') {
        status = 'ມັດຈຳ';
        statusClass = 'deposit';
        badgeClass = 'deposit';
    } else if (transaction.payment_status === 'ຍົກເລີກ') {
        status = 'ຍົກເລີກ';
        statusClass = 'cancelled';
        badgeClass = 'cancelled';
    } else {
        status = 'ລາຍຮັບ';
        statusClass = 'income';
        badgeClass = 'paid';
    }
    
    // คำนวณค่าต่างๆ
    const incomeAmount = parseFloat(transaction.amount) || 0;
    const expenseAmount = parseFloat(transaction.expense_amount) || 0;
    const costAmount = parseFloat(transaction.cost_amount) || 0;
    const profit = incomeAmount - costAmount;
    
    // แสดงวันที่
    let dateDisplay;
    if (transaction.source === 'payment_received') {
        const date = new Date(transaction.updated_at);
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        dateDisplay = `${day}-${month}-${year} <span class='payment-date'>(ວັນທີຊຳລະ)</span>`;
    } else {
        const date = new Date(transaction.transaction_date);
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        dateDisplay = `${day}-${month}-${year}`;
    }
    
    // If pending, show '-' for all columns (income, expense, cost, profit, real profit)
    if (transaction.payment_status === 'ຍັງບໍ່ຈ່າຍ' || transaction.source === 'pending') {
        return `
            <tr class="${rowClass}">
                <td>${index}</td>
                <td>${dateDisplay}</td>
                <td class="${statusClass}">${status} <span class="status-badge ${badgeClass}">${status}</span></td>
                <td>${transaction.source_type}</td>
                <td>${transaction.details}</td>
                <td class="text-right">-</td>
                <td class="text-right">-</td>
                <td class="text-right">-</td>
                <td class="text-right profit">-</td>
                <td class="text-right balance-cell">-</td>
            </tr>
        `;
    }
    return `
        <tr class="${rowClass}">
            <td>${index}</td>
            <td>${dateDisplay}</td>
            <td class="${statusClass}">${status} <span class="status-badge ${badgeClass}">${status}</span></td>
            <td>${transaction.source_type}</td>
            <td>${transaction.details}</td>
            <td class="text-right">${incomeAmount > 0 ? numberWithCommas(incomeAmount) + ' ₭' : '-'}</td>
            <td class="text-right">${expenseAmount > 0 ? numberWithCommas(expenseAmount) + ' ₭' : '-'}</td>
            <td class="text-right">${costAmount > 0 ? numberWithCommas(costAmount) + ' ₭' : '-'}</td>
            <td class="text-right profit">${incomeAmount > 0 ? numberWithCommas(profit) + ' ₭' : '-'}</td>
            <td class="text-right balance-cell">0 ₭</td>
        </tr>
    `;
}

function performSearch(searchTerm, isLoadMore = false) {
    // ถ้าไม่ใช่ load more ให้ reset offset
    if (!isLoadMore) {
        currentOffset = 0;
        isEndOfData = false;
    }
    isLoading = true;
    lastSearchTerm = searchTerm;
    // --- Logic: ถ้ามี searchTerm ให้ค้นหาทุกวัน (ส่งค่าว่าง)
    let start = '';
    let end = '';
    if (!searchTerm) {
        // ถ้าไม่มี searchTerm ให้ใช้วันที่จาก input
        start = document.getElementById('start_date').value;
        end = document.getElementById('end_date').value;
    }
    lastStartDate = start;
    lastEndDate = end;
    // แสดง loading
    const tableBody = document.querySelector('.table-body-container .summary-table tbody');
    if (tableBody && !isLoadMore) {
        tableBody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 20px;">ກຳລັງຄົ້ນຫາ...</td></tr>';
    }
    // --- ปรับ header/footer เป็น "ทุกวัน" ถ้ามี searchTerm ---
    setAllDateHeaderFooter(!!searchTerm);
    // ส่งคำขอ AJAX
    const formData = new FormData();
    formData.append('search_term', searchTerm);
    formData.append('start_date', start);
    formData.append('end_date', end);
    formData.append('offset', currentOffset);
    formData.append('limit', pageSize);
    fetch('search_transactions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        isLoading = false;
        if (data.error) {
            console.error('Search error:', data.error);
            return;
        }
        // อัปเดตตาราง
        if (tableBody && data.transactions) {
            let html = '';
            let balance = 0;
            let startIndex = isLoadMore ? tableBody.querySelectorAll('tr').length : 0;
            data.transactions.forEach((transaction, index) => {
                const isPending = transaction.payment_status === 'ຍັງບໍ່ຈ່າຍ' || transaction.source === 'pending';
                const isCancelled = transaction.payment_status === 'ຍົກເລີກ';
                const incomeAmount = parseFloat(transaction.amount) || 0;
                const expenseAmount = parseFloat(transaction.expense_amount) || 0;
                let showBalance = false;
                if (!isPending && !isCancelled && ((transaction.payment_status !== 'ຍັງບໍ່ຈ່າຍ' && transaction.source !== 'pending') || transaction.source === 'payment_received')) {
                    balance += (incomeAmount - expenseAmount);
                    showBalance = true;
                }
                let rowHtml = renderTransactionRow(transaction, startIndex + index + 1, true);
                if (rowHtml) {
                    if (showBalance) {
                        rowHtml = rowHtml.replace('<td class="text-right balance-cell">0 ₭</td>', '<td class="text-right balance-cell">' + numberWithCommas(balance) + ' ₭</td>');
                    }
                    html += rowHtml;
                }
            });
            if (!isLoadMore) {
                tableBody.innerHTML = html || '<tr><td colspan="10" style="text-align: center; padding: 20px;">ບໍ່ພົບຂໍ້ມູນທີ່ຄົ້ນຫາ</td></tr>';
            } else {
                tableBody.insertAdjacentHTML('beforeend', html);
            }
            // อัปเดตยอดรวม (total) จาก server
            if (data.totals) {
                document.getElementById('totalIncomeCell').textContent = numberWithCommas(data.totals.total_income) + ' ₭';
                document.getElementById('totalExpenseCell').textContent = numberWithCommas(data.totals.total_expense) + ' ₭';
                document.getElementById('totalCostCell').textContent = numberWithCommas(data.totals.total_cost) + ' ₭';
                document.getElementById('totalProfitCell').textContent = numberWithCommas(data.totals.total_profit) + ' ₭';
                document.getElementById('totalBalanceCell').textContent = numberWithCommas(data.totals.total_income - data.totals.total_expense) + ' ₭';
            }
            // ถ้าจำนวนที่ได้ < pageSize แสดงว่าไม่มีข้อมูลเพิ่ม
            var loadMoreBtn = document.getElementById('loadMoreBtn');
            if (loadMoreBtn) {
                if (data.transactions.length < pageSize) {
                    isEndOfData = true;
                    loadMoreBtn.style.display = 'none';
                } else {
                    loadMoreBtn.style.display = 'inline-block';
                }
            }
        }
    })
    .catch(error => {
        isLoading = false;
        console.error('Search failed:', error);
        if (tableBody && !isLoadMore) {
            tableBody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 20px; color: red;">เกิดข้อผิดพลาดในการค้นหา</td></tr>';
        }
    });
}

// โหลดเพิ่มข้อมูล (load more)
function loadMoreTransactions() {
    if (isLoading || isEndOfData) return;
    currentOffset += pageSize;
    performSearch(lastSearchTerm, true);
}


// เมื่อค้นหา
document.getElementById('searchInput').addEventListener('keyup', function() {
    const filter = this.value.toLowerCase().trim();
    if (searchTimeout) clearTimeout(searchTimeout);
    searchTimeout = setTimeout(function() {
        // trigger AJAX search (all dates, paginated)
        performSearch(filter);
    }, 300);
});

// เมื่อกรองวันที่ (submit form)
document.getElementById('filterForm').addEventListener('submit', function(e) {
    e.preventDefault();
    // ถ้า searchInput มีค่า ให้ค้นหาตามนั้น (AJAX)
    const searchVal = document.getElementById('searchInput').value.trim();
    performSearch(searchVal);
});

// ปุ่มโหลดเพิ่ม
document.addEventListener('DOMContentLoaded', function() {
    performSearch('');
    const loadMoreBtn = document.getElementById('loadMoreBtn');
    if (loadMoreBtn) {
        loadMoreBtn.addEventListener('click', function() {
            loadMoreTransactions();
        });
    }
    // ...existing code...
});
    </script>
</body>
</html>
























































