<?php
require_once("auth.php");
require_once("db_con.php");

header('Content-Type: application/json');


if (!isset($_POST['search_term'])) {
    echo json_encode(['error' => 'No search term provided']);
    exit;
}


$search_term = trim($_POST['search_term']);
$start_date = isset($_POST['start_date']) ? $_POST['start_date'] : '';
$end_date = isset($_POST['end_date']) ? $_POST['end_date'] : '';
$offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
$limit = isset($_POST['limit']) ? intval($_POST['limit']) : 500;
if ($limit < 1 || $limit > 500) $limit = 500;
if ($offset < 0) $offset = 0;

if (empty($search_term)) {
    // ถ้าไม่มีคำค้นหา ให้ส่งข้อมูลทั้งหมดในช่วงวันที่
    $transactions = getTransactions($conn, $start_date, $end_date, $offset, $limit);
    $is_search = false;
} else {
    // ค้นหาในฐานข้อมูล
    $transactions = searchTransactions($conn, $search_term, $start_date, $end_date, $offset, $limit);
    $is_search = true;
}

// --- Calculate totals ---



// ถ้า start_date หรือ end_date ว่าง ให้ไม่ filter วันที่ (ใช้ช่วงกว้างสุด)
if (empty($start_date)) {
    $start_date_formatted = '2000-01-01';
} else {
    $start_date_formatted = date('Y-m-d', strtotime($start_date));
}
if (empty($end_date)) {
    $end_date_formatted = '2099-12-31';
} else {
    $end_date_formatted = date('Y-m-d', strtotime($end_date));
}

if (!$is_search) {
    // --- เดิม: กรณีไม่มี search term ---
    // ...existing code for totals (เหมือนเดิม)...
    $sql = "SELECT
        SUM(CASE
            WHEN (o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ') THEN (o.subtotal - COALESCE(o.discount, 0))
            WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
            ELSE 0
        END) as total_income,
        SUM(CASE
            WHEN (o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ') THEN (
                SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
            )
            WHEN o.payment_status = 'ມັດຈຳ' THEN (
                (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (
                    SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
                )
            )
            ELSE 0
        END) as total_cost
    FROM orders o
    WHERE ((DATE(o.order_date) BETWEEN ? AND ?) OR (DATE(o.updated_at) BETWEEN ? AND ? AND o.created_at != o.updated_at))
    AND o.payment_status != 'ຍັງບໍ່ຈ່າຍ' AND o.payment_status != 'ຍົກເລີກ'";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $start_date_formatted, $end_date_formatted, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_income = floatval($row['total_income'] ?? 0);
    $total_cost = floatval($row['total_cost'] ?? 0);

    // Add other_income
    $sql = "SELECT SUM(amount) as total_income FROM other_income WHERE DATE(income_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_income += floatval($row['total_income'] ?? 0);

    // Total expense (expenses + purchase_orders)
    $sql = "SELECT SUM(amount) as total_expense FROM expenses WHERE DATE(expense_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_expense = floatval($row['total_expense'] ?? 0);

    $sql = "SELECT SUM(total_amount) as total_expense FROM purchase_orders WHERE DATE(order_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_expense += floatval($row['total_expense'] ?? 0);

    $total_profit = $total_income - $total_cost;

    $totals = [
        'total_income' => $total_income,
        'total_expense' => $total_expense,
        'total_cost' => $total_cost,
        'total_profit' => $total_profit
    ];
} else {
    // --- ใหม่: กรณีมี search term ---
    $search_pattern = '%' . $search_term . '%';
    // Orders
    $sql = "SELECT
        SUM(CASE
            WHEN (o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ') THEN (o.subtotal - COALESCE(o.discount, 0))
            WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
            ELSE 0
        END) as total_income,
        SUM(CASE
            WHEN (o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ') THEN (
                SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
            )
            WHEN o.payment_status = 'ມັດຈຳ' THEN (
                (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (
                    SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id
                )
            )
            ELSE 0
        END) as total_cost
    FROM orders o
    WHERE ((CONCAT('Order #', o.order_id) LIKE ? OR o.payment_status LIKE ? OR o.order_id LIKE ?)
        AND ((DATE(o.order_date) BETWEEN ? AND ?) OR (DATE(o.updated_at) BETWEEN ? AND ? AND o.created_at != o.updated_at))
        AND o.payment_status != 'ຍັງບໍ່ຈ່າຍ' AND o.payment_status != 'ຍົກເລີກ')";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sssssss", $search_pattern, $search_pattern, $search_pattern, $start_date_formatted, $end_date_formatted, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_income = floatval($row['total_income'] ?? 0);
    $total_cost = floatval($row['total_cost'] ?? 0);

    // Other income
    $sql = "SELECT SUM(amount) as total_income FROM other_income WHERE details LIKE ? AND DATE(income_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sss", $search_pattern, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_income += floatval($row['total_income'] ?? 0);

    // Expenses
    $sql = "SELECT SUM(amount) as total_expense FROM expenses WHERE details LIKE ? AND DATE(expense_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sss", $search_pattern, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_expense = floatval($row['total_expense'] ?? 0);

    // Purchase orders
    $sql = "SELECT SUM(total_amount) as total_expense FROM purchase_orders WHERE (CONCAT('ໃບສັ່ງຊື້ #', order_code) LIKE ? OR order_code LIKE ?) AND DATE(order_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $search_pattern, $search_pattern, $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $total_expense += floatval($row['total_expense'] ?? 0);

    $total_profit = $total_income - $total_cost;

    $totals = [
        'total_income' => $total_income,
        'total_expense' => $total_expense,
        'total_cost' => $total_cost,
        'total_profit' => $total_profit
    ];
}

echo json_encode(['transactions' => $transactions, 'totals' => $totals]);

function searchTransactions($conn, $search_term, $start_date, $end_date, $offset = 0, $limit = 500) {
    $transactions = [];
    $search_pattern = '%' . $search_term . '%';
    // ถ้า start_date หรือ end_date ว่าง ให้ไม่ filter วันที่ (ใช้ช่วงกว้างสุด)
    if (empty($start_date)) {
        $start_date_formatted = '2000-01-01';
    } else {
        $start_date_formatted = date('Y-m-d', strtotime($start_date));
    }
    if (empty($end_date)) {
        $end_date_formatted = '2099-12-31';
    } else {
        $end_date_formatted = date('Y-m-d', strtotime($end_date));
    }


    // Search in orders - filter by date (รองรับทุกวัน)
    if ($start_date == '' && $end_date == '') {
        $sql = "SELECT o.order_id as id, o.order_date as transaction_date, 
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'orders'
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຍັງບໍ່ຈ່າຍ' THEN 'pending'
                ELSE 'other'
            END as source,
            'POS' as source_type, 
            CONCAT('Order #', o.order_id) as details,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN o.deposit_amount
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (o.subtotal - COALESCE(o.discount, 0))
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
                ELSE (o.subtotal - COALESCE(o.discount, 0))
            END as amount,
            0 as expense_amount,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 
                            (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 
                    (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                ELSE 0
            END as cost_amount,
            o.payment_status,
            o.total_amount as full_amount,
            o.subtotal,
            COALESCE(o.discount, 0) as discount,
            o.deposit_amount,
            o.created_at,
            o.updated_at,
            CASE
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'original'
                    END
                ELSE 'original'
            END as entry_type
            FROM orders o
            WHERE (CONCAT('Order #', o.order_id) LIKE ? OR o.payment_status LIKE ? OR o.order_id LIKE ?)
            LIMIT ? OFFSET ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("sssii", $search_pattern, $search_pattern, $search_pattern, $limit, $offset);
        $stmt->execute();
        $result = $stmt->get_result();
    } else {
        $sql = "SELECT o.order_id as id, o.order_date as transaction_date, 
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'orders'
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຍັງບໍ່ຈ່າຍ' THEN 'pending'
                ELSE 'other'
            END as source,
            'POS' as source_type, 
            CONCAT('Order #', o.order_id) as details,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN o.deposit_amount
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (o.subtotal - COALESCE(o.discount, 0))
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
                ELSE (o.subtotal - COALESCE(o.discount, 0))
            END as amount,
            0 as expense_amount,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 
                            (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 
                    (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                ELSE 0
            END as cost_amount,
            o.payment_status,
            o.total_amount as full_amount,
            o.subtotal,
            COALESCE(o.discount, 0) as discount,
            o.deposit_amount,
            o.created_at,
            o.updated_at,
            CASE
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'original'
                    END
                ELSE 'original'
            END as entry_type
            FROM orders o
            WHERE ((CONCAT('Order #', o.order_id) LIKE ? OR o.payment_status LIKE ? OR o.order_id LIKE ?)
                AND ((DATE(o.order_date) BETWEEN ? AND ?) OR (DATE(o.updated_at) BETWEEN ? AND ? AND o.created_at != o.updated_at)))
            LIMIT ? OFFSET ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssssssssi", $search_pattern, $search_pattern, $search_pattern, $start_date_formatted, $end_date_formatted, $start_date_formatted, $end_date_formatted, $limit, $offset);
        $stmt->execute();
        $result = $stmt->get_result();
    }

    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;

        // Add payment received entries if applicable
        if (($row['payment_status'] == 'ຈ່າຍເເລ້ວ' || $row['payment_status'] == 'ຈ່າຍແລ້ວ') && 
            $row['created_at'] != $row['updated_at']) {

            // Get cost data
            $cost_sql = "SELECT SUM(oi.quantity * p.cost_price) as total_cost 
                         FROM order_items oi 
                         JOIN products p ON oi.product_id = p.product_id 
                         WHERE oi.order_id = ?";
            $cost_stmt = $conn->prepare($cost_sql);
            $cost_stmt->bind_param("s", $row['id']);
            $cost_stmt->execute();
            $cost_result = $cost_stmt->get_result();
            $cost_data = $cost_result->fetch_assoc();
            $total_cost = $cost_data['total_cost'] ?? 0;

            $discount = $row['discount'] ?? 0;
            $deposit_amount = $row['deposit_amount'] ?? 0;
            $subtotal = $row['subtotal'] ?? $row['full_amount'];

            $net_after_discount = $subtotal - $discount;
            $remaining_amount = $net_after_discount - $deposit_amount;
            $payment_amount = ($deposit_amount > 0) ? $remaining_amount : $net_after_discount;
            $payment_cost = ($deposit_amount > 0 && $net_after_discount > 0) ? 
                            (($remaining_amount / $net_after_discount) * $total_cost) : 
                            $total_cost;

            $payment_date = date('Y-m-d', strtotime($row['updated_at']));
            if ($payment_date >= $start_date_formatted && $payment_date <= $end_date_formatted && $payment_amount > 0) {
                $payment_entry = [
                    'id' => $row['id'],
                    'transaction_date' => $row['updated_at'],
                    'source' => 'payment_received',
                    'source_type' => 'ຊຳລະເງິນ',
                    'details' => ($deposit_amount > 0) ? 
                        'ຊຳລະສ່ວນທີ່ເຫຼືອ Order #' . $row['id'] . ' (' . number_format($payment_amount, 0) . '₭)' :
                        'ຊຳລະເງິນ Order #' . $row['id'],
                    'amount' => $payment_amount,
                    'expense_amount' => 0,
                    'cost_amount' => $payment_cost,
                    'payment_status' => 'ຈ່າຍແລ້ວ',
                    'full_amount' => $row['full_amount'],
                    'deposit_amount' => $deposit_amount,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'entry_type' => 'payment'
                ];
                $transactions[] = $payment_entry;
            }
        }
    }

    // Search in other_income - filter by date
    $sql = "SELECT income_id as id, income_date as transaction_date, 
            'other_income' as source, 'ລາຍຮັບອື່ນໆ' as source_type, 
            details, amount, 0 as expense_amount, 0 as cost_amount,
            'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount,
            created_at, created_at as updated_at, 'income' as entry_type
            FROM other_income 
            WHERE details LIKE ? AND DATE(income_date) BETWEEN ? AND ?
            LIMIT ? OFFSET ?";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sssii", $search_pattern, $start_date_formatted, $end_date_formatted, $limit, $offset);
    $stmt->execute();
    $result = $stmt->get_result();

    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }

    // Search in expenses - filter by date
    $sql = "SELECT expense_id as id, expense_date as transaction_date, 
            'expenses' as source, 'ລາຍຈ່າຍ' as source_type, 
            details, 0 as amount, amount as expense_amount, 0 as cost_amount,
            'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount,
            created_at, created_at as updated_at, 'expense' as entry_type
            FROM expenses 
            WHERE details LIKE ? AND DATE(expense_date) BETWEEN ? AND ?
            LIMIT ? OFFSET ?";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sssii", $search_pattern, $start_date_formatted, $end_date_formatted, $limit, $offset);
    $stmt->execute();
    $result = $stmt->get_result();

    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }

    // Search in purchase orders - filter by date
    $sql = "SELECT po.id, po.order_date as transaction_date,
            'expenses' as source, 'ໃບສັ່ງຊື້ສິນຄ້າ' as source_type,
            CONCAT('ໃບສັ່ງຊື້ #', po.order_code) as details, 0 as amount, po.total_amount as expense_amount, 0 as cost_amount,
            'ລາຍຈ່າຍ' as payment_status, po.total_amount as full_amount, 0 as deposit_amount,
            po.created_at, po.created_at as updated_at, 'expense' as entry_type
            FROM purchase_orders po
            WHERE ((CONCAT('ໃບສັ່ງຊື້ #', po.order_code) LIKE ? OR po.order_code LIKE ?) AND DATE(po.order_date) BETWEEN ? AND ?)
            LIMIT ? OFFSET ?";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssssii", $search_pattern, $search_pattern, $start_date_formatted, $end_date_formatted, $limit, $offset);
    $stmt->execute();
    $result = $stmt->get_result();

    while($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }

    // Sort by created_at/updated_at timestamp
    usort($transactions, function($a, $b) {
        $time_a = ($a['source'] == 'payment_received') ? strtotime($a['updated_at']) : strtotime($a['created_at']);
        $time_b = ($b['source'] == 'payment_received') ? strtotime($b['updated_at']) : strtotime($b['created_at']);

        if ($time_a == $time_b) {
            return $a['id'] <=> $b['id'];
        }

        return $time_a - $time_b;
    });

    return $transactions;
}

// Include the original getTransactions function for when search is empty
function getTransactions($conn, $start_date, $end_date, $offset = 0, $limit = 500) {
    $transactions = [];
    
    // Convert dates to ensure proper format for SQL comparison
    // ถ้า start_date หรือ end_date ว่าง ให้ไม่ filter วันที่ (ใช้ช่วงกว้างสุด)
    if (empty($start_date)) {
        $start_date_formatted = '2000-01-01';
    } else {
        $start_date_formatted = date('Y-m-d', strtotime($start_date));
    }
    if (empty($end_date)) {
        $end_date_formatted = '2099-12-31';
    } else {
        $end_date_formatted = date('Y-m-d', strtotime($end_date));
    }
    
    // รวมข้อมูลจากทุกตาราง แล้วค่อย sort และตัดเหลือ $limit ตาม offset
    $all = [];

    // Orders
    $sql = "SELECT o.order_id as id, o.order_date as transaction_date, 
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'orders'
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຍັງບໍ່ຈ່າຍ' THEN 'pending'
                ELSE 'other'
            END as source,
            'POS' as source_type, 
            CONCAT('Order #', o.order_id) as details,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN o.deposit_amount
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (o.subtotal - COALESCE(o.discount, 0))
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
                ELSE (o.subtotal - COALESCE(o.discount, 0))
            END as amount,
            0 as expense_amount,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN 
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 
                            (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                        WHEN o.created_at != o.updated_at THEN 0
                        ELSE (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                    END
                WHEN o.payment_status = 'ມັດຈຳ' THEN 
                    (o.deposit_amount / (o.subtotal - COALESCE(o.discount, 0))) * (SELECT SUM(oi.quantity * p.cost_price) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = o.order_id)
                ELSE 0
            END as cost_amount,
            o.payment_status,
            o.total_amount as full_amount,
            o.subtotal,
            COALESCE(o.discount, 0) as discount,
            o.deposit_amount,
            o.created_at,
            o.updated_at,
            CASE
                WHEN o.payment_status = 'ມັດຈຳ' THEN 'deposit'
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' OR o.payment_status = 'ຈ່າຍແລ້ວ' THEN
                    CASE 
                        WHEN o.created_at != o.updated_at AND o.deposit_amount > 0 THEN 'deposit'
                        ELSE 'original'
                    END
                ELSE 'original'
            END as entry_type
            FROM orders o
            WHERE DATE(o.order_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $all[] = $row;
        if (($row['payment_status'] == 'ຈ່າຍເເລ້ວ' || $row['payment_status'] == 'ຈ່າຍແລ້ວ') && $row['created_at'] != $row['updated_at']) {
            $cost_sql = "SELECT SUM(oi.quantity * p.cost_price) as total_cost FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE oi.order_id = ?";
            $cost_stmt = $conn->prepare($cost_sql);
            $cost_stmt->bind_param("s", $row['id']);
            $cost_stmt->execute();
            $cost_result = $cost_stmt->get_result();
            $cost_data = $cost_result->fetch_assoc();
            $total_cost = $cost_data['total_cost'] ?? 0;
            $discount = $row['discount'] ?? 0;
            $deposit_amount = $row['deposit_amount'] ?? 0;
            $subtotal = $row['subtotal'] ?? $row['full_amount'];
            $net_after_discount = $subtotal - $discount;
            $remaining_amount = $net_after_discount - $deposit_amount;
            $payment_amount = ($deposit_amount > 0) ? $remaining_amount : $net_after_discount;
            $payment_cost = ($deposit_amount > 0 && $net_after_discount > 0) ? (($remaining_amount / $net_after_discount) * $total_cost) : $total_cost;
            $payment_date = date('Y-m-d', strtotime($row['updated_at']));
            if ($payment_date >= $start_date_formatted && $payment_date <= $end_date_formatted && $payment_amount > 0) {
                $payment_entry = [
                    'id' => $row['id'],
                    'transaction_date' => $row['updated_at'],
                    'source' => 'payment_received',
                    'source_type' => 'ຊຳລະເງິນ',
                    'details' => ($deposit_amount > 0) ? 'ຊຳລະສ່ວນທີ່ເຫຼືອ Order #' . $row['id'] . ' (' . number_format($payment_amount, 0) . '₭)' : 'ຊຳລະເງິນ Order #' . $row['id'],
                    'amount' => $payment_amount,
                    'expense_amount' => 0,
                    'cost_amount' => $payment_cost,
                    'payment_status' => 'ຈ່າຍແລ້ວ',
                    'full_amount' => $row['full_amount'],
                    'deposit_amount' => $deposit_amount,
                    'created_at' => $row['created_at'],
                    'updated_at' => $row['updated_at'],
                    'entry_type' => 'payment'
                ];
                $all[] = $payment_entry;
            }
        }
    }

    // Other income
    $sql = "SELECT income_id as id, income_date as transaction_date, 'other_income' as source, 'ລາຍຮັບອື່ນໆ' as source_type, details, amount, 0 as expense_amount, 0 as cost_amount, 'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount, created_at, created_at as updated_at, 'income' as entry_type FROM other_income WHERE DATE(income_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $all[] = $row;
    }

    // Expenses
    $sql = "SELECT expense_id as id, expense_date as transaction_date, 'expenses' as source, 'ລາຍຈ່າຍ' as source_type, details, 0 as amount, amount as expense_amount, 0 as cost_amount, 'ຈ່າຍແລ້ວ' as payment_status, amount as full_amount, 0 as deposit_amount, created_at, created_at as updated_at, 'expense' as entry_type FROM expenses WHERE DATE(expense_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $all[] = $row;
    }

    // Purchase orders
    $sql = "SELECT po.id, po.order_date as transaction_date, 'expenses' as source, 'ໃບສັ່ງຊື້ສິນຄ້າ' as source_type, CONCAT('ໃບສັ່ງຊື້ #', po.order_code) as details, 0 as amount, po.total_amount as expense_amount, 0 as cost_amount, 'ລາຍຈ່າຍ' as payment_status, po.total_amount as full_amount, 0 as deposit_amount, po.created_at, po.created_at as updated_at, 'expense' as entry_type FROM purchase_orders po WHERE DATE(po.order_date) BETWEEN ? AND ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date_formatted, $end_date_formatted);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $all[] = $row;
    }

    // Sort all, then slice for pagination
    usort($all, function($a, $b) {
        $time_a = ($a['source'] == 'payment_received') ? strtotime($a['updated_at']) : strtotime($a['created_at']);
        $time_b = ($b['source'] == 'payment_received') ? strtotime($b['updated_at']) : strtotime($b['created_at']);
        if ($time_a == $time_b) {
            return $a['id'] <=> $b['id'];
        }
        return $time_a - $time_b;
    });

    // Apply offset/limit after merge
    $transactions = array_slice($all, $offset, $limit);
    return $transactions;
}
?>
