<?php
require_once("db_con.php");

// เล่มการตั้งค่า timezone
date_default_timezone_set('Asia/Bangkok'); // timezone ที่ต้องการ

// ข้อมูล JSON จาก request
$json_data = file_get_contents('php://input');
$order_data = json_decode($json_data, true);

// ตรวจสอบว่าข้อมูลต้องไม่
if (!$order_data) {
    echo json_encode(['status' => 'error', 'message' => 'ຂໍ້ມູນບໍ່ຖືກຕ້ອງ']);
    exit;
}

// เล่ม transaction
$conn->begin_transaction();

try {
    // สร้าง order_id โดยใช้�んก์�ん generateOrderID แทนการ lemma>�ん่ม
    $order_id = generateOrderID($conn);
    
    // ข้อมูล=<ค้า
    $customer_name = $order_data['customer']['name'];
    $customer_phone = $order_data['customer']['phone'];
    $customer_address = $order_data['customer']['address'];
    
    // ตรวจสอบและ lemma>ข้อมูล=<ค้า
    $customer_id = saveCustomer($conn, $customer_name, $customer_phone, $customer_address);
    
    // ข้อมูลการส่งซื้อ
    $order_date = date('Y-m-d H:i:s'); // เปลี่ยนเป็น
    $shipping_company = $order_data['shipping_company'];
    $store_id = $order_data['store_id']; // ค่า store_id
    $payment_status = $order_data['payment_status'];
    $payment_method = $order_data['payment_method'];
    $cod_type = $order_data['cod_type'];
    $deposit_amount = $order_data['deposit_amount'];
    $discount_amount = $order_data['discount_amount'];
    $subtotal = $order_data['summary']['subtotal'];
    $total_amount = $order_data['summary']['total_amount'];
    
    // รับเงินสดและเงินทอน (เฉพาะหน้าร้าน)
    $cash_received = null;
    $cash_change = null;
    if (isset($order_data['cash_received'])) {
        $cash_received = $order_data['cash_received'];
    }
    if (isset($order_data['cash_change'])) {
        $cash_change = $order_data['cash_change'];
    }
    // ประเภทการขาย (online / pos)
    $sale_type = isset($order_data['sale_type']) ? $order_data['sale_type'] : null;
    
    // ข้อมูลการส่งซื้อ
    $sql = "INSERT INTO orders (order_id, customer_id, order_date, shipping_company, shipping_branch, 
            store_id, payment_status, payment_method, deposit_amount, cod_type, subtotal, discount, total_amount, cash_received, cash_change, sale_type) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssssssssdsddddds", 
        $order_id, 
        $customer_id, 
        $order_date, 
        $shipping_company, 
        $customer_address, 
        $store_id, // ค่า store_id
        $payment_status, 
        $payment_method, 
        $deposit_amount, 
        $cod_type, 
        $subtotal, 
        $discount_amount, 
        $total_amount,
        $cash_received,
        $cash_change,
        $sale_type
    );
    
    $stmt->execute();
    
    // ึกรายการค้า
    foreach ($order_data['items'] as $item) {
        $product_id = $item['id'];
        $quantity = $item['quantity'];
        $price = $item['price'];
        $total = $item['total'];
        
        // ค่าเพื่อตรวจสอบ
        error_log("Processing order item - ID: " . $product_id . ", Quantity: " . $quantity);
        
        $sql = "INSERT INTO order_items (order_id, product_id, quantity, price, total) 
                VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("siddd", $order_id, $product_id, $quantity, $price, $total);
        $stmt->execute();
        
        // ค้นหา pro_id จาก product_id
        $find_sql = "SELECT pro_id FROM products WHERE product_id = ?";
        $find_stmt = $conn->prepare($find_sql);
        $find_stmt->bind_param("i", $product_id);
        $find_stmt->execute();
        $find_result = $find_stmt->get_result();
        
        if ($row = $find_result->fetch_assoc()) {
            $pro_id = $row['pro_id'];
            
            // ปเดตสต็อกด้วย pro_id
            $update_sql = "UPDATE products SET quantity = quantity - ? WHERE pro_id = ?";
            $update_stmt = $conn->prepare($update_sql);
            $update_stmt->bind_param("is", $quantity, $pro_id);
            $update_stmt->execute();
        }
    }
    
    // Commit transaction
    $conn->commit();
    
    echo json_encode(['status' => 'success', 'message' => 'ບັນທຶກການສັ່ງຊື້ສຳເລັດແລ້ວ', 'order_id' => $order_id]);
    
} catch (Exception $e) {
    // Rollback transaction ถ้าข้อຫຼີ
    $conn->rollback();
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}

// ข้อมูล<|im_start|>ค้า
function saveCustomer($conn, $name, $phone, $address) {
    // เช็คว่ามีลูกค้าที่ชื่อหรือเบอร์นี้อยู่หรือไม่
    $sql = "SELECT * FROM customers WHERE name = ? OR phone = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $name, $phone);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        // วนลูปเช็คแต่ละแถว
        while ($row = $result->fetch_assoc()) {
            // ถ้าชื่อและเบอร์ตรงกันพอดี ให้ใช้ customer_id เดิม
            if ($row['name'] === $name && $row['phone'] === $phone) {
                return $row['customer_id'];
            }
        }
        // ถ้าชื่อหรือเบอร์ซ้ำแต่ไม่ตรงกัน ให้เพิ่มเป็นลูกค้าใหม่
    }

    // เพิ่มลูกค้าใหม่
    // ดึง customer_id ทั้งหมด แล้วหาเลขมากที่สุด
    $sql = "SELECT customer_id FROM customers WHERE customer_id LIKE 'CUS_%'";
    $result = $conn->query($sql);
    $max_id = 0;
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            if (preg_match('/^CUS_\d{9}$/', $row['customer_id'])) {
                $num = intval(substr($row['customer_id'], 4));
                if ($num > $max_id) {
                    $max_id = $num;
                }
            }
        }
    }
    $new_id = $max_id + 1;
    $customer_id = 'CUS_' . str_pad($new_id, 9, '0', STR_PAD_LEFT);
    $sql = "INSERT INTO customers (customer_id, name, phone) VALUES (?, ?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sss", $customer_id, $name, $phone);
    $stmt->execute();

    return $customer_id;
}

// สร้าง order_id ไม่ซ้ำ
function generateOrderID($conn) {
    // ใช้ order_ตามด้วยเลข
    $prefix = "order_";
    
    // หาค่า order_id ล่าibase
    $sql = "SELECT MAX(CAST(SUBSTRING(order_id, 7) AS UNSIGNED)) as max_id 
            FROM orders 
            WHERE order_id LIKE '{$prefix}%'";
    
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    $next_id = 1;
    if ($row['max_id']) {
        $next_id = $row['max_id'] + 1;
    }
    
    // สร้าง order_id ใหม่ใน order_XX
    return $prefix . sprintf('%02d', $next_id);
}

$conn->close();
?>












