<?php
require_once("auth.php");
require_once("db_con.php");

// Generate new income ID
function generateIncomeID($conn) {
    $sql = "SELECT MAX(CAST(SUBSTRING(income_id, 4) AS UNSIGNED)) as max_id FROM other_income WHERE income_id LIKE 'INC%'";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    $next_id = 1;
    if ($row['max_id']) {
        $next_id = $row['max_id'] + 1;
    }
    
    return 'INC' . str_pad($next_id, 4, '0', STR_PAD_LEFT);
}

$new_income_id = generateIncomeID($conn);
$alert_message = '';
$alert_type = '';

// Handle form submission for adding/updating income
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $income_id = $_POST['income_id'];
    $income_date = $_POST['income_date'];
    $details = $_POST['details'];
    $amount = $_POST['amount'];
    
    // Check if income_id already exists
    $check_sql = "SELECT income_id FROM other_income WHERE income_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->bind_param("s", $income_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        // Update existing record
        $sql = "UPDATE other_income SET income_date = ?, details = ?, amount = ? WHERE income_id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssds", $income_date, $details, $amount, $income_id);
        
        if ($stmt->execute()) {
            $alert_message = '<div class="alert alert-success">ອັບເດດຂໍ້ມູນລາຍຮັບສຳເລັດແລ້ວ</div>';
            $alert_type = 'success';
        } else {
            $alert_message = '<div class="alert alert-danger">ເກີດຂໍ້ຜິດພາດ: ' . $stmt->error . '</div>';
            $alert_type = 'danger';
        }
    } else {
        // Insert new record
        $sql = "INSERT INTO other_income (income_id, income_date, details, amount) VALUES (?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("sssd", $income_id, $income_date, $details, $amount);
        
        if ($stmt->execute()) {
            $alert_message = '<div class="alert alert-success">ບັນທຶກຂໍ້ມູນລາຍຮັບສຳເລັດແລ້ວ</div>';
            $alert_type = 'success';
            $new_income_id = generateIncomeID($conn);
        } else {
            $alert_message = '<div class="alert alert-danger">ເກີດຂໍ້ຜິດພາດ: ' . $stmt->error . '</div>';
            $alert_type = 'danger';
        }
    }
}

// Handle delete request
if (isset($_GET['delete']) && !empty($_GET['delete'])) {
    $delete_id = $_GET['delete'];
    
    $sql = "DELETE FROM other_income WHERE income_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $delete_id);
    
    if ($stmt->execute()) {
        $alert_message = '<div class="alert alert-success">ລົບຂໍ້ມູນລາຍຮັບສຳເລັດແລ້ວ</div>';
        $alert_type = 'success';
    } else {
        $alert_message = '<div class="alert alert-danger">ເກີດຂໍ້ຜິດພາດ: ' . $stmt->error . '</div>';
        $alert_type = 'danger';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/home.css">
    <link rel="stylesheet" href="css/other_income.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+Lao:wght@100;300;400;500;700;900&display=swap" rel="stylesheet">
    <title>ລາຍຮັບອື່ນໆ - ຮ້ານ Jຕຸກຂາຍເຄື່ອງໃຊ້ໄຟຟ້າ</title>
    <link rel="icon" type="image/png" href="/test01/favicon.png">

</head>
<body>
    <?php include_once('home.php'); ?>

    <div class="content-area">
        <div class="income-table-container">
            <div class="sticky-header-wrapper">
                <div class="table-header">
                    <h1 style="font-size: 2em;"><i class="fas fa-file-invoice" style="font-size: 28px;"></i> ຂໍ້ມູນລາຍຮັບອື່ນໆ</h1>
                    <div class="search-container">
                        <input type="text" id="income-search" placeholder="ຄົ້ນຫາລາຍຮັບ...">
                        <button class="btn primary-btn" id="add-income-btn"><i class="fas fa-plus"></i> ເພີ່ມລາຍຮັບໃໝ່</button>
                    </div>
                </div>
                
                <?php 
                if (!empty($alert_message)) {
                    echo $alert_message;
                }
                ?>
                
                <div class="table-header-row">
                    <table class="income-table">
                        <thead>
                            <tr>
                                <th>ລຳດັບ</th>
                                <th>ລະຫັດໄອດີ</th>
                                <th>ວັນທີ</th>
                                <th>ລາຍລະອຽດ</th>
                                <th>ຈຳນວນເງິນ</th>
                                <th>ຈັດການ</th>
                            </tr>
                        </thead>
                    </table>
                </div>
            </div>
            
            <div class="table-body-container">
                <table class="income-table">
                    <thead style="visibility: hidden; position: absolute;">
                        <tr>
                            <th>ລຳດັບ</th>
                            <th>ລະຫັດໄອດີ</th>
                            <th>ວັນທີ</th>
                            <th>ລາຍລະອຽດ</th>
                            <th>ຈຳນວນເງິນ</th>
                            <th>ຈັດການ</th>
                        </tr>
                    </thead>
                <tbody id="income-table-body">
                    <?php
                    $sql = "SELECT * FROM other_income ORDER BY income_date DESC";
                    $result = $conn->query($sql);
                    
                    if ($result->num_rows > 0) {
                        $i = 1;
                        while ($row = $result->fetch_assoc()) {
                            echo "<tr>";
                            echo "<td>" . $i++ . "</td>";
                            echo "<td>" . htmlspecialchars($row['income_id']) . "</td>";
                            echo "<td>" . htmlspecialchars($row['income_date']) . "</td>";
                            echo "<td>" . htmlspecialchars($row['details']) . "</td>";
                            echo "<td>" . number_format($row['amount'], 0) . "</td>";
                            echo "<td class='action-buttons'>";
                            echo "<button class='edit-btn' data-id='" . $row['income_id'] . "'><i class='fas fa-edit'></i></button>";
                            echo "<button class='delete-btn' data-id='" . $row['income_id'] . "'><i class='fas fa-trash'></i></button>";
                            echo "</td>";
                            echo "</tr>";
                        }
                    } else {
                        echo "<tr><td colspan='6' style='text-align:center'>ບໍ່ພົບຂໍ້ມູນລາຍຮັບ</td></tr>";
                    }
                    ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Modal for adding/editing income -->
    <input type="hidden" id="new_income_id" value="<?php echo $new_income_id; ?>">
    
    <div id="incomeModal" class="modal">
        <div class="modal-content">
            <span class="close">&times;</span>
            <h2 class="modal-title"><i class="fas fa-money-bill-wave"></i> ເພີ່ມຂໍ້ມູນລາຍຮັບອື່ນໆ</h2>
            
            <form method="post" action="">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="income_id">ລະຫັດລາຍຮັບ</label>
                        <input type="text" id="income_id" name="income_id" value="<?php echo $new_income_id; ?>" readonly>
                    </div>
                    
                    <div class="form-group">
                        <label for="income_date">ວັນທີ</label>
                        <input type="date" id="income_date" name="income_date" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="amount">ຈຳນວນເງິນ</label>
                        <input type="number" id="amount" name="amount" step="1" required>
                    </div>
                    
                    <div class="form-group full-width">
                        <label for="details">ລາຍລະອຽດ</label>
                        <textarea id="details" name="details" rows="3" required></textarea>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="reset" class="btn secondary-btn"><i class="fas fa-undo"></i> ລ້າງຂໍ້ມູນ</button>
                    <button type="submit" class="btn primary-btn"><i class="fas fa-save"></i> ບັນທຶກຂໍ້ມູນ</button>
                </div>
            </form>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Modal functionality
        const modal = document.getElementById("incomeModal");
        const addBtn = document.getElementById("add-income-btn");
        const closeBtn = document.getElementsByClassName("close")[0];

        // เพิ่มคลาส show เพื่อใช้กับ fade out
        function showModal() {
            modal.style.display = "block";
            setTimeout(() => {
                modal.classList.add("show");
            }, 10);
        }

        function hideModal() {
            modal.classList.remove("show");
            setTimeout(() => {
                modal.style.display = "none";
            }, 300);
        }

        // Open modal when add button is clicked
        addBtn.onclick = function() {
            showModal();
            document.querySelector('.modal-title').innerHTML = '<i class="fas fa-money-bill-wave"></i> ເພີ່ມຂໍ້ມູນລາຍຮັບອື່ນໆ';
            document.querySelector('form').reset();
            document.getElementById('income_id').value = document.getElementById('new_income_id').value;
            document.getElementById('income_date').value = new Date().toISOString().split('T')[0];
            document.getElementById('amount').value = '';
            document.getElementById('details').value = '';
        }

        // Close modal when X is clicked
        if (closeBtn) {
            closeBtn.addEventListener('click', function() {
                hideModal();
            });
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            if (event.target == modal) {
                hideModal();
            }
        }
        
        // Edit button functionality
document.querySelectorAll('.edit-btn').forEach(button => {
    button.addEventListener('click', function() {
        const incomeId = this.getAttribute('data-id');
        fetch('get_income?id=' + incomeId)
            .then(response => response.json())
            .then(data => {
                document.getElementById('income_id').value = data.income_id;
                document.getElementById('income_date').value = data.income_date;
                document.getElementById('details').value = data.details;
                document.getElementById('amount').value = parseInt(data.amount);

                document.querySelector('.modal-title').innerHTML = '<i class="fas fa-edit"></i> ແກ້ໄຂຂໍ້ມູນລາຍຮັບອື່ນໆ';
                showModal(); // เรียกฟังก์ชันนี้แทน
            })
            .catch(error => console.error('Error:', error));
    });
});
        
        // Delete button functionality
document.querySelectorAll('.delete-btn').forEach(button => {
    button.addEventListener('click', function() {
        const incomeId = this.getAttribute('data-id');
        Swal.fire({
            title: 'ຢືນຢັນການລົບຂໍ້ມູນ',
            text: 'ທ່ານແນ່ໃຈບໍ່ວ່າຕ້ອງການລົບຂໍ້ມູນລາຍຮັບນີ້?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'ຕົກລົງ',
            cancelButtonText: 'ຍົກເລີກ'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = 'other_income?delete=' + incomeId;
            }
        });
    });
});
        
        // Search functionality
document.getElementById('income-search').addEventListener('keyup', function() {
    const searchValue = this.value.toLowerCase();
    const tableRows = document.querySelectorAll('#income-table-body tr');
    let anyVisible = false;

    tableRows.forEach(row => {
        let found = false;
        const cells = row.querySelectorAll('td');
        cells.forEach(cell => {
            if (cell.textContent.toLowerCase().includes(searchValue)) {
                found = true;
            }
        });
        row.style.display = found ? '' : 'none';
        if (found) anyVisible = true;
    });

    // ลบแถว "ບໍ່ພົບສິນຄ້າ" เดิมก่อน
    let noDataRow = document.getElementById('no-data-row');
    if (noDataRow) noDataRow.remove();

    // ถ้าไม่พบข้อมูลเลย ให้เพิ่มแถวใหม่
    if (!anyVisible) {
        const tbody = document.getElementById('income-table-body');
        const tr = document.createElement('tr');
        tr.id = 'no-data-row';
        tr.innerHTML = `<td colspan="6" style="text-align:center;color:#888;">ບໍ່ພົບສິນຄ້າ</td>`;
        tbody.appendChild(tr);
    }
});
    });
    </script>
</body>
</html>











