document.addEventListener('DOMContentLoaded', function() {
    // Set default dates to today
    const today = new Date();
    document.getElementById('date-from').valueAsDate = today;
    document.getElementById('date-to').valueAsDate = today;
    
    // Load sold products when page loads
    loadSoldProducts(); 
     
    // Filter button click event
    document.getElementById('filter-btn').addEventListener('click', function() {
        loadSoldProducts();
    });
    
    // Reset button click event
    document.getElementById('reset-btn').addEventListener('click', function() {
        document.getElementById('date-from').valueAsDate = today;
        document.getElementById('date-to').valueAsDate = today;
        document.getElementById('category-filter').value = 'all';
        document.getElementById('product-search').value = '';
        loadSoldProducts();
    });
    
    // Category filter change event
    document.getElementById('category-filter').addEventListener('change', function() {
        loadSoldProducts();
    });
    
    // ค้นหาแบบ realtime
    const searchInput = document.getElementById('product-search');
    searchInput.addEventListener('input', function() {
        loadSoldProducts();
    });
});

// Global variable to store all loaded products
let allLoadedProducts = [];

function loadSoldProducts() {
    const dateFrom = document.getElementById('date-from').value;
    const dateTo = document.getElementById('date-to').value;
    const categoryId = document.getElementById('category-filter').value;
    const search = document.getElementById('product-search').value;
    const tableBody = document.getElementById('sold-products-table-body');

    tableBody.innerHTML = '<tr><td colspan="9" class="loading-message">ກຳລັງໂຫຼດ...</td></tr>';

    fetch(`reduce_products_table.php?date_from=${dateFrom}&date_to=${dateTo}&category=${categoryId}&search=${encodeURIComponent(search)}`)
        .then(response => response.text())
        .then(html => {
            tableBody.innerHTML = html;
        })
        .catch(error => {
            tableBody.innerHTML = `<tr><td colspan="9" class="loading-message">ເກີດຂໍ້ຜິດພາດ: ${error.message}</td></tr>`;
        });
}

function displayProducts(products) {
    const tableBody = document.getElementById('sold-products-table-body');
    tableBody.innerHTML = '';
    
    // Check if there's an error message
    if (products.error) {
        tableBody.innerHTML = `<tr><td colspan="9" class="loading-message">ເກີດຂໍ້ຜິດພາດ: ${products.error}</td></tr>`;
        return;
    }
    
    if (products.length === 0) {
        tableBody.innerHTML = '<tr><td colspan="9" class="loading-message">ບໍ່ພົບຂໍ້ມູນສິນຄ້າທີ່ຂາຍໄດ້</td></tr>';
        return;
    }
    
    // Group products by product ID
    const productGroups = {};
    let totalDays = new Set();
    
    products.forEach(product => {
        totalDays.add(product.sale_date);
        
        if (!productGroups[product.pro_id]) {
            productGroups[product.pro_id] = {
                pro_id: product.pro_id,
                product_name: product.product_name,
                product_image: product.product_image,
                category_name: product.category_name || 'N/A',
                product_type: product.product_type || 'N/A',
                selling_price: product.selling_price,
                total_quantity: 0,
                total_amount: 0,
                daily_sales: {}
            };
        }
        
        // Add to total for this product
        productGroups[product.pro_id].total_quantity += parseInt(product.quantity_sold);
        productGroups[product.pro_id].total_amount += parseFloat(product.quantity_sold) * parseFloat(product.selling_price);
        
        // Add to daily sales for this product
        if (!productGroups[product.pro_id].daily_sales[product.sale_date]) {
            productGroups[product.pro_id].daily_sales[product.sale_date] = {
                quantity: 0,
                amount: 0
            };
        }
        
        productGroups[product.pro_id].daily_sales[product.sale_date].quantity += parseInt(product.quantity_sold);
        productGroups[product.pro_id].daily_sales[product.sale_date].amount += 
            parseFloat(product.quantity_sold) * parseFloat(product.selling_price);
    });
    
    // If we have more than one day, show product summaries
    if (totalDays.size > 1) {
        // Sort products by total quantity sold (highest first)
        const sortedProducts = Object.values(productGroups).sort((a, b) => b.total_quantity - a.total_quantity);
        
        // For each product, show a summary and then daily breakdown
        sortedProducts.forEach(product => {
            // Add product summary header
            const productHeaderRow = document.createElement('tr');
            productHeaderRow.className = 'product-header';
            productHeaderRow.innerHTML = `
                <td colspan="9">
                    <div class="product-summary-header">
                        <div class="product-info">
                            <img src="img/products/${product.product_image}" alt="${product.product_name}" onerror="this.src='img/Logo.png'">
                            <div class="product-details">
                                <h3>${product.product_name}</h3>
                                <p>ລະຫັດ: ${product.pro_id} | ປະເພດ: ${product.category_name}</p>
                            </div>
                        </div>
                        <div class="product-summary">
                            <div class="summary-item">
                                <span class="summary-label">ຈຳນວນຂາຍທັງໝົດ:</span>
                                <span class="summary-value">${product.total_quantity} ອັນ</span>
                            </div>
                            <div class="summary-item">
                                <span class="summary-label">ລາຄາຂາຍ:</span>
                                <span class="summary-value">₭${parseFloat(product.selling_price).toLocaleString()}</span>
                            </div>
                            <div class="summary-item">
                                <span class="summary-label">ຍອດຂາຍລວມ:</span>
                                <span class="summary-value">₭${product.total_amount.toLocaleString()}</span>
                            </div>
                        </div>
                    </div>
                </td>
            `;
            tableBody.appendChild(productHeaderRow);
            
            // Add daily sales for this product
            const dailySalesRow = document.createElement('tr');
            dailySalesRow.className = 'daily-sales-row';
            
            // Create daily sales table
            let dailySalesHTML = `
                <td colspan="9">
                    <table class="daily-sales-table">
                        <thead>
                            <tr>
                                <th>ວັນທີ</th>
                                <th>ຈຳນວນຂາຍ</th>
                                <th>ຍອດຂາຍ</th>
                            </tr>
                        </thead>
                        <tbody>
            `;
            
            // Sort dates in descending order (newest first)
            const sortedDates = Object.keys(product.daily_sales).sort((a, b) => new Date(b) - new Date(a));
            
            sortedDates.forEach(date => {
                const dailySale = product.daily_sales[date];
                const dateParts = new Date(date).toLocaleDateString('en-GB', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric'
                }).split('/');
                const formattedDate = `${dateParts[0]}-${dateParts[1]}-${dateParts[2]}`;
                
                dailySalesHTML += `
                    <tr>
                        <td>${formattedDate}</td>
                        <td>${dailySale.quantity} ອັນ</td>
                        <td>₭${dailySale.amount.toLocaleString()}</td>
                    </tr>
                `;
            });
            
            dailySalesHTML += `
                        </tbody>
                    </table>
                </td>
            `;
            
            dailySalesRow.innerHTML = dailySalesHTML;
            tableBody.appendChild(dailySalesRow);
            
            // Add separator
            const separatorRow = document.createElement('tr');
            separatorRow.className = 'separator-row';
            separatorRow.innerHTML = `<td colspan="9" class="separator"></td>`;
            tableBody.appendChild(separatorRow);
        });
    } else {
        // If only one day, show regular daily view
        // Group products by date
        const productsByDate = {};
        
        products.forEach(product => {
            if (!productsByDate[product.sale_date]) {
                productsByDate[product.sale_date] = [];
            }
            productsByDate[product.sale_date].push(product);
        });
        
        // Display products grouped by date
        Object.keys(productsByDate).sort((a, b) => new Date(b) - new Date(a)).forEach(date => {
            const products = productsByDate[date];
            const dateParts = new Date(date).toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            }).split('/');
            const formattedDateDDMMYYYY = `${dateParts[0]}-${dateParts[1]}-${dateParts[2]}`;
            
            // Add date header
            const dateRow = document.createElement('tr');
            dateRow.className = 'date-header';
            dateRow.innerHTML = `<td colspan="9"><strong>ວັນທີ: ${formattedDateDDMMYYYY}</strong></td>`;
            tableBody.appendChild(dateRow);
            
            // Add products for this date
            products.forEach(product => {
                const row = document.createElement('tr');
                const totalAmount = parseFloat(product.quantity_sold) * parseFloat(product.selling_price);
                
                row.innerHTML = `
                    <td>${product.pro_id}</td>
                    <td><img src="img/products/${product.product_image}" alt="${product.product_name}" onerror="this.src='img/Logo.png'"></td>
                    <td>${product.product_name}</td>
                    <td>${product.category_name || 'N/A'}</td>
                    <td>${product.product_type || 'N/A'}</td>
                    <td>${product.quantity_sold}</td>
                    <td>₭${parseFloat(product.selling_price).toLocaleString()}</td>
                    <td>₭${totalAmount.toLocaleString()}</td>
                    <td>${formattedDateDDMMYYYY}</td>
                `;
                
                tableBody.appendChild(row);
            });
        });
    }
}

function filterProductsBySearch() {
    const searchTerm = document.getElementById('product-search').value.toLowerCase();
    
    if (!searchTerm) {
        // If search is empty, show all products
        displayProducts(allLoadedProducts);
        return;
    }
    
    // Filter products based on search term
    const filteredProducts = allLoadedProducts.filter(product => {
        return (
            (product.pro_id && product.pro_id.toLowerCase().includes(searchTerm)) ||
            (product.product_name && product.product_name.toLowerCase().includes(searchTerm)) ||
            (product.category_name && product.category_name.toLowerCase().includes(searchTerm)) ||
            (product.product_type && product.product_type.toLowerCase().includes(searchTerm)) ||
            (product.sale_date && product.sale_date.includes(searchTerm))
        );
    });
    
    // Display filtered products
    displayProducts(filteredProducts);
}










