document.addEventListener('DOMContentLoaded', function() {
    // Initialize variables
    let cart = [];
    let allProducts = [];
    let currentCategory = 'all';
    let searchTerm = ''; 
    let currentSaleMode = 'online'; // เพิ่มตัวแปรเก็บโหมดการขาย
 
    // DOM elements
    const productsContainer = document.getElementById('products-container');
    const categoriesContainer = document.querySelector('.categories');
    const cartBody = document.getElementById('cart-body');
    const subtotalElement = document.getElementById('subtotal');
    const totalElement = document.getElementById('total');
    const productSearch = document.getElementById('product-search');
    const checkoutBtn = document.getElementById('checkout-btn');
    const clearCartBtn = document.getElementById('clear-cart');
    const checkoutModal = document.getElementById('checkout-modal');
    
    // Event listeners
    if (productSearch) {
        productSearch.addEventListener('input', filterProducts);
    }
    
    if (checkoutBtn) {
        checkoutBtn.addEventListener('click', function(e) {
            e.preventDefault();
            if (cart.length === 0) {
                Swal.fire('ກະຕ່າວ່າງ', 'ກະລຸນາເພີ່ມສິນຄ້າກ່ອນ', 'warning');
                return;
            }
            Swal.fire({
                title: 'ປະເພດການຂາຍ',
                text: 'ທ່ານຕ້ອງການຂາຍເເບບໃດ ?',
                icon: 'question',
                showDenyButton: true,
                showCancelButton: true,
                confirmButtonText: 'ຂາຍອອນລາຍ',
                denyButtonText: 'ໜ້າຮ້ານ',
                cancelButtonText: 'ຍົກເລີກ'
            }).then((result) => {
                if (result.isConfirmed) {
                    // ออนไลน์: แสดง modal ทั้งหมด
                    currentSaleMode = 'online'; // เซ็ตโหมด
                    if (checkoutModal) {
                        checkoutModal.style.display = 'block';
                        document.querySelectorAll('#checkout-modal .form-section').forEach(sec => sec.style.display = 'block');
                        toggleCashSection(false);
                        updateModalSummary();
                        setModalTitlesForSaleType('online');
                    }
                } else if (result.isDenied) {
                    // หน้าร้าน: แสดง modal เฉพาะ section 4-7
                    currentSaleMode = 'pos'; // เซ็ตโหมด
                    if (checkoutModal) {
                        checkoutModal.style.display = 'block';
                        document.querySelectorAll('#checkout-modal .form-section').forEach((sec, idx) => {
                            if (idx < 3) {
                                sec.style.display = 'none';
                            } else {
                                sec.style.display = 'block';
                            }
                        });
                        toggleCashSection(true);
                        updateModalSummary();
                        setModalTitlesForSaleType('pos');
                    }
                }
            });
        });
    }
    
    if (clearCartBtn) {
        clearCartBtn.addEventListener('click', clearCart);
    }
    
    if (checkoutModal) {
        const closeModal = checkoutModal.querySelector('.close');
        const confirmOrderBtn = document.getElementById('confirm-order');
        const cancelOrderBtn = document.getElementById('cancel-order');
        
        if (closeModal) {
            closeModal.addEventListener('click', hideCheckoutModal);
        }
        
        if (cancelOrderBtn) {
            cancelOrderBtn.addEventListener('click', hideCheckoutModal);
        }
        
        if (confirmOrderBtn) {
            confirmOrderBtn.addEventListener('click', saveOrder);
        }
        
        // Payment status change
        const paymentStatusInputs = document.querySelectorAll('input[name="payment_status"]');
        if (paymentStatusInputs.length > 0) {
            paymentStatusInputs.forEach(input => {
                input.addEventListener('change', function() {
                    const depositAmountContainer = document.getElementById('deposit-amount-container');
                    // เลือก section ່ payment-method-container โดยใช้ querySelector ້ไป
                    const paymentMethodContainer = document.querySelector('.payment-method-container');
                    const paymentMethodSection = paymentMethodContainer ? paymentMethodContainer.closest('.form-section') : null;
                    const paymentMethodInputs = document.querySelectorAll('input[name="payment_method"]');
                    
                    // ซ่อน/แสดงส่วนมัดจำ
                    if (this.value === 'ມັດຈຳ' && depositAmountContainer) {
                        depositAmountContainer.style.display = 'block';
                    } else if (depositAmountContainer) {
                        depositAmountContainer.style.display = 'none';
                    }
                    
                    // ซ่อน/แสดงส่วนประเภทการจ่าย (່หมด)
                    if (this.value === 'ຍັງບໍ່ຈ່າຍ' && paymentMethodSection) {
                        paymentMethodSection.style.display = 'none';
                        
                        // ซ่อน cash-section เมื่อเลือก "ຍັງບໍ່ຈ່າຍ"
                        toggleCashSection(false);
                        
                        // ยกົ້າการເລືອກໃນประเภทการจ่าย
                        paymentMethodInputs.forEach(radio => {
                            radio.checked = false;
                        });
                        
                        // เຟີ້ມ hidden input ເັດສຳລັບ payment_method
                        if (!document.getElementById('empty-payment-method')) {
                            const emptyInput = document.createElement('input');
                            emptyInput.type = 'hidden';
                            emptyInput.id = 'empty-payment-method';
                            emptyInput.name = 'payment_method';
                            emptyInput.value = '';
                            document.querySelector('.modal-content').appendChild(emptyInput);
                        }
                    } else if (paymentMethodSection) {
                        paymentMethodSection.style.display = 'block';
                        
                        // ซ่อน cash-section เมื่อเลือก "ມັດຈຳ" (จะแสดงก็ต่อเมื่อเลือกจ่ายสดและสถานะเป็น "ຈ່าຍເເလ້ວ")
                        if (this.value === 'ມັດຈຳ') {
                            toggleCashSection(false);
                        } else if (this.value === 'ຈ່າຍເເລ້ວ') {
                            // ตรวจสอบว่าเลือกจ่ายสดหรือไม่ ถ้าใช่ให้แสดง cash-section
                            const currentPaymentMethod = document.querySelector('input[name="payment_method"]:checked');
                            if (currentPaymentMethod && currentPaymentMethod.value === 'ຈ່าຍສົດ') {
                                toggleCashSection(true);
                            }
                        }
                        
                        // ลบ hidden input ถ้า
                        const emptyInput = document.getElementById('empty-payment-method');
                        if (emptyInput) {
                            emptyInput.remove();
                        }
                        
                        // ตรวจสอบว่าມີການເລືອກຫຼື່ງບໍ່ ถ່ວນມີການເລືອກຫຼື່ງ ໃຫ້ເກີດການເລືອກຫຼື່ງ
                        const hasChecked = Array.from(paymentMethodInputs).some(radio => radio.checked);
                        if (!hasChecked && paymentMethodInputs.length > 0) {
                            paymentMethodInputs[0].checked = true;
                        }
                        
                        // Trigger payment method change event เพื่อให้ cash-section แสดงถูกต้อง
                        const currentCheckedMethod = document.querySelector('input[name="payment_method"]:checked');
                        if (currentCheckedMethod) {
                            const changeEvent = new Event('change');
                            currentCheckedMethod.dispatchEvent(changeEvent);
                        }
                    }
                });
            });
            
            // ตรวจสอบสถานะเริ่มຕ้น
            const initialStatus = document.querySelector('input[name="payment_status"]:checked');
            if (initialStatus) {
                // จำลองการเปลี่ยนแปลงเพื่อตั้งค่าเริ่มต้น
                const event = new Event('change');
                initialStatus.dispatchEvent(event);
            }
        }
        
        // Payment method change
        const paymentMethodInputs = document.querySelectorAll('input[name="payment_method"]');
        if (paymentMethodInputs.length > 0) {
            paymentMethodInputs.forEach(input => {
                input.addEventListener('change', function() {
                    // ตรวจสอบสถานะการจ่าย
                    const paymentStatus = document.querySelector('input[name="payment_status"]:checked');
                    const currentStatus = paymentStatus ? paymentStatus.value : '';
                    
                    // แสดง cash-section เฉพาะเมื่อเลือก "ຈ່າຍສົດ" และสถานะเป็น "ຈ່າຍເຕັມ"
                    if (this.value === 'ຈ່າຍສົດ' && currentStatus === 'ຈ່າຍເເລ້ວ') {
                        toggleCashSection(true);
                    } else {
                        // ซ่อน cash-section สำหรับกรณีอื่นๆ (โอน, มัดจำ, ยังไม่จ่าย)
                        toggleCashSection(false);
                    }
                });
            });
        }

        // Discount input
        const discountAmountInput = document.getElementById('discount-amount');
        if (discountAmountInput) {
            discountAmountInput.addEventListener('input', function() {
                updateModalSummary();
                // อัพเดทปุ่มย่อยเมื่อส่วนลดเปลี่ยน
                if (cashSection && cashSection.style.display !== 'none') {
                    setTimeout(() => {
                        updateQuickAmountButtons();
                    }, 100); // เพิ่มเวลารอ
                }
            });
        }

        // Deposit input
        const depositAmountInput = document.getElementById('deposit-amount');
        if (depositAmountInput) {
            depositAmountInput.addEventListener('input', function() {
                updateModalSummary();
                // อัพเดทปุ่มย่อยเมื่อมัดจำเปลี่ยน
                if (cashSection && cashSection.style.display !== 'none') {
                    setTimeout(() => {
                        updateQuickAmountButtons();
                    }, 100); // เพิ่มเวลารอ
                }
            });
        }
    }
    
    // Load products and categories
    if (productsContainer) {
        loadProducts();
    }
    
    if (categoriesContainer) {
        loadCategories();
    }
    
    // Functions
    function loadProducts() {
        fetch('get_products')
            .then(response => response.json())
            .then(data => {
                allProducts = data;
                filterProducts();
            })
            .catch(error => console.error('Error loading products:', error));
    }
    
    function loadCategories() {
        fetch('get_categories')
            .then(response => response.json())
            .then(data => {
                const categorySelect = document.getElementById('category-select');
                
                // Add category options
                data.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category.category_id;
                    option.textContent = category.category_name;
                    categorySelect.appendChild(option);
                });
                
                // Add change event listener
                categorySelect.addEventListener('change', function() {
                    currentCategory = this.value;
                    filterProducts();
                });
            })
            .catch(error => console.error('Error loading categories:', error));
    }
    
    function filterProducts() {
        let filteredProducts = allProducts;
        
        // Filter by category
        if (currentCategory !== 'all') {
            filteredProducts = filteredProducts.filter(product => 
                product.category_id === currentCategory
            );
        }
        
        // Filter by search term
        if (searchTerm) {
            filteredProducts = filteredProducts.filter(product => 
                product.product_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                (product.pro_id && product.pro_id.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (product.category_name && product.category_name.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (product.product_type && product.product_type.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (product.selling_price && product.selling_price.toString().includes(searchTerm))
            );
        }
        
        displayProducts(filteredProducts);
    }
    
    function displayProducts(products) {
        const container = document.getElementById('products-container');
        container.innerHTML = '';
        
        if (products.length === 0) {
            container.innerHTML = '<div class="no-products"><i class="fas fa-search" style="font-size: 24px; margin-bottom: 10px; opacity: 0.5;"></i><br>ບໍ່ພົບສິນຄ້າ</div>';
            return;
        }
        
        products.forEach(product => {
            const card = document.createElement('div');
            card.className = 'product-card'; 
            
            // เพิ่ม class สินค้าหมด
            if (parseInt(product.quantity) <= 0) {
                card.classList.add('out-of-stock');
            }
            
            // กำหนดสถานะสินค้า
            let stockStatus = '';
            let stockBadgeClass = '';
            let stockIcon = '';
            
            if (parseInt(product.quantity) <= 0) {
                stockStatus = '<div class="stock-badge out-of-stock-badge">ສິນຄ້າໝົດ</div>';
                stockIcon = '<div class="stock-indicator out-of-stock-indicator"><i class="fas fa-times-circle"></i> ໝົດແລ້ວ</div>';
                stockBadgeClass = 'out-of-stock';
            } else if (parseInt(product.quantity) <= 20) {
                stockIcon = '<div class="stock-indicator low-stock-indicator"><i class="fas fa-exclamation-triangle"></i> ເຫຼືອ ' + product.quantity + ' ອັນ</div>';
                stockBadgeClass = 'low-stock';
            } else {
                stockIcon = '<div class="stock-indicator in-stock-indicator"><i class="fas fa-check-circle"></i> ມີສິນຄ້າ</div>';
                stockBadgeClass = 'in-stock';
            }
            
            // ปรับปรุง HTML ของการ์ดสินค้า
            card.innerHTML = `
                <div class="product-image-container">
                    <div class="product-image" style="background-image: url('${product.product_image ? 'img/products/' + product.product_image : 'img/Logo.png'}')">
                        ${stockStatus}
                    </div>
                </div>
                <div class="product-info">
                    <div class="product-name" title="${product.product_name}">${product.product_name}</div>
                    <div class="product-meta-section">
                        <div class="product-category">${product.category_name || ''}</div>
                        <div class="product-type">${product.product_type || ''}</div>
                    </div>
                    <div class="product-price-section">
                        <div class="product-price">₭${parseFloat(product.selling_price).toLocaleString()}</div>
                        
                    </div>
                    <div class="product-stock ${stockBadgeClass}">
                        ${stockIcon}
                    </div>
                </div>
            `;
            
            // เพิ่ม hover effect
            card.addEventListener('mouseenter', function() {
                if (parseInt(product.quantity) > 0) {
                    this.classList.add('hover-effect');
                }
            });
            
            card.addEventListener('mouseleave', function() {
                this.classList.remove('hover-effect');
            });
            
            if (parseInt(product.quantity) > 0) {
                card.addEventListener('click', () => addToCart(product));
            } else {
                card.classList.add('disabled');
                card.addEventListener('click', () => {
                    Swal.fire({
                        title: 'ສິນຄ້ານີ້ໝົດແລ້ວ',
                        text: 'ຂໍອະໄພ ສິນຄ້ານີ້ໝົດໃນສາງແລ້ວ',
                        icon: 'warning',
                        confirmButtonText: 'ເຂົ້າໃຈແລ້ວ'
                    });
                });
            }
            
            container.appendChild(card);
        });
    }
    
    // Add event listener for search input
    document.getElementById('product-search').addEventListener('input', function() {
        searchTerm = this.value;
        filterProducts();
    });
    
    function addToCart(product) {
        // Get current stock quantity
        const stockQuantity = parseInt(product.quantity);
        
        // ตรวจสอบว่า ค้าในคลัง ไม่
        if (stockQuantity <= 0) {
            alert('ສິນຄ້ານີ້ໝົດແລ້ວ');
            return;
        }
        
        // Check if product is already in cart
        const existingItemIndex = cart.findIndex(item => item.id === product.product_id);
        
        if (existingItemIndex !== -1) {
            // Check if adding more would exceed stock
            if (cart[existingItemIndex].quantity >= stockQuantity) {
                Swal.fire('ບໍ່ສາມາດເພີ່ມໄດ້ອີກ', `ມີສິນຄ້າໃນສາງພຽງ ${stockQuantity} ອັນ`, 'warning');
                return;
            }
            // Increment quantity if already in cart
            cart[existingItemIndex].quantity += 1;
            cart[existingItemIndex].total = cart[existingItemIndex].price * cart[existingItemIndex].quantity;
        } else {
            // Add new item to cart
            cart.push({
                id: product.product_id,
                pro_id: product.pro_id,
                name: product.product_name,
                price: parseFloat(product.selling_price),
                quantity: 1,
                total: parseFloat(product.selling_price),
                stockQuantity: stockQuantity // Store stock quantity for reference
            });
        }
        
        updateCartDisplay();
        calculateTotal();
    }
    
    function updateCartDisplay() {
        cartBody.innerHTML = '';
        
        if (cart.length === 0) {
            cartBody.innerHTML = '<tr><td colspan="5" class="empty-cart">ບໍ່ມີສິນຄ້າໃນກະຕ່າ</td></tr>';
            return;
        }
        
        cart.forEach((item, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td class="cart-item-name">${item.name}</td>
                <td>₭${item.price.toLocaleString()}</td>
                <td>
                    <div class="quantity-control">
                        <div class="quantity-btn minus">-</div>
                        <input type="number" class="quantity-input" value="${item.quantity}" min="1">
                        <div class="quantity-btn plus">+</div>
                    </div>
                </td>
                <td>₭${item.total.toLocaleString()}</td>
                <td><i class="fas fa-trash remove-item"></i></td>
            `;
            // Add event listener for minus button
            const minusBtn = row.querySelector('.quantity-btn.minus');
            if (minusBtn) {
                minusBtn.addEventListener('click', function() {
                    if (cart[index].quantity > 1) {
                        cart[index].quantity -= 1;
                        cart[index].total = cart[index].price * cart[index].quantity;
                    } else {
                        cart.splice(index, 1);
                    }
                    updateCartDisplay();
                    calculateTotal();
                });
            }
            // Add event listener for plus button
            const plusBtn = row.querySelector('.quantity-btn.plus');
            if (plusBtn) {
                plusBtn.addEventListener('click', function() {
                    const stockQuantity = parseInt(cart[index].stockQuantity);
                    if (cart[index].quantity >= stockQuantity) {
                        Swal.fire('ບໍ່ສາມາດເພີ່ມໄດ້ອີກ', `ມີສິນຄ້າໃນສາງພຽງ ${stockQuantity} ອັນ`, 'warning');
                        return;
                    }
                    cart[index].quantity += 1;
                    cart[index].total = cart[index].price * cart[index].quantity;
                    updateCartDisplay();
                    calculateTotal();
                });
            }
            // Add event listener for quantity input
            const qtyInput = row.querySelector('.quantity-input');
            if (qtyInput) {
                qtyInput.addEventListener('change', function() {
                    const newQuantity = parseInt(this.value);
                    const stockQuantity = parseInt(cart[index].stockQuantity);
                    if (newQuantity <= 0) {
                        cart.splice(index, 1);
                    } else if (newQuantity > stockQuantity) {
                        alert(`ບໍ່ສາມາດເພີ່ມໄດ້ອີກ. ມີສິນຄໍາໃນສາງພຽງ ${stockQuantity} ອັນ`);
                        cart[index].quantity = stockQuantity;
                    } else {
                        cart[index].quantity = newQuantity;
                        cart[index].total = cart[index].price * newQuantity;
                    }
                    updateCartDisplay();
                    calculateTotal();
                });
            }
            // Add event listener for remove button
            const removeBtn = row.querySelector('.remove-item');
            if (removeBtn) {
                removeBtn.addEventListener('click', function() {
                    cart.splice(index, 1);
                    updateCartDisplay();
                    calculateTotal();
                });
            }
            cartBody.appendChild(row);
        });
    }
    
    function calculateTotal() {
        // คำนวณยอดรวมจากรายการในตะกร้า
        const total = cart.reduce((sum, item) => sum + item.total, 0);
        
        // แสดงเฉพาะยอดรวม
        if (totalElement) {
            totalElement.textContent = `₭${total.toLocaleString()}`;
        }
        
        // Update modal summary too if it's open
        if (checkoutModal && checkoutModal.style.display === 'block') {
            updateModalSummary();
        }
    }
    
    function updateModalSummary() {
        const subtotal = cart.reduce((sum, item) => sum + item.total, 0);
        const discountAmountInput = document.getElementById('discount-amount') || { value: '0' };
        const discount = parseFloat(discountAmountInput.value) || 0;
        const depositAmountInput = document.getElementById('deposit-amount') || { value: '0' };
        const deposit = parseFloat(depositAmountInput.value) || 0;
        const total = subtotal - discount - deposit;
        
        const modalSubtotalElement = document.getElementById('modal-subtotal');
        if (modalSubtotalElement) modalSubtotalElement.textContent = `₭${subtotal.toLocaleString()}`;
        
        const modalDiscountElement = document.getElementById('modal-discount');
        if (modalDiscountElement) modalDiscountElement.textContent = `₭${discount.toLocaleString()}`;
        
        const modalDepositElement = document.getElementById('modal-deposit');
        const depositSummaryRow = document.getElementById('deposit-summary-row');
        if (modalDepositElement && depositSummaryRow) {
            modalDepositElement.textContent = `₭${deposit.toLocaleString()}`;
            // แสดง/ซ่อนแถวมัดจำตามว่ามีการป้อนมัดจำหรือไม่
            depositSummaryRow.style.display = deposit > 0 ? 'flex' : 'none';
        }
        
        const modalTotalElement = document.getElementById('modal-total');
        if (modalTotalElement) modalTotalElement.textContent = `₭${total.toLocaleString()}`;
    }
    
    function showCheckoutModal() {
        if (cart.length === 0) {
            alert('ກະລຸນາເພີ່ມສິນຄ້າໃນກະຕ່າກ່ອນ');
            return;
        }
        
        if (!checkoutModal) return;
        
        // Reset form fields
        const customerName = document.getElementById('customer-name');
        if (customerName) customerName.value = '';
        
        const customerPhone = document.getElementById('customer-phone');
        if (customerPhone) customerPhone.value = '';
        
        const customerAddress = document.getElementById('customer-address');
        if (customerAddress) customerAddress.value = '';
        
        const shippingCompany = document.getElementById('shipping-company');
        if (shippingCompany) shippingCompany.value = '';
        
        const discountAmount = document.getElementById('discount-amount');
        if (discountAmount) discountAmount.value = '0';
        
        const depositAmount = document.getElementById('deposit-amount');
        if (depositAmount) depositAmount.value = '0';
        
        // Hide conditional fields
        const depositAmountContainer = document.getElementById('deposit-amount-container');
        if (depositAmountContainer) depositAmountContainer.style.display = 'none';
        
        const codTypeContainer = document.getElementById('cod-type-container');
        if (codTypeContainer) codTypeContainer.style.display = 'none';
        
        // Update summary
        updateModalSummary();
        
        // Show modal
        checkoutModal.style.display = 'block';
    }
    
    function hideCheckoutModal() {
        checkoutModal.style.display = 'none';
    }
    
    function togglePaymentFields() {
        const paymentStatus = paymentStatusSelect.value;
        
        // Hide all conditional fields first
        depositAmountContainer.style.display = 'none';
        codTypeContainer.style.display = 'none';
        
        // Show relevant fields based on payment status
        if (paymentStatus === 'deposit') {
            depositAmountContainer.style.display = 'block';
        } else if (paymentStatus === 'cod') {
            codTypeContainer.style.display = 'block';
        }
    }
    
    // ກັບສຳລັບ
    // ลบ event listener อันนี้ออก เพราะมี logic validateOrderForm + Swal.fire อยู่แล้วด้านล่าง
    /*
    document.getElementById('confirm-order').addEventListener('click', function() {
        // ตรวจสอบข้อมูลຕ້ອງການ
        const customerName = document.getElementById('customer-name').value;
        const customerPhone = document.getElementById('customer-phone').value;
        const shippingCompany = document.getElementById('shipping-company').value;
        const storeSelect = document.getElementById('store-select').value;
        
        // ตรวจสอบว่ากรอกข้อมูลครบไม่
        if (!customerName || !customerPhone) {
            alert('ກະລຸນາປ້ອນຂໍ້ມູນລູກຄ້າໃຫ້ຄົບຖ້ວນ');
            return;
        }
        
        if (!shippingCompany) {
            alert('ກະລຸນາເລືອກບໍລິສັດຂົນສົ່ງ');
            return;
        }
        
        if (!storeSelect) {
            alert('ກະລຸນາເລືອກຮ້ານຄ້າ');
            return;
        }
        
        // ถ้าตรวจสอบผ่านแล้ว ให้บັນທຶກข้อมูล
        saveOrder();
    });
    */

    function saveOrder() {
        // ตรวจสอบข้อมูลฟอร์มก่อนบันทึก
        const check = validateOrderForm(currentSaleMode); // ใช้ currentSaleMode แทน
        if (!check.valid) {
            Swal.fire('ຂໍ້ມູນບໍ່ຄົບຖ້ວນ', check.msg, 'error');
            return;
        }
        
        // ตรวจสอบว่า<|im_start|>ี่
        if (window.isSaving) {
            alert('ທ່ານຕ້ອງການບັນທຶກ order ນີ້ບໍ ?');
            return;
        }
        
        // ตั�ງค่าสถานะการ<|im_start|>ี่
        window.isSaving = true;
        
        // รวบรวมข้อมูลจากฟอร์ม
        const customerName = document.getElementById('customer-name').value;
        const customerPhone = document.getElementById('customer-phone').value;
        const customerAddress = document.getElementById('customer-address').value;
        const shippingCompany = document.getElementById('shipping-company').value;
        const storeId = document.getElementById('store-select').value;
        
        // ข้อมูลการชำระ<|im_start|>ี่
        const paymentStatus = document.querySelector('input[name="payment_status"]:checked').value;
        
        // ตรวจสอบประเภทการจ่ายตามสถานะการจ่าย
        let paymentMethod = '';
        if (paymentStatus !== 'ຍັງບໍ່ຈ່າຍ') {
            const paymentMethodChecked = document.querySelector('input[name="payment_method"]:checked');
            if (paymentMethodChecked) {
                paymentMethod = paymentMethodChecked.value;
            }
        }
        
        // ข้อมูลอืenne
        const codType = document.querySelector('input[name="cod_type"]:checked')?.value || '';
        const depositAmount = document.getElementById('deposit-amount')?.value || 0;
        const discountAmount = document.getElementById('discount-amount')?.value || 0;
        
        // สร้างข้อมูลส่งเซิร์ฟเวอร์
        const orderData = {
            customer: {
                name: customerName,
                phone: customerPhone,
                address: customerAddress
            },
            shipping_company: shippingCompany,
            store_id: storeId,
            payment_status: paymentStatus,
            payment_method: paymentMethod,
            cod_type: codType,
            deposit_amount: parseFloat(depositAmount),
            discount_amount: parseFloat(discountAmount),
            items: cart,
            summary: {
                subtotal: cart.reduce((sum, item) => sum + item.total, 0),
                discount: parseFloat(discountAmount),
                total_amount: cart.reduce((sum, item) => sum + item.total, 0) - parseFloat(discountAmount)
            }
        };
        
        // เพิ่ม cash_received, cash_change, sale_type ใน orderData
        let cash_received = null;
        let cash_change = null;
        let sale_type = null;
        if (document.getElementById('cash-section') && document.getElementById('cash-section').style.display !== 'none') {
            // หน้าร้าน
            cash_received = parseFloat((cashReceivedInput.value || '0').replace(/,/g, '')) || 0;
            const total = parseFloat((modalTotalElement?.textContent || '0').replace(/[^0-9.]/g, '')) || 0;
            cash_change = cash_received - total;
            sale_type = 'ຂາຍໜ້າຮ້ານ';
        } else if (currentSaleMode === 'pos') {
            // หน้าร้าน - โอน
            sale_type = 'ຂາຍໜ້າຮ້ານ';
        } else {
            // ออนไลน์
            sale_type = 'ຂາຍອອນລາຍ';
        }
        orderData.cash_received = cash_received;
        orderData.cash_change = cash_change;
        orderData.sale_type = sale_type;
        
        // ນະປຸມและแสดงสถานะ
        const confirmOrderBtn = document.getElementById('confirm-order');
        const originalText = confirmOrderBtn.innerHTML;
        confirmOrderBtn.disabled = true;
        confirmOrderBtn.innerHTML = 'ກຳລັງບັນທຶກ...';
        
        // ส่งข้อมูลเซิร์ฟเวอร์
        fetch('save_order', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(orderData)
        })
        .then(async response => {
            let data = await response.json();
            if (data.status === 'success') {
                // SweetAlert2: ถามว่าจะ print ใบบิลไหม
                Swal.fire({
                    title: 'ບັນທຶກສຳເລັດ!',
                    text: 'ທ່ານຕ້ອງການ print ໃບບິນຫຼືບໍ?',
                    icon: 'success',
                    showCancelButton: true,
                    confirmButtonText: 'Print',
                    cancelButtonText: 'ບໍ່ print',
                    reverseButtons: true
                }).then((result) => {
                    if (result.isConfirmed) {
                        // ปริ้นตามประเภท sale_type
                        if (orderData.sale_type === 'ຂາຍໜ້າຮ້ານ') {
                            window.open('print_invoice?order_id=' + encodeURIComponent(data.order_id), '_blank');
                        } else {
                            // เปลี่ยนจาก order_id เป็น order_ids (array) เพื่อรองรับหลายใบในอนาคต
                            window.open('print_multiple_invoices.php?order_ids=' + encodeURIComponent(data.order_id), '_blank');
                        }
                    } else {
                        Swal.fire('ບັນທຶກສຳເລັດ!', '', 'success');
                    }
                }); 
                // ล้างตะกร้าและ modal
                cart = [];
                updateCartDisplay();
                calculateTotal();
                document.getElementById('checkout-modal').style.display = 'none';
                resetCheckoutModal();
                loadProducts(); // <-- เพิ่มบรรทัดนี้ เพื่อรีเฟรชสินค้าแบบ realtime
            } else {
                Swal.fire('ເກີດຂໍ້ຜິດພາດ', data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('ເກີດຂໍ້ຜິດພາດໃນການບັນທຶກຂໍ້ມູນ');
        })
        .finally(() => {
            // ນ້າຄາสถานะและปຸມ
            window.isSaving = false;
            confirmOrderBtn.disabled = false;
            confirmOrderBtn.innerHTML = originalText; 
        });
    }
    
function clearCart() {
    if (cart.length === 0) return;

    Swal.fire({
        title: 'ຢືນຢັນການລ້າງລາຍການ',
        text: 'ທ່ານແນ່ໃຈບໍ່ວ່າຕ້ອງການລ້າງລາຍການສິນຄ້າທັງໝົດ?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'ຕົກລົງ',
        cancelButtonText: 'ຍົກເລີກ'
    }).then((result) => {
        if (result.isConfirmed) {
            cart = [];
            updateCartDisplay();
            calculateTotal();
        }
    });
}
    
    // --- เงินทอนหน้าร้าน ---
    const cashSection = document.getElementById('cash-section');
    const cashReceivedInput = document.getElementById('cash-received');
    const changeAmountSpan = document.getElementById('change-amount');
    const modalTotalElement = document.getElementById('modal-total');
    // แก้ไข bug: ต้องกำหนด confirmOrderBtn ที่นี่ด้วย
    const confirmOrderBtn = document.getElementById('confirm-order');

    // ฟังก์ชันคำนวณเงินทอน
    function updateChangeAmount() {
        if (!cashSection || cashSection.style.display === 'none') return;
        const total = parseFloat((modalTotalElement?.textContent || '0').replace(/[^\d.]/g, '')) || 0;
        // ปรับตรงนี้ให้รองรับ ,
        const received = parseFloat((cashReceivedInput.value || '0').replace(/,/g, '')) || 0;
        let change = received - total;
        if (change < 0) change = 0;
        changeAmountSpan.textContent = `₭${change.toLocaleString()}`;
        // highlight ถ้าเงินไม่พอ
        if (received < total) {
            changeAmountSpan.style.color = '#EE4035';
        } else {
            changeAmountSpan.style.color = '';
        }
    }

    // แสดง/ซ่อน cash-section ตามโหมด modal
    function toggleCashSection(show) {
        if (cashSection) {
            cashSection.style.display = show ? 'block' : 'none';
            if (!show) {
                cashReceivedInput.value = '';
                changeAmountSpan.textContent = '₭0';
                changeAmountSpan.style.color = '';
                // ล้างปุ่มเมื่อซ่อน
                const quickButtonsContainer = document.querySelector('.quick-amount-buttons');
                if (quickButtonsContainer) quickButtonsContainer.innerHTML = '';
            } else {
                // เมื่อแสดง cash section ให้อัพเดทปุ่มย่อยหลังจาก DOM ถูกอัพเดท
                setTimeout(() => {
                    updateQuickAmountButtons();
                }, 50); // รอ 50ms ให้ DOM อัพเดท
            }
        }
    }

    // เพิ่ม event input
    if (cashReceivedInput) {
        cashReceivedInput.addEventListener('input', updateChangeAmount);
    }

    // ฟังก์ชันสร้างปุ่มจำนวนเงินย่อย
    function updateQuickAmountButtons() {
        const quickButtonsContainer = document.querySelector('.quick-amount-buttons');
        if (!quickButtonsContainer || !modalTotalElement) return;
        
        // ใช้ยอดหลังหักส่วนลดแล้ว
        const totalText = modalTotalElement.textContent || '0';
        const total = parseFloat(totalText.replace(/[^\d.]/g, '')) || 0;
        if (total === 0) {
            quickButtonsContainer.innerHTML = '';
            return;
        }
        
        // ธนบัตรกีบ (เรียงจากน้อยไปมาก)
        const kipBanknotes = [1000, 2000, 5000, 10000, 20000, 50000, 100000];
        const suggestions = [];
        
        // เพิ่มจำนวนใกล้เคียงที่สะดวกใช้ (ปัดขึ้นเป็นหลักพัน)
        const roundedThousand = Math.ceil(total / 1000) * 1000;
        if (roundedThousand > total && roundedThousand < 100000) {
            suggestions.push(roundedThousand);
        }
        
        // เพิ่มจำนวนปัดขึ้นเป็นหลักหมื่นที่ใกล้เคียง
        const roundedTenThousand = Math.ceil(total / 10000) * 10000;
        if (roundedTenThousand > total && roundedTenThousand !== roundedThousand) {
            suggestions.push(roundedTenThousand);
        }
        
        // หาธนบัตรที่น้อยที่สุดที่มากกว่ายอดรวม
        for (let i = 0; i < kipBanknotes.length; i++) {
            const banknote = kipBanknotes[i];
            if (banknote > total) {
                // เพิ่มธนบัตรจริงเฉพาะถ้ายังไม่มี
                if (!suggestions.includes(banknote)) {
                    suggestions.push(banknote);
                }
                break;
            }
        }
        
        // เพิ่มธนบัตรใบถัดไปอีก 1-2 ใบ
        const firstBanknoteIndex = kipBanknotes.findIndex(b => suggestions.includes(b));
        if (firstBanknoteIndex !== -1) {
            // เพิ่มธนบัตรใบถัดไป
            if (firstBanknoteIndex + 1 < kipBanknotes.length) {
                const nextBanknote = kipBanknotes[firstBanknoteIndex + 1];
                if (!suggestions.includes(nextBanknote)) {
                    suggestions.push(nextBanknote);
                }
            }
        }
        
        // ถ้ายอดใหญ่กว่าธนบัตรที่ใหญ่ที่สุด ให้ปัดขึ้นเป็นหลักแสน
        if (total >= 100000) {
            const roundedHundredThousand = Math.ceil(total / 100000) * 100000;
            suggestions.length = 0; // ล้างค่าเดิม
            
            // เพิ่มจำนวนใกล้เคียงที่ปัดขึ้นเป็นหลักหมื่น
            const nearbyTenThousand = Math.ceil(total / 10000) * 10000;
            if (nearbyTenThousand > total && nearbyTenThousand < roundedHundredThousand) {
                suggestions.push(nearbyTenThousand);
            }
            
            suggestions.push(roundedHundredThousand);
            suggestions.push(roundedHundredThousand + 100000);
        }
        
        // เรียงลำดับและลบค่าซ้ำ จำกัดแค่ 4 ปุ่ม
        const uniqueSuggestions = [...new Set(suggestions)].sort((a, b) => a - b).slice(0, 4);
        
        // สร้างปุ่ม
        quickButtonsContainer.innerHTML = uniqueSuggestions.map(amount => 
            `<button type="button" class="quick-amount-btn" data-amount="${amount}" 
                     style="display: inline-block; padding: 6px 10px; margin: 2px; border: 2px solid #28a745; background: #ffffff; color: #28a745; border-radius: 6px; font-size: 12px; font-weight: 600; cursor: pointer; transition: all 0.2s ease; min-width: 60px; height: 32px; position: relative; z-index: 1000; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                ₭${amount.toLocaleString()}
            </button>`
        ).join('');
        
        // เพิ่ม event listeners
        quickButtonsContainer.querySelectorAll('.quick-amount-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const amount = parseInt(this.dataset.amount);
                cashReceivedInput.value = amount.toLocaleString('en-US');
                updateChangeAmount();
                
                // เพิ่ม effect เมื่อคลิก
                this.style.background = '#28a745';
                this.style.color = '#ffffff';
                setTimeout(() => {
                    this.style.background = '#ffffff';
                    this.style.color = '#28a745';
                }, 200);
            });
            
            // Hover effect
            btn.addEventListener('mouseenter', function() {
                this.style.background = '#28a745';
                this.style.color = '#ffffff';
            });
            
            btn.addEventListener('mouseleave', function() {
                this.style.background = '#ffffff';
                this.style.color = '#28a745';
            });
        });
    }

    // เรียก updateChangeAmount ทุกครั้งที่ updateModalSummary
    const _updateModalSummary = updateModalSummary;
    updateModalSummary = function() {
        _updateModalSummary();
        updateChangeAmount();
        updateQuickAmountButtons(); // เพิ่มการอัพเดทปุ่มย่อย
    }

    // Validate เงินทอนก่อนบันทึก order (เฉพาะหน้าร้าน)
    // if (confirmOrderBtn) {
    //     confirmOrderBtn.addEventListener('click', function(e) {
    //         if (cashSection && cashSection.style.display !== 'none') {
    //             const total = parseFloat((modalTotalElement?.textContent || '0').replace(/[^\d.]/g, '')) || 0;
    //             const received = parseFloat(cashReceivedInput.value) || 0;
    //             if (received < total) {
    //                 Swal.fire('ຈຳນວນເງິນສົດບໍ່ພຽງພໍ', 'ກະລຸນາປ້ອນເງິນສົດຫຼາຍກວ່າຍອດທັງໝົດ', 'error');
    //                 e.preventDefault();
    //                 return false;
    //             }
    //         }
    //     }, true);
    // }
    
    // --- ฟังก์ชันรีเซ็ตฟอร์ม modal checkout ---
    function resetCheckoutModal() {
        // รีเซ็ต input/textarea ทั้งหมดใน modal-content
        const modal = document.getElementById('checkout-modal');
        if (!modal) return;
        const modalContent = modal.querySelector('.modal-content');
        if (!modalContent) return;
        // รีเซ็ต input, textarea
        modalContent.querySelectorAll('input[type="text"], input[type="number"], textarea').forEach(el => {
            el.value = '';
        });
        // รีเซ็ต select
        modalContent.querySelectorAll('select').forEach(sel => {
            sel.selectedIndex = 0;
        });
        // รีเซ็ต radio (เลือกค่า default ถ้ามี checked)
        modalContent.querySelectorAll('input[type="radio"]').forEach(radio => {
            if (radio.defaultChecked) {
                radio.checked = true;
            } else {
                radio.checked = false;
            }
        });
        // รีเซ็ตเงินทอน
        const changeAmount = document.getElementById('change-amount');
        if (changeAmount) {
            changeAmount.textContent = '₭0';
            changeAmount.style.color = '';
        }
        // รีเซ็ตเงินสดรับ
        const cashReceived = document.getElementById('cash-received');
        if (cashReceived) cashReceived.value = '';
        // รีเซ็ตส่วนลด
        const discountAmount = document.getElementById('discount-amount');
        if (discountAmount) discountAmount.value = '0';
        // รีเซ็ตมัดจำ
        const depositAmount = document.getElementById('deposit-amount');
        if (depositAmount) depositAmount.value = '0';
        // รีเซ็ต summary
        updateModalSummary && updateModalSummary();
        // ซ่อน deposit/cash section
        const depositAmountContainer = document.getElementById('deposit-amount-container');
        if (depositAmountContainer) depositAmountContainer.style.display = 'none';
        const cashSection = document.getElementById('cash-section');
        if (cashSection) cashSection.style.display = 'none';
    }

    // เรียก resetCheckoutModal เมื่อปิด modal หรือบันทึกสำเร็จ
    if (checkoutModal) {
        const closeModal = checkoutModal.querySelector('.close');
        const cancelOrderBtn = document.getElementById('cancel-order');
        if (closeModal) closeModal.addEventListener('click', resetCheckoutModal);
        if (cancelOrderBtn) cancelOrderBtn.addEventListener('click', resetCheckoutModal);
    }

    // หมายเหตุ: resetCheckoutModal จะถูกเรียกหลังบันทึก order สำเร็จในฟังก์ชัน saveOrder()
 

    // --- เปลี่ยนชื่อหัวข้อ modal ตามโหมดการขาย ---
    function setModalTitlesForSaleType(type) {
        // type: 'pos' หรือ 'online'
        const modal = document.getElementById('checkout-modal');
        if (!modal) return;
        // หัวข้อที่ต้องเปลี่ยน
        const h3s = modal.querySelectorAll('.form-section h3');
        if (type === 'pos') {
            // เปลี่ยนเฉพาะ section ที่ต้องการ (index เริ่มที่ 3)
            if (h3s[3]) h3s[3].textContent = '1. ສະຖານະການຈ່າຍຄ່າເຄື່ອງ';
            if (h3s[4]) h3s[4].textContent = '2. ປະເພດການຈ່າຍ';
            if (h3s[5]) h3s[5].textContent = '3. ເລືອກຮ້ານຄ້າ';
            if (h3s[6]) h3s[6].textContent = '4. ສ່ວນຫຼຸດ';
        } else {
            // คืนค่าชื่อเดิม (ออนไลน์)
            if (h3s[3]) h3s[3].textContent = '4. ສະຖານະການຈ່າຍຄ່າເຄື່ອງ';
            if (h3s[4]) h3s[4].textContent = '5. ປະເພດການຈ່າຍ';
            if (h3s[5]) h3s[5].textContent = '6. ເລືອກຮ້ານຄ້າ';
            if (h3s[6]) h3s[6].textContent = '7. ສ່ວນຫຼຸດ';
        }
    }

    // คืนค่าชื่อเดิมเมื่อปิด modal
    if (checkoutModal) {
        const closeModal = checkoutModal.querySelector('.close');
        const cancelOrderBtn = document.getElementById('cancel-order');
        if (closeModal) closeModal.addEventListener('click', function() { setModalTitlesForSaleType('online'); });
        if (cancelOrderBtn) cancelOrderBtn.addEventListener('click', function() { setModalTitlesForSaleType('online'); });
    }

    // --- ฟังก์ชันตรวจสอบข้อมูลฟอร์มตามโหมดการขาย ---
    function validateOrderForm(saleType) {
        let valid = true;
        let msg = '';
        let missing = [];
        if (saleType === 'online') {
            // ออนไลน์: ต้องกรอกชื่อ, เบอร์, ขนส่ง, ที่อยู่, ค่าฝาก, สถานะจ่าย, ร้านค้า
            const customerName = document.getElementById('customer-name').value.trim();
            const customerPhone = document.getElementById('customer-phone').value.trim();
            const shippingCompany = document.getElementById('shipping-company').value.trim();
            const customerAddress = document.getElementById('customer-address').value.trim();
            const codType = document.querySelector('input[name="cod_type"]:checked');
            const paymentStatus = document.querySelector('input[name="payment_status"]:checked');
            const storeSelect = document.getElementById('store-select').value.trim();
            // เฉพาะ field ที่ขาดไม่ได้
            if (!customerName) missing.push('ຊື່ລູກຄ້າ');
            if (!customerPhone) missing.push('ເບີຕິດຕໍ່');
            if (!shippingCompany) missing.push('ບໍລິສັດຂົນສົ່ງ');
            if (!customerAddress) missing.push('ສາຂາຂົນສົ່ງ');
            if (!codType) missing.push('ຂໍ້ມູນຄ່າຝາກ');
            if (!paymentStatus) missing.push('ສະຖານະການຈ່າຍຄ່າເຄື່ອງ');
            if (!storeSelect) missing.push('ຮ້ານຄ້າ');
            if (missing.length > 0) {
                valid = false;
                msg = 'ກະລຸນາປ້ອນ: ' + missing.join(', ');
            }
        } else if (saleType === 'pos') {
            // หน้าร้าน: ต้องกรอกสถานะจ่าย, ประเภทจ่าย (ยกเว้นถ้าเป็น "ຍັງບໍ່ຈ່າຍ"), ร้านค้า
            const paymentStatus = document.querySelector('input[name="payment_status"]:checked');
            const paymentMethod = document.querySelector('input[name="payment_method"]:checked');
            const storeSelect = document.getElementById('store-select').value.trim();
            
            if (!paymentStatus) missing.push('ສະຖານະການຈ່າຍຄ່າເຄື່ອງ');
            
            // ตรวจสอบ payment_method เฉพาะเมื่อไม่ใช่ "ຍັງບໍ່ຈ່າຍ"
            if (paymentStatus && paymentStatus.value !== 'ຍັງບໍ່ຈ່າຍ' && !paymentMethod) {
                missing.push('ປະເພດການຈ່າຍ');
            }
            
            if (!storeSelect) missing.push('ຮ້ານຄ້າ');
            
            if (missing.length > 0) {
                valid = false;
                msg = 'ກະລຸນາປ້ອນ: ' + missing.join(', ');
            }
        }
        return { valid, msg };
    }

    // --- ปรับ logic ปุ่ม confirm-order ให้ validate ตามโหมด ---
    if (confirmOrderBtn) {
        // ลบ event listener ที่เรียก saveOrder ตรงๆ (ถ้ามี)
        confirmOrderBtn.replaceWith(confirmOrderBtn.cloneNode(true));
        const newConfirmOrderBtn = document.getElementById('confirm-order');
        newConfirmOrderBtn.addEventListener('click', function(e) {
            const check = validateOrderForm(currentSaleMode); // ใช้ currentSaleMode
            if (!check.valid) {
                Swal.fire('ຂໍ້ມູນບໍ່ຄົบຖ້ວນ', check.msg, 'error');
                e.preventDefault();
                return false;
            }
            // Validate เงินทอนก่อนบันทึก order (เฉพาะหน้าร้าน + จ่ายสด)
            const cashSection = document.getElementById('cash-section');
            if (cashSection && cashSection.style.display !== 'none') {
                const total = parseFloat((modalTotalElement?.textContent || '0').replace(/[^\d.]/g, '')) || 0;
                const received = parseFloat(cashReceivedInput.value.replace(/,/g, '')) || 0;
                if (received < total) {
                    Swal.fire('ຈຳນວນເງິນສົດບໍ່ພຽງພໍ', 'ກະລຸນາປ້ອນເງິນສົດຫຼາຍກວ່າຍອດທັງໝົດ', 'error');
                    e.preventDefault();
                    return false;
                }
            }
            // ถ้าผ่านทุก validation ค่อย saveOrder
            saveOrder();
        }, true);
    }

    // Make functions available globally
    window.incrementQuantity = function(index) {
        const stockQuantity = parseInt(cart[index].stockQuantity);
        if (cart[index].quantity >= stockQuantity) {
            Swal.fire('ບໍ່ສາມາດເພີ່ມໄດ້ອີກ', `ມີສິນຄ້າໃນສາງພຽງ ${stockQuantity} ອັນ`, 'warning');
            return;
        }
        cart[index].quantity += 1;
        cart[index].total = cart[index].price * cart[index].quantity;
        updateCartDisplay();
        calculateTotal();
    };
    
    window.decrementQuantity = function(index) {
        if (cart[index].quantity > 1) {
            cart[index].quantity -= 1;
            cart[index].total = cart[index].price * cart[index].quantity;
        } else {
            cart.splice(index, 1);
        }
        
        updateCartDisplay();
        calculateTotal();
    };
    
    window.updateQuantity = function(index, value) {
        const newQuantity = parseInt(value);
        const stockQuantity = parseInt(cart[index].stockQuantity || products.find(p => p.product_id === cart[index].id).quantity);
        
        if (newQuantity <= 0) {
            cart.splice(index, 1);
        } else if (newQuantity > stockQuantity) {
            alert(`ບໍ່ສາມາດເພີ່ມໄດ້ອີກ. ມີສິນຄໍາໃນສາງພຽງ ${stockQuantity} ອັນ`);
            cart[index].quantity = stockQuantity;
        } else {
            cart[index].quantity = newQuantity;
            cart[index].total = cart[index].price * newQuantity;
        }
        
        updateCartDisplay();
        calculateTotal();
    };
    
    window.removeFromCart = function(index) {
        cart.splice(index, 1);
        updateCartDisplay();
        calculateTotal();
    };
});























































