document.addEventListener('DOMContentLoaded', function() {
    const salesChartElem = document.getElementById('salesChart');
    if (!salesChartElem) return; // ป้องกัน error ถ้าไม่มี element

    const ctx = salesChartElem.getContext('2d'); 
    let salesChart;
 
    // ฟังก์ชันสำหรับอัพเดทข้อมูลสถิติ
    function updateSummaryStats(summary) {
        // สร้างหรืออัพเดทแถบแสดงสถิติ
        let summaryDiv = document.getElementById('chart-summary');
        if (!summaryDiv) {
            summaryDiv = document.createElement('div');
            summaryDiv.id = 'chart-summary';
            // ตรวจสอบขนาดหน้าจอสำหรับ responsive styling
            const isMobile = window.innerWidth <= 480;
            summaryDiv.style.cssText = `
                background: #f8f9fa; 
                padding: ${isMobile ? '12px 8px' : '15px'}; 
                border-radius: 8px; 
                margin: 15px 0; 
                display: flex; 
                justify-content: space-around; 
                text-align: center;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                flex-wrap: ${isMobile ? 'wrap' : 'nowrap'};
                gap: ${isMobile ? '8px' : '4px'};
            `;
            salesChartElem.parentNode.insertBefore(summaryDiv, salesChartElem);
        }
        
        // ตรวจสอบขนาดหน้าจอสำหรับ responsive summary
        const isMobile = window.innerWidth <= 480;
        const isTablet = window.innerWidth <= 768;
        
        summaryDiv.innerHTML = `
            <div style="flex: 1; ${isMobile ? 'min-width: calc(50% - 8px); margin: 4px 0;' : ''}">
                <div style="font-size: ${isMobile ? '16px' : '18px'}; font-weight: bold; color: #28a745; line-height: 1.3; word-break: break-all; overflow-wrap: break-word;">
                    ₭${summary.formatted.sales}
                </div>
                <div style="font-size: ${isMobile ? '12px' : '14px'}; color: #666; line-height: 1.2;">
                    ລາຍຮັບ (${summary.sales_percentage}%)
                </div>
            </div>
            <div style="flex: 1; ${isMobile ? 'min-width: calc(50% - 8px); margin: 4px 0;' : ''}">
                <div style="font-size: ${isMobile ? '16px' : '18px'}; font-weight: bold; color: #007bff; line-height: 1.3; word-break: break-all; overflow-wrap: break-word;">
                    ₭${summary.formatted.cost_of_sales ?? '-'}
                </div>
                <div style="font-size: ${isMobile ? '12px' : '14px'}; color: #666; line-height: 1.2;">
                    ຕົ້ນທຶນຂາຍ
                </div>
            </div>
            <div style="flex: 1; ${isMobile ? 'min-width: calc(50% - 8px); margin: 4px 0;' : ''}">
                <div style="font-size: ${isMobile ? '16px' : '18px'}; font-weight: bold; color: #EE4035; line-height: 1.3; word-break: break-all; overflow-wrap: break-word;">
                    ₭${summary.formatted.expenses}
                </div>
                <div style="font-size: ${isMobile ? '12px' : '14px'}; color: #666; line-height: 1.2;">
                    ລາຍຈ່າຍ (${summary.expenses_percentage}%)
                </div>
            </div>
            <div style="flex: 1; ${isMobile ? 'min-width: calc(50% - 8px); margin: 4px 0;' : ''}">
                <div style="font-size: ${isMobile ? '16px' : '18px'}; font-weight: bold; color: ${summary.net_profit >= 0 ? '#28a745' : '#dc3545'}; line-height: 1.3; word-break: break-all; overflow-wrap: break-word;">
                    ₭${summary.formatted.profit}
                </div>
                <div style="font-size: ${isMobile ? '12px' : '14px'}; color: #666; line-height: 1.2;">
                    ກຳໄລສຸທິ
                </div>
            </div>
        `;
    }

    function fetchData(start, end) {
        let url = 'get_sales_data.php';
        if (start && end) {
            url += `?start=${start}&end=${end}`; 
        }
        fetch(url)
            .then(res => res.json())
            .then(json => {
                // แสดงข้อมูลสถิติ
                if (json.summary) {
                    updateSummaryStats(json.summary);
                }
                
                if (salesChart) salesChart.destroy(); 
                
                // ตรวจสอบขนาดหน้าจอสำหรับ responsive
                const isMobile = window.innerWidth <= 480;
                const isTablet = window.innerWidth <= 768;
                
                salesChart = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: json.labels,
                        datasets: [
                            {
                                label: `ລາຍຮັບ (${json.summary ? json.summary.sales_percentage + '%' : ''})`,
                                data: json.sales,
                                borderColor: '#28a745',
                                backgroundColor: 'rgba(53, 238, 90, 0.08)',
                                fill: true,
                                tension: 0.3
                            },
                            {
                                label: `ລາຍຈ່າຍ (${json.summary ? json.summary.expenses_percentage + '%' : ''})`,
                                data: json.expenses,
                                borderColor: '#EE4035',
                                backgroundColor: 'rgba(238,64,53,0.08)',
                                fill: true,
                                tension: 0.3
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: !isMobile, // ให้ mobile ปรับ aspect ratio ได้
                        aspectRatio: isMobile ? 1.0 : 2, // mobile ใช้ aspect ratio สูงกว่า (1:1)
                        plugins: {
                            legend: {
                                display: true,
                                position: isMobile ? 'bottom' : 'top',
                                labels: {
                                    font: {
                                        size: isMobile ? 13 : 14 // เพิ่มขนาดฟอนต์เล็กน้อย
                                    },
                                    padding: isMobile ? 12 : 20,
                                    usePointStyle: isMobile, // ใช้จุดแทนสี่เหลี่ยมใน mobile
                                    boxWidth: isMobile ? 15 : 40 // เพิ่มขนาด box
                                }
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    font: {
                                        size: isMobile ? 11 : 12 // เพิ่มขนาดฟอนต์
                                    },
                                    callback: function(value) {
                                        return '₭' + value.toLocaleString();
                                    },
                                    maxTicksLimit: isMobile ? 7 : 10, // เพิ่มจำนวน tick ใน mobile
                                    padding: isMobile ? 8 : 4
                                }
                            },
                            x: {
                                ticks: {
                                    font: {
                                        size: isMobile ? 10 : 12 // เพิ่มขนาดฟอนต์
                                    },
                                    maxTicksLimit: isMobile ? 10 : 12, // เพิ่มจำนวน label ใน mobile
                                    maxRotation: isMobile ? 45 : 0, // หมุน label ใน mobile
                                    minRotation: isMobile ? 45 : 0,
                                    padding: isMobile ? 8 : 4
                                }
                            }
                        },
                        elements: {
                            point: {
                                radius: isMobile ? 4 : 6, // เพิ่มขนาด point
                                hoverRadius: isMobile ? 6 : 8
                            },
                            line: {
                                borderWidth: isMobile ? 2.5 : 3 // เพิ่มความหนาเส้น
                            }
                        },
                        // เพิ่ม interaction สำหรับ mobile
                        interaction: {
                            intersect: false,
                            mode: isMobile ? 'nearest' : 'index'
                        },
                        // ปรับ tooltip สำหรับ mobile
                        tooltip: {
                            titleFont: {
                                size: isMobile ? 13 : 14 // เพิ่มขนาดฟอนต์ tooltip
                            },
                            bodyFont: {
                                size: isMobile ? 12 : 13
                            }
                        }
                    }
                });
            });
    }

    // ตั้งค่า default เป็น 7 วันล่าสุด
    const today = new Date();
    const priorDate = new Date();
    priorDate.setDate(today.getDate() - 6);
    
    // ให้ฟิลด์วันที่แสดงเป็นวันปัจจุบัน
    document.getElementById('start-date').value = today.toISOString().slice(0,10);
    document.getElementById('end-date').value = today.toISOString().slice(0,10);

    // โหลดกราฟด้วยค่าจากฟิลด์วันที่ (จะแสดงข้อมูลตามชั่วโมงถ้าเป็นวันเดียว)
    const startValue = document.getElementById('start-date').value;
    const endValue = document.getElementById('end-date').value;
    fetchData(startValue, endValue);

    document.getElementById('filter-date').addEventListener('click', function() {
        const start = document.getElementById('start-date').value;
        const end = document.getElementById('end-date').value;
        if (start && end) {
            fetchData(start, end);
        }
    });

    // เพิ่ม event listener สำหรับการ resize หน้าจอ
    window.addEventListener('resize', function() {
        if (salesChart) {
            // ดีเลย์เล็กน้อยเพื่อให้ resize เสร็จสิ้น
            setTimeout(() => {
                const start = document.getElementById('start-date').value;
                const end = document.getElementById('end-date').value;
                fetchData(start, end);
            }, 100);
        }
    });
});