<?php
require_once("auth.php");
require_once("db_con.php");

// Set date range
$start_date = date('Y-m-01'); // First day of current month
$end_date = date('Y-m-t');    // Last day of current month

// Debug function to show SQL queries and results
function debugGetTransactions($conn, $start_date, $end_date) {
    echo "<h2>Debug Transaction Queries</h2>";
    
    // First query - Get orders
    $sql = "SELECT o.order_id as id, o.order_date as transaction_date, 
            o.payment_status, o.created_at, o.updated_at,
            CASE 
                WHEN o.payment_status = 'ຈ່າຍເເລ້ວ' THEN (o.subtotal - o.discount)
                WHEN o.payment_status = 'ມັດຈຳ' THEN o.deposit_amount
                ELSE 0
            END as amount
            FROM orders o
            WHERE (o.order_date BETWEEN ? AND ? OR o.updated_at BETWEEN ? AND ?)";
    
    echo "<h3>Query 1 - Get Orders</h3>";
    echo "<pre>" . htmlspecialchars($sql) . "</pre>";
    echo "Parameters: start_date = $start_date, end_date = $end_date<br><br>";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $start_date, $end_date, $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    echo "Results: " . $result->num_rows . " rows<br>";
    if ($result->num_rows > 0) {
        echo "<table border='1'><tr><th>ID</th><th>Transaction Date</th><th>Payment Status</th><th>Amount</th><th>Created At</th><th>Updated At</th></tr>";
        while($row = $result->fetch_assoc()) {
            echo "<tr><td>" . $row["id"] . "</td><td>" . $row["transaction_date"] . "</td><td>" . $row["payment_status"] . "</td><td>" . $row["amount"] . "</td><td>" . $row["created_at"] . "</td><td>" . $row["updated_at"] . "</td></tr>";
        }
        echo "</table>";
    }
    
    // Second query - Get payment status changes
    $sql = "SELECT o.order_id as id, o.updated_at as transaction_date, 
            o.payment_status, o.deposit_amount, o.total_amount,
            CASE
                WHEN o.deposit_amount > 0 THEN (o.total_amount - o.deposit_amount)
                ELSE o.total_amount
            END as amount
            FROM orders o
            WHERE o.payment_status = 'ຈ່າຍເເລ້ວ' 
            AND DATE(o.order_date) != DATE(o.updated_at)
            AND o.deposit_amount > 0
            AND DATE(o.updated_at) BETWEEN ? AND ?";
    
    echo "<h3>Query 2 - Get Payment Status Changes</h3>";
    echo "<pre>" . htmlspecialchars($sql) . "</pre>";
    echo "Parameters: start_date = $start_date, end_date = $end_date<br><br>";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $start_date, $end_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    echo "Results: " . $result->num_rows . " rows<br>";
    if ($result->num_rows > 0) {
        echo "<table border='1'><tr><th>ID</th><th>Transaction Date</th><th>Payment Status</th><th>Deposit</th><th>Total</th><th>Amount</th></tr>";
        while($row = $result->fetch_assoc()) {
            echo "<tr><td>" . $row["id"] . "</td><td>" . $row["transaction_date"] . "</td><td>" . $row["payment_status"] . "</td><td>" . $row["deposit_amount"] . "</td><td>" . $row["total_amount"] . "</td><td>" . $row["amount"] . "</td></tr>";
        }
        echo "</table>";
    }
}

// Run the debug function
debugGetTransactions($conn, $start_date, $end_date);
?>