<?php
$page_title = 'ໃບບິນ';
require_once 'includes/header.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo '<div class="alert alert-danger">ກະລຸນາເລືອກຮ້ານກ່ອນ</div>';
    require_once 'includes/footer.php';
    exit;
}

// Get shop info
$shop_stmt = $conn->prepare("SELECT shop_name, phone, address FROM shops WHERE id = ?");
$shop_stmt->bind_param("i", $shop_id);
$shop_stmt->execute();
$shop_info = $shop_stmt->get_result()->fetch_assoc();
$shop_stmt->close();

// Date filters
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$active_tab = $_GET['tab'] ?? 'online';

// Query orders with all needed fields including items
$sql = "SELECT o.*, 
               c.name as customer_name, 
               c.phone as customer_phone,
               COALESCE(cur.currencys_symbol, '₭') as currency_symbol,
               o.currency_rate,
               sc.company_name as shipping_company,
               sb.branch_name as shipping_branch
        FROM orders o
        LEFT JOIN customers c ON o.customer_id = c.CU_id
        LEFT JOIN currencys cur ON o.currencys_id = cur.currencys_id AND cur.shop_id = o.shop_id
        LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id
        LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id
        WHERE o.shop_id = ? 
          AND o.deleted_at IS NULL
          AND DATE(o.order_date) BETWEEN ? AND ?
        ORDER BY o.created_at ASC";

$stmt = $conn->prepare($sql);
$stmt->bind_param("iss", $shop_id, $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$online_orders = [];
$shop_orders = [];

while ($row = $result->fetch_assoc()) {
    // Get items for this order
    $items_stmt = $conn->prepare("SELECT product_name, quantity, unit_price, total FROM order_items WHERE order_id = ?");
    $items_stmt->bind_param("i", $row['id']);
    $items_stmt->execute();
    $items_result = $items_stmt->get_result();
    $row['items'] = [];
    while ($item = $items_result->fetch_assoc()) {
        $row['items'][] = $item;
    }
    $items_stmt->close();

    // Calculate total with VAT
    $subtotal = floatval($row['subtotal'] ?? 0);
    $vat_enabled = intval($row['vat_enabled'] ?? 0);
    $currency_rate = floatval($row['currency_rate'] ?? 1);
    $tax_amount = floatval($row['tax_amount'] ?? 0);

    if ($vat_enabled && $currency_rate > 0) {
        $vat_in_currency = $tax_amount / $currency_rate;
        $row['display_total'] = $subtotal + $vat_in_currency;
    } else {
        $row['display_total'] = floatval($row['total_amount'] ?? 0);
    }

    // COD type
    $row['cod_type'] = 'ຄ່າຝາກປາຍທາງ';
    if (strpos($row['notes'] ?? '', 'ຫັກ COD') !== false) {
        $row['cod_type'] = 'ຫັກ COD';
    }

    if ($row['order_type'] === 'online') {
        $online_orders[] = $row;
    } else {
        $shop_orders[] = $row;
    }
}
$stmt->close();

// Status badge helper
function getPaymentStatus($status)
{
    switch ($status) {
        case 'paid':
            return 'ຈ່າຍແລ້ວ';
        case 'partial':
            return 'ມັດຈຳ';
        case 'pending':
            return 'ຍັງບໍ່ຈ່າຍ';
        default:
            return $status;
    }
}
?>

<style>
    /* ===== Invoice Page Premium Styles ===== */
    .invoice-page {
        padding: 24px;
        max-width: 1400px;
        margin: 0 auto;
    }

    .page-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 24px;
    }

    .page-title {
        font-size: 24px;
        font-weight: 700;
        color: var(--text-color);
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .page-title i {
        color: var(--accent);
    }

    /* Filter Section */
    .filter-section {
        background: var(--card-bg);
        border-radius: 16px;
        padding: 20px;
        margin-bottom: 24px;
        display: flex;
        gap: 16px;
        flex-wrap: wrap;
        align-items: center;
        border: 1px solid var(--border-color);
    }

    .filter-section input[type="date"] {
        background: var(--bg-color);
        border: 1px solid var(--border-color);
        border-radius: 10px;
        padding: 10px 16px;
        color: var(--text-color);
    }

    .btn-search {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        border: none;
        border-radius: 10px;
        padding: 10px 24px;
        color: white;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 8px;
        transition: all 0.3s ease;
    }

    .btn-search:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(0, 0, 0, 0.2);
    }

    /* Tabs */
    .tabs-container {
        display: flex;
        gap: 8px;
        margin-bottom: 24px;
        background: var(--card-bg);
        padding: 8px;
        border-radius: 14px;
        width: fit-content;
        border: 1px solid var(--border-color);
    }

    .tab-btn {
        padding: 12px 28px;
        border-radius: 10px;
        border: none;
        background: transparent;
        color: var(--text-muted);
        font-size: 15px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .tab-btn:hover {
        background: var(--bg-color);
        color: var(--text-color);
    }

    .tab-btn.active {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
    }

    .tab-btn .count {
        background: rgba(255, 255, 255, 0.2);
        padding: 3px 10px;
        border-radius: 20px;
        font-size: 12px;
    }

    /* Invoice Grid */
    .invoices-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
        gap: 20px;
    }

    /* ========= BILL CARD STYLE - Theme Aware ========= */
    .bill-card {
        background: var(--card-bg);
        border-radius: 16px;
        overflow: hidden;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        border: 1px solid var(--border-color);
    }

    .bill-card:hover {
        transform: translateY(-6px);
        box-shadow: 0 12px 30px rgba(0, 0, 0, 0.25);
        border-color: var(--accent);
    }

    .bill-card-header {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        padding: 14px 16px;
        text-align: center;
    }

    .bill-card-header h4 {
        font-size: 16px;
        font-weight: 700;
        margin: 0;
    }

    .bill-card-header .bill-id {
        font-size: 13px;
        opacity: 0.9;
        margin-top: 4px;
        font-family: monospace;
    }

    .bill-card-body {
        padding: 16px;
        color: var(--text-color);
    }

    .bill-info-row {
        display: flex;
        justify-content: space-between;
        padding: 8px 0;
        border-bottom: 1px dashed var(--border-color);
        font-size: 14px;
    }

    .bill-info-row:last-child {
        border-bottom: none;
    }

    .bill-info-row .label {
        color: var(--text-muted);
    }

    .bill-info-row .value {
        font-weight: 600;
        color: var(--text-color);
    }

    .bill-items-preview {
        background: var(--bg-color);
        border-radius: 10px;
        padding: 12px;
        margin: 12px 0;
        max-height: 100px;
        overflow: hidden;
        border: 1px solid var(--border-color);
    }

    .bill-item-row {
        display: flex;
        justify-content: space-between;
        font-size: 14px;
        padding: 4px 0;
        color: var(--text-muted);
    }

    .bill-item-row .item-name {
        flex: 1;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        margin-right: 8px;
    }

    .bill-total-section {
        border-top: 2px solid var(--accent);
        padding-top: 12px;
        margin-top: 12px;
    }

    .bill-total-row {
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .bill-total-row .total-label {
        font-size: 16px;
        font-weight: 700;
        color: var(--text-color);
    }

    .bill-total-row .total-amount {
        font-size: 24px;
        font-weight: 800;
        color: var(--accent);
    }

    .bill-status-section {
        display: flex;
        gap: 8px;
        margin-top: 12px;
        flex-wrap: wrap;
    }

    .status-tag {
        font-size: 13px;
        padding: 5px 12px;
        border-radius: 20px;
        font-weight: 600;
    }

    .status-tag.paid {
        background: rgba(46, 213, 115, 0.15);
        color: #2ed573;
    }

    .status-tag.pending {
        background: rgba(255, 107, 107, 0.15);
        color: #ff6b6b;
    }

    .status-tag.partial {
        background: rgba(255, 193, 7, 0.15);
        color: #ffc107;
    }

    .status-tag.delivered {
        background: rgba(46, 213, 115, 0.15);
        color: #2ed573;
    }

    .status-tag.processing {
        background: rgba(54, 162, 235, 0.15);
        color: #36a2eb;
    }

    .status-tag.shipped {
        background: rgba(156, 136, 255, 0.15);
        color: #9c88ff;
    }

    /* Modal Bill View */
    .invoice-modal {
        position: fixed;
        inset: 0;
        background: rgba(0, 0, 0, 0.7);
        backdrop-filter: blur(8px);
        z-index: 9999;
        display: none;
        align-items: center;
        justify-content: center;
        padding: 20px;
    }

    .invoice-modal.show {
        display: flex;
    }

    .modal-content {
        background: var(--card-bg);
        border-radius: 20px;
        width: 100%;
        max-width: 450px;
        max-height: 90vh;
        overflow: hidden;
        animation: modalSlideIn 0.3s ease;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.4);
        border: 1px solid var(--border-color);
    }

    @keyframes modalSlideIn {
        from {
            opacity: 0;
            transform: translateY(-30px) scale(0.95);
        }

        to {
            opacity: 1;
            transform: translateY(0) scale(1);
        }
    }

    /* Bill Receipt Style Modal - Theme Aware */
    .bill-receipt {
        font-family: 'Noto Sans Lao', sans-serif;
        background: var(--card-bg);
        color: var(--text-color);
    }

    .bill-receipt-header {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        padding: 20px;
        text-align: center;
        position: relative;
    }

    .bill-receipt-header h2 {
        font-size: 20px;
        font-weight: 700;
        margin: 0 0 8px 0;
    }

    .bill-receipt-header .shop-name {
        font-size: 16px;
        opacity: 0.9;
    }

    .bill-receipt-header .bill-number {
        font-size: 14px;
        margin-top: 8px;
        background: rgba(255, 255, 255, 0.2);
        display: inline-block;
        padding: 4px 12px;
        border-radius: 20px;
        font-family: monospace;
    }

    .modal-close-btn {
        position: absolute;
        top: 12px;
        right: 12px;
        width: 32px;
        height: 32px;
        border-radius: 50%;
        border: none;
        background: rgba(255, 255, 255, 0.2);
        color: white;
        cursor: pointer;
        font-size: 16px;
        transition: all 0.3s ease;
    }

    .modal-close-btn:hover {
        background: rgba(255, 255, 255, 0.3);
        transform: rotate(90deg);
    }

    .bill-receipt-body {
        padding: 20px;
        max-height: calc(90vh - 200px);
        overflow-y: auto;
    }

    .bill-section {
        margin-bottom: 16px;
    }

    .bill-section-title {
        font-size: 13px;
        color: var(--text-muted);
        text-transform: uppercase;
        letter-spacing: 1px;
        margin-bottom: 8px;
        padding-bottom: 4px;
        border-bottom: 1px solid var(--border-color);
    }

    .bill-info-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
    }

    .bill-info-item {
        background: var(--bg-color);
        padding: 10px;
        border-radius: 8px;
        border: 1px solid var(--border-color);
    }

    .bill-info-item .label {
        font-size: 12px;
        color: var(--text-muted);
    }

    .bill-info-item .value {
        font-size: 14px;
        font-weight: 600;
        color: var(--text-color);
        margin-top: 2px;
    }

    .bill-items-table {
        width: 100%;
        border-collapse: collapse;
    }

    .bill-items-table th {
        font-size: 12px;
        color: var(--text-muted);
        text-align: left;
        padding: 8px 4px;
        border-bottom: 1px solid var(--border-color);
    }

    .bill-items-table td {
        font-size: 14px;
        padding: 8px 4px;
        border-bottom: 1px solid var(--border-color);
        color: var(--text-color);
    }

    .bill-items-table .price {
        text-align: right;
    }

    .bill-summary {
        background: var(--bg-color);
        border-radius: 10px;
        padding: 12px;
        border: 1px solid var(--border-color);
    }

    .bill-summary-row {
        display: flex;
        justify-content: space-between;
        padding: 6px 0;
        font-size: 14px;
        color: var(--text-color);
    }

    .bill-summary-row.total {
        border-top: 2px solid var(--accent);
        padding-top: 10px;
        margin-top: 8px;
        font-size: 18px;
        font-weight: 700;
        color: var(--accent);
    }

    .bill-receipt-footer {
        padding: 16px 20px;
        border-top: 1px solid var(--border-color);
        display: flex;
        gap: 10px;
        background: var(--card-bg);
    }

    .bill-receipt-footer .btn {
        flex: 1;
        padding: 12px;
        border-radius: 10px;
        border: none;
        font-weight: 600;
        font-size: 14px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        transition: all 0.3s ease;
    }

    .btn-print-bill {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
    }

    .btn-edit-bill {
        background: var(--bg-color);
        color: var(--text-color);
        border: 1px solid var(--border-color) !important;
    }

    .btn-edit-bill:hover {
        background: var(--accent);
        color: white;
        border-color: var(--accent) !important;
    }

    .btn:hover {
        transform: translateY(-2px);
    }

    /* Status in modal */
    .bill-status-row {
        display: flex;
        gap: 8px;
        margin-top: 8px;
    }

    /* Empty State */
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: var(--text-muted);
    }

    .empty-state i {
        font-size: 48px;
        margin-bottom: 16px;
        opacity: 0.4;
    }

    /* Tab Content */
    .tab-content {
        display: none;
    }

    .tab-content.active {
        display: block;
    }

    /* ========= Bulk Select Feature ========= */
    .bulk-actions-bar {
        display: flex;
        gap: 12px;
        margin-bottom: 20px;
        align-items: center;
        flex-wrap: wrap;
    }

    .btn-bulk {
        padding: 10px 20px;
        border-radius: 10px;
        border: none;
        font-weight: 600;
        font-size: 13px;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 8px;
        transition: all 0.3s ease;
    }

    .btn-bulk.secondary {
        background: var(--bg-color);
        color: var(--text-color);
        border: 1px solid var(--border-color);
    }

    .btn-bulk.success {
        background: linear-gradient(135deg, #2ed573, #00b894);
        color: white;
    }

    .btn-bulk.danger {
        background: linear-gradient(135deg, #ff6b6b, #ee5a5a);
        color: white;
    }

    .btn-bulk:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
    }

    .btn-bulk:disabled {
        opacity: 0.5;
        cursor: not-allowed;
        transform: none;
    }

    .bulk-actions {
        display: none;
        gap: 10px;
        align-items: center;
    }

    .bulk-actions.show {
        display: flex;
    }

    .selected-count {
        background: var(--accent);
        color: white;
        padding: 4px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
    }

    /* Card Checkbox */
    .card-checkbox {
        position: absolute;
        top: 12px;
        left: 12px;
        width: 24px;
        height: 24px;
        z-index: 10;
        display: none;
    }

    .card-checkbox.show {
        display: block;
    }

    .card-checkbox input {
        width: 20px;
        height: 20px;
        cursor: pointer;
        accent-color: var(--accent);
    }

    .bill-card {
        position: relative;
    }

    .bill-card.selected {
        border-color: var(--accent);
        box-shadow: 0 0 0 3px rgba(var(--accent-rgb), 0.2);
    }

    @media (max-width: 768px) {
        .invoices-grid {
            grid-template-columns: 1fr;
        }

        .filter-section {
            flex-direction: column;
        }

        .tabs-container {
            width: 100%;
        }

        .tab-btn {
            flex: 1;
            justify-content: center;
        }
    }

    /* ========= Light Mode Specific Styles ========= */
    /* Use darker version of accent for light mode visibility */
    :root:not(.dark-mode) .bill-card-header,
    body:not(.dark-mode) .bill-card-header {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
    }

    :root:not(.dark-mode) .tab-btn.active,
    body:not(.dark-mode) .tab-btn.active {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
    }

    :root:not(.dark-mode) .bill-receipt-header,
    body:not(.dark-mode) .bill-receipt-header {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
    }

    :root:not(.dark-mode) .btn-search,
    body:not(.dark-mode) .btn-search {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
    }

    :root:not(.dark-mode) .btn-print-bill,
    body:not(.dark-mode) .btn-print-bill {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: white;
        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.2);
    }
</style>

<div class="invoice-page">
    <div class="page-header">
        <h1 class="page-title"><i class="fas fa-file-invoice"></i> ໃບບິນທັງໝົດ</h1>
    </div>

    <!-- Filter Section -->
    <form class="filter-section" method="GET">
        <input type="hidden" name="tab" value="<?= htmlspecialchars($active_tab) ?>">
        <input type="date" name="start_date" value="<?= htmlspecialchars($start_date) ?>">
        <span style="color: var(--text-muted);">ຫາ</span>
        <input type="date" name="end_date" value="<?= htmlspecialchars($end_date) ?>">
        <button type="submit" class="btn-search"><i class="fas fa-search"></i> ຄົ້ນຫາ</button>
    </form>

    <!-- Tabs -->
    <div class="tabs-container">
        <button class="tab-btn <?= $active_tab === 'all' ? 'active' : '' ?>" onclick="switchTab('all')">
            <i class="fas fa-list"></i> ທັງໝົດ
            <span class="count"><?= count($online_orders) + count($shop_orders) ?></span>
        </button>
        <button class="tab-btn <?= $active_tab === 'online' ? 'active' : '' ?>" onclick="switchTab('online')">
            <i class="fas fa-globe"></i> ໃບບິນອອນລາຍ
            <span class="count"><?= count($online_orders) ?></span>
        </button>
        <button class="tab-btn <?= $active_tab === 'shop' ? 'active' : '' ?>" onclick="switchTab('shop')">
            <i class="fas fa-store"></i> ໃບບິນໜ້າຮ້ານ
            <span class="count"><?= count($shop_orders) ?></span>
        </button>
    </div>

    <!-- Bulk Actions Bar -->
    <div class="bulk-actions-bar">
        <button type="button" id="toggle-bulk-select" class="btn-bulk secondary" onclick="toggleBulkSelect()">
            <i class="fas fa-check-square"></i> ເລືອກຫຼາຍໃບບິນ
        </button>
        <div class="bulk-actions" id="bulk-actions">
            <span class="selected-count" id="selected-count">0 ໃບບິນ</span>
            <button id="print-selected" class="btn-bulk success" disabled onclick="printSelectedInvoices()">
                <i class="fas fa-print"></i> ພິມທີ່ເລືອກ
            </button>
            <button id="cancel-selection" class="btn-bulk danger" onclick="cancelBulkSelect()">
                <i class="fas fa-times"></i> ຍົກເລີກ
            </button>
        </div>
    </div>

    <!-- All Orders Tab -->
    <div id="tab-all" class="tab-content <?= $active_tab === 'all' ? 'active' : '' ?>">
        <?php if (empty($online_orders) && empty($shop_orders)): ?>
            <div class="empty-state"><i class="fas fa-inbox"></i>
                <p>ບໍ່ພົບໃບບິນ</p>
            </div>
        <?php else: ?>
            <!-- Online Orders Section -->
            <?php if (!empty($online_orders)): ?>
                <h3 style="color:var(--text-color);margin-bottom:16px;font-size:16px;">
                    <i class="fas fa-globe" style="color:var(--accent);"></i> ໃບບິນອອນລາຍ (
                    <?= count($online_orders) ?>)
                </h3>
                <div class="invoices-grid" style="margin-bottom:30px;">
                    <?php foreach ($online_orders as $order): ?>
                        <div class="bill-card" data-order-id="<?= $order['OR_id'] ?>" data-order-type="online"
                            onclick="handleCardClick(event, '<?= $order['OR_id'] ?>', 'online')">
                            <label class="card-checkbox" onclick="event.stopPropagation()">
                                <input type="checkbox" class="order-checkbox" data-order-id="<?= $order['OR_id'] ?>"
                                    data-order-type="online" onchange="updateSelectedCount()">
                            </label>
                            <div class="bill-card-header online">
                                <h4>ບິນຝາກພັດສະດຸ</h4>
                                <div class="bill-id">
                                    <?= htmlspecialchars($order['OR_id']) ?>
                                </div>
                            </div>
                            <div class="bill-card-body">
                                <div class="bill-info-row">
                                    <span class="label">ວັນທີ</span>
                                    <span class="value">
                                        <?= date('d/m/Y', strtotime($order['order_date'])) ?>
                                    </span>
                                </div>
                                <div class="bill-info-row">
                                    <span class="label">ລູກຄ້າ</span>
                                    <span class="value">
                                        <?= htmlspecialchars($order['customer_name'] ?? 'ລູກຄ້າທົ່ວໄປ') ?>
                                    </span>
                                </div>
                                <div class="bill-total-section">
                                    <div class="bill-total-row">
                                        <span class="total-label">ລາຄາລວມ</span>
                                        <span class="total-amount">
                                            <?= $order['currency_symbol'] ?>
                                            <?= number_format($order['display_total'], 0) ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <!-- Shop Orders Section -->
            <?php if (!empty($shop_orders)): ?>
                <h3 style="color:var(--text-color);margin-bottom:16px;font-size:16px;">
                    <i class="fas fa-store" style="color:var(--accent);"></i> ໃບບິນໜ້າຮ້ານ (
                    <?= count($shop_orders) ?>)
                </h3>
                <div class="invoices-grid">
                    <?php foreach ($shop_orders as $order): ?>
                        <div class="bill-card" data-order-id="<?= $order['OR_id'] ?>" data-order-type="pos"
                            onclick="handleCardClick(event, '<?= $order['OR_id'] ?>', 'shop')">
                            <label class="card-checkbox" onclick="event.stopPropagation()">
                                <input type="checkbox" class="order-checkbox" data-order-id="<?= $order['OR_id'] ?>"
                                    data-order-type="pos" onchange="updateSelectedCount()">
                            </label>
                            <div class="bill-card-header shop">
                                <h4>ໃບບິນ-ຂາຍ</h4>
                                <div class="bill-id">
                                    <?= htmlspecialchars($order['OR_id']) ?>
                                </div>
                            </div>
                            <div class="bill-card-body">
                                <div class="bill-info-row">
                                    <span class="label">ວັນທີ</span>
                                    <span class="value">
                                        <?= date('d/m/Y H:i', strtotime($order['order_date'])) ?>
                                    </span>
                                </div>
                                <div class="bill-info-row">
                                    <span class="label">ລູກຄ້າ</span>
                                    <span class="value">
                                        <?= htmlspecialchars($order['customer_name'] ?? 'ລູກຄ້າທົ່ວໄປ') ?>
                                    </span>
                                </div>
                                <div class="bill-total-section">
                                    <div class="bill-total-row">
                                        <span class="total-label">ລາຄາລວມ</span>
                                        <span class="total-amount">
                                            <?= $order['currency_symbol'] ?>
                                            <?= number_format($order['display_total'], 0) ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <!-- Online Orders Tab -->
    <div id="tab-online" class="tab-content <?= $active_tab === 'online' ? 'active' : '' ?>">
        <?php if (empty($online_orders)): ?>
            <div class="empty-state"><i class="fas fa-inbox"></i>
                <p>ບໍ່ພົບໃບບິນອອນລາຍ</p>
            </div>
        <?php else: ?>
            <div class="invoices-grid">
                <?php foreach ($online_orders as $order): ?>
                    <div class="bill-card" data-order-id="<?= $order['OR_id'] ?>" data-order-type="online"
                        onclick="handleCardClick(event, '<?= $order['OR_id'] ?>', 'online')">
                        <label class="card-checkbox" onclick="event.stopPropagation()">
                            <input type="checkbox" class="order-checkbox" data-order-id="<?= $order['OR_id'] ?>"
                                data-order-type="online" onchange="updateSelectedCount()">
                        </label>
                        <div class="bill-card-header online">
                            <h4>ບິນຝາກພັດສະດຸ</h4>
                            <div class="bill-id"><?= htmlspecialchars($order['OR_id']) ?></div>
                        </div>
                        <div class="bill-card-body">
                            <div class="bill-info-row">
                                <span class="label">ວັນທີ</span>
                                <span class="value"><?= date('d/m/Y', strtotime($order['order_date'])) ?></span>
                            </div>
                            <div class="bill-info-row">
                                <span class="label">ລູກຄ້າ</span>
                                <span class="value"><?= htmlspecialchars($order['customer_name'] ?? 'ລູກຄ້າທົ່ວໄປ') ?></span>
                            </div>
                            <div class="bill-info-row">
                                <span class="label">ຂົນສົ່ງ</span>
                                <span class="value"><?= htmlspecialchars($order['shipping_company'] ?? '-') ?></span>
                            </div>

                            <div class="bill-items-preview">
                                <?php $count = 0;
                                foreach ($order['items'] as $item):
                                    if ($count++ >= 3)
                                        break; ?>
                                    <div class="bill-item-row">
                                        <span class="item-name"><?= htmlspecialchars($item['product_name']) ?></span>
                                        <span>x<?= intval($item['quantity']) ?></span>
                                    </div>
                                <?php endforeach; ?>
                                <?php if (count($order['items']) > 3): ?>
                                    <div class="bill-item-row" style="color:#0984e3;">+<?= count($order['items']) - 3 ?> ລາຍການ
                                    </div>
                                <?php endif; ?>
                            </div>

                            <div class="bill-total-section">
                                <div class="bill-total-row">
                                    <span class="total-label">ລາຄາລວມ</span>
                                    <span
                                        class="total-amount"><?= $order['currency_symbol'] ?><?= number_format($order['display_total'], 0) ?></span>
                                </div>
                            </div>

                            <div class="bill-status-section">
                                <span
                                    class="status-tag <?= $order['payment_status'] ?>"><?= getPaymentStatus($order['payment_status']) ?></span>
                                <span
                                    class="status-tag <?= $order['order_status'] ?>"><?= htmlspecialchars($order['cod_type']) ?></span>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Shop Orders Tab -->
    <div id="tab-shop" class="tab-content <?= $active_tab === 'shop' ? 'active' : '' ?>">
        <?php if (empty($shop_orders)): ?>
            <div class="empty-state"><i class="fas fa-inbox"></i>
                <p>ບໍ່ພົບໃບບິນໜ້າຮ້ານ</p>
            </div>
        <?php else: ?>
            <div class="invoices-grid">
                <?php foreach ($shop_orders as $order): ?>
                    <div class="bill-card" data-order-id="<?= $order['OR_id'] ?>" data-order-type="pos"
                        onclick="handleCardClick(event, '<?= $order['OR_id'] ?>', 'shop')">
                        <label class="card-checkbox" onclick="event.stopPropagation()">
                            <input type="checkbox" class="order-checkbox" data-order-id="<?= $order['OR_id'] ?>"
                                data-order-type="pos" onchange="updateSelectedCount()">
                        </label>
                        <div class="bill-card-header shop">
                            <h4>ໃບບິນ-ຂາຍ</h4>
                            <div class="bill-id"><?= htmlspecialchars($order['OR_id']) ?></div>
                        </div>
                        <div class="bill-card-body">
                            <div class="bill-info-row">
                                <span class="label">ວັນທີ</span>
                                <span class="value"><?= date('d/m/Y H:i', strtotime($order['order_date'])) ?></span>
                            </div>
                            <div class="bill-info-row">
                                <span class="label">ລູກຄ້າ</span>
                                <span class="value"><?= htmlspecialchars($order['customer_name'] ?? 'ລູກຄ້າທົ່ວໄປ') ?></span>
                            </div>

                            <div class="bill-items-preview">
                                <?php $count = 0;
                                foreach ($order['items'] as $item):
                                    if ($count++ >= 3)
                                        break; ?>
                                    <div class="bill-item-row">
                                        <span class="item-name"><?= htmlspecialchars($item['product_name']) ?></span>
                                        <span>x<?= intval($item['quantity']) ?></span>
                                    </div>
                                <?php endforeach; ?>
                                <?php if (count($order['items']) > 3): ?>
                                    <div class="bill-item-row" style="color:#00b894;">+<?= count($order['items']) - 3 ?> ລາຍການ
                                    </div>
                                <?php endif; ?>
                            </div>

                            <div class="bill-total-section">
                                <div class="bill-total-row">
                                    <span class="total-label">ລາຄາລວມ</span>
                                    <span
                                        class="total-amount"><?= $order['currency_symbol'] ?><?= number_format($order['display_total'], 0) ?></span>
                                </div>
                            </div>

                            <div class="bill-status-section">
                                <span
                                    class="status-tag <?= $order['payment_status'] ?>"><?= getPaymentStatus($order['payment_status']) ?></span>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Invoice Modal -->
<div id="invoice-modal" class="invoice-modal" onclick="closeModal(event)">
    <div class="modal-content" onclick="event.stopPropagation()">
        <div class="bill-receipt">
            <div class="bill-receipt-header" id="modal-header">
                <button class="modal-close-btn" onclick="closeModal()"><i class="fas fa-times"></i></button>
                <h2 id="modal-bill-type">ບິນຝາກພັດສະດຸ</h2>
                <div class="shop-name"><?= htmlspecialchars($shop_info['shop_name'] ?? 'ຮ້ານ') ?></div>
                <div class="bill-number" id="modal-bill-number">OR_XXX</div>
            </div>
            <div class="bill-receipt-body" id="modal-body">
                <div style="text-align:center;padding:40px;">
                    <i class="fas fa-spinner fa-spin" style="font-size:32px;color:#0984e3;"></i>
                    <p style="margin-top:16px;color:#636e72;">ກຳລັງໂຫຼດ...</p>
                </div>
            </div>
            <div class="bill-receipt-footer">
                <button class="btn btn-edit-bill" id="btn-edit" onclick="editInvoice()"><i class="fas fa-pen"></i>
                    ແກ້ໄຂ</button>
                <button class="btn btn-print-bill" id="btn-print" onclick="printInvoice()"><i class="fas fa-print"></i>
                    ພິມ</button>
            </div>
        </div>
    </div>
</div>

<script>
    let currentOrderId = null;
    let currentOrderType = null;
    const shopName = '<?= addslashes($shop_info['shop_name'] ?? 'ຮ້ານ') ?>';

    function switchTab(tab) {
        document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));

        // Find and activate the correct tab button
        const tabs = ['all', 'online', 'shop'];
        const tabIndex = tabs.indexOf(tab);
        if (tabIndex >= 0) {
            document.querySelector(`.tab-btn:nth-child(${tabIndex + 1})`).classList.add('active');
        }

        document.getElementById('tab-' + tab).classList.add('active');
        const url = new URL(window.location);
        url.searchParams.set('tab', tab);
        window.history.replaceState({}, '', url);

        // Reset bulk selection when switching tabs
        cancelBulkSelect();
    }

    function viewInvoice(orderId, orderType) {
        currentOrderId = orderId;
        currentOrderType = orderType;
        document.getElementById('invoice-modal').classList.add('show');
        document.getElementById('modal-bill-number').textContent = orderId;

        const header = document.getElementById('modal-header');
        header.className = 'bill-receipt-header ' + orderType;
        document.getElementById('modal-bill-type').textContent = orderType === 'online' ? 'ບິນຝາກພັດສະດຸ' : 'ໃບບິນ-ຂາຍ';

        fetch('api/get_order_details.php?order_id=' + encodeURIComponent(orderId))
            .then(res => res.json())
            .then(data => {
                if (data.status === 'success') {
                    renderBillReceipt(data, orderType);
                } else {
                    document.getElementById('modal-body').innerHTML = `<div class="empty-state"><i class="fas fa-exclamation-circle"></i><p>${data.message || 'ເກີດຂໍ້ຜິດພາດ'}</p></div>`;
                }
            });
    }

    function renderBillReceipt(data, orderType) {
        const o = data.order;
        const items = data.items;
        const s = o.currency_symbol || '₭';

        let itemsHtml = '';
        items.forEach((item, i) => {
            itemsHtml += `<tr>
            <td>${i + 1}</td>
            <td>${escapeHtml(item.product_name)}</td>
            <td class="price">${item.quantity}</td>
            <td class="price">${s}${formatNumber(item.total)}</td>
        </tr>`;
        });

        const paymentText = { 'paid': 'ຈ່າຍແລ້ວ', 'partial': 'ມັດຈຳ', 'pending': 'ຍັງບໍ່ຈ່າຍ' }[o.payment_status] || o.payment_status;
        const orderStatusText = o.order_type === 'online' ? (o.cod_type || 'ຄ່າຝາກປາຍທາງ') : (o.order_status || '-');

        document.getElementById('modal-body').innerHTML = `
        <div class="bill-section">
            <div class="bill-section-title"><i class="fas fa-calendar"></i> ວັນທີ: ${escapeHtml(o.order_date)}</div>
        </div>
        
        ${orderType === 'online' ? `
        <div class="bill-section">
            <div class="bill-section-title"><i class="fas fa-users"></i> ຂໍ້ມູນ</div>
            <div class="bill-info-grid">
                <div class="bill-info-item"><div class="label">ຜູ້ຝາກ</div><div class="value">${escapeHtml(o.customer_name)}</div></div>
                <div class="bill-info-item"><div class="label">ເບີໂທ</div><div class="value">${escapeHtml(o.customer_phone)}</div></div>
                <div class="bill-info-item"><div class="label">ຂົນສົ່ງ</div><div class="value">${escapeHtml(o.shipping_company)}</div></div>
                <div class="bill-info-item"><div class="label">ສາຂາ</div><div class="value">${escapeHtml(o.shipping_branch)}</div></div>
            </div>
        </div>
        ` : `
        <div class="bill-section">
            <div class="bill-section-title"><i class="fas fa-user"></i> ລູກຄ້າ</div>
            <div class="bill-info-grid">
                <div class="bill-info-item"><div class="label">ຊື່</div><div class="value">${escapeHtml(o.customer_name)}</div></div>
                <div class="bill-info-item"><div class="label">ເບີໂທ</div><div class="value">${escapeHtml(o.customer_phone)}</div></div>
            </div>
        </div>
        `}
        
        <div class="bill-section">
            <div class="bill-section-title"><i class="fas fa-shopping-cart"></i> ລາຍການສິນຄ້າ</div>
            <table class="bill-items-table">
                <thead><tr><th>#</th><th>ສິນຄ້າ</th><th class="price">ຈຳນວນ</th><th class="price">ລາຄາ</th></tr></thead>
                <tbody>${itemsHtml || '<tr><td colspan="4" style="text-align:center;color:#999">ບໍ່ພົບລາຍການ</td></tr>'}</tbody>
            </table>
        </div>
        
        <div class="bill-section">
            <div class="bill-summary">
                <div class="bill-summary-row"><span>ລວມເງິນ:</span><span>${s}${formatNumber(o.subtotal)}</span></div>
                <div class="bill-summary-row"><span>ສ່ວນຫຼຸດ:</span><span>${s}${formatNumber(o.discount_amount)}</span></div>
                ${o.vat_enabled ? `<div class="bill-summary-row" style="color:#00b894"><span>VAT (10%):</span><span>${s}${formatNumber(o.vat_amount)}</span></div>` : ''}
                <div class="bill-summary-row total"><span>ຍອດລວມ:</span><span>${s}${formatNumber(o.total_amount)}</span></div>
            </div>
        </div>
        
        <div class="bill-section">
            <div class="bill-section-title"><i class="fas fa-flag"></i> ສະຖານະ</div>
            <div class="bill-status-row">
                <span class="status-tag ${o.payment_status}">${paymentText}</span>
                ${orderType === 'online' ? `<span class="status-tag processing">${escapeHtml(orderStatusText)}</span>` : ''}
            </div>
        </div>
        
        <div style="text-align:center;margin-top:20px;color:#999;font-size:12px;">
            ຂອບໃຈທີ່ໃຊ້ບໍລິການ 💚
        </div>
    `;
    }

    function closeModal(e) {
        if (e && e.target !== document.getElementById('invoice-modal')) return;
        document.getElementById('invoice-modal').classList.remove('show');
    }

    function printInvoice() {
        if (!currentOrderId) return;
        const printPage = currentOrderType === 'online' ? 'print_online.php' : 'print_shop.php';
        window.open(`${printPage}?order_ids=${currentOrderId}`, '_blank');
    }

    function editInvoice() {
        if (!currentOrderId) return;
        window.location.href = `edit_order.php?order_id=${currentOrderId}`;
    }

    function formatNumber(num) {
        return parseFloat(num || 0).toLocaleString('en-US', { maximumFractionDigits: 0 });
    }

    function escapeHtml(str) {
        if (!str || str === '-') return '-';
        const div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    }

    document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape') closeModal();
    });

    // ============ Bulk Select Feature ============
    let bulkSelectActive = false;

    function toggleBulkSelect() {
        bulkSelectActive = !bulkSelectActive;
        const btn = document.getElementById('toggle-bulk-select');
        const bulkActions = document.getElementById('bulk-actions');

        if (bulkSelectActive) {
            btn.innerHTML = '<i class="fas fa-check-square"></i> ໂໝດເລືອກ: ເປີດ';
            btn.classList.add('success');
            btn.classList.remove('secondary');
            bulkActions.classList.add('show');
            document.querySelectorAll('.card-checkbox').forEach(el => el.classList.add('show'));
        } else {
            cancelBulkSelect();
        }
    }

    function cancelBulkSelect() {
        bulkSelectActive = false;
        const btn = document.getElementById('toggle-bulk-select');
        const bulkActions = document.getElementById('bulk-actions');

        btn.innerHTML = '<i class="fas fa-check-square"></i> ເລືອກຫຼາຍໃບບິນ';
        btn.classList.remove('success');
        btn.classList.add('secondary');
        bulkActions.classList.remove('show');

        document.querySelectorAll('.card-checkbox').forEach(el => el.classList.remove('show'));
        document.querySelectorAll('.order-checkbox').forEach(cb => cb.checked = false);
        document.querySelectorAll('.bill-card').forEach(card => card.classList.remove('selected'));
        updateSelectedCount();
    }

    function handleCardClick(e, orderId, orderType) {
        if (bulkSelectActive) {
            // In bulk select mode, toggle checkbox
            const card = e.currentTarget;
            const checkbox = card.querySelector('.order-checkbox');
            if (checkbox && e.target !== checkbox) {
                checkbox.checked = !checkbox.checked;
                updateSelectedCount();
            }
        } else {
            // Normal mode, view invoice
            viewInvoice(orderId, orderType);
        }
    }

    function updateSelectedCount() {
        const checked = document.querySelectorAll('.order-checkbox:checked');
        const count = checked.length;

        document.getElementById('selected-count').textContent = count + ' ໃບບິນ';
        document.getElementById('print-selected').disabled = count === 0;

        // Update card selection styling
        document.querySelectorAll('.bill-card').forEach(card => {
            const checkbox = card.querySelector('.order-checkbox');
            if (checkbox && checkbox.checked) {
                card.classList.add('selected');
            } else {
                card.classList.remove('selected');
            }
        });
    }

    function printSelectedInvoices() {
        const selectedCheckboxes = Array.from(document.querySelectorAll('.order-checkbox:checked'));
        if (selectedCheckboxes.length === 0) {
            alert('ກະລຸນາເລືອກໃບບິນ');
            return;
        }

        // Group by order type
        const onlineOrders = [];
        const shopOrders = [];

        selectedCheckboxes.forEach(cb => {
            const orderType = cb.dataset.orderType;
            const orderId = cb.dataset.orderId;

            if (orderType === 'online') {
                onlineOrders.push(orderId);
            } else {
                shopOrders.push(orderId);
            }
        });

        // Open appropriate print pages
        if (onlineOrders.length > 0) {
            window.open('print_online.php?order_ids=' + onlineOrders.join(','), '_blank');
        }
        if (shopOrders.length > 0) {
            window.open('print_shop.php?order_ids=' + shopOrders.join(','), '_blank');
        }

        // Reset selection after print
        cancelBulkSelect();
    }
</script>

<?php require_once 'includes/footer.php'; ?>