<?php
$page_title = 'ແກ້ໄຂໃບບິນ';
require_once 'includes/header.php';
require_once dirname(__DIR__) . '/includes/secondary_ids.php';

$shop_id = $_SESSION['shop_id'] ?? null;
$user_id = $_SESSION['user_id'] ?? null;
$order_id = $_GET['id'] ?? $_GET['order_id'] ?? '';

// Get return navigation parameters
$return_date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
$return_date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;
$return_search = isset($_GET['search']) ? $_GET['search'] : '';

if (!$shop_id) {
    echo '<div class="alert alert-danger">ກະລຸນາເລືອກຮ້ານກ່ອນ</div>';
    require_once 'includes/footer.php';
    exit;
}

if (!$order_id) {
    echo '<div class="alert alert-danger">ບໍ່ພົບໃບບິນ</div>';   
    require_once 'includes/footer.php';
    exit;
}

// Get order details
$orderStmt = $conn->prepare("
    SELECT o.*, c.name as customer_name, c.phone as customer_phone, c.address as customer_address, c.CU_id,
           sc.company_name as shipping_company_name, sc.id as shipping_company_id_int,
           sb.branch_name as shipping_branch_name, sb.id as shipping_branch_id_int
    FROM orders o
    LEFT JOIN customers c ON o.customer_id = c.CU_id
    LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id AND sc.shop_id = ?
    LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id AND sb.shop_id = ?
    WHERE o.OR_id = ? AND o.shop_id = ? AND o.deleted_at IS NULL
");
$orderStmt->bind_param("iisi", $shop_id, $shop_id, $order_id, $shop_id);
$orderStmt->execute();
$order = $orderStmt->get_result()->fetch_assoc();

if (!$order) {
    echo '<div class="alert alert-danger">ບໍ່ພົບໃບບິນນີ້</div>';
    require_once 'includes/footer.php';
    exit;
}

// Build return URL for navigation back to sales_history
$order_date = date('Y-m-d', strtotime($order['order_date']));

// If no date filters passed, use the order's date
if ($return_date_from === null) {
    $return_date_from = $order_date;
}
if ($return_date_to === null) {
    $return_date_to = $order_date;
}

$return_url = 'sales_history.php?date_from=' . urlencode($return_date_from) . '&date_to=' . urlencode($return_date_to);
if (!empty($return_search)) {
    $return_url .= '&search=' . urlencode($return_search);
}

// Get order items - use orders.id (INT) for order_items.order_id
$itemsStmt = $conn->prepare("
    SELECT oi.*, 
           COALESCE(oi.product_name, p.product_name) as display_name, 
           p.quantity as stock_quantity, 
           p.selling_price as current_price
    FROM order_items oi
    LEFT JOIN products p ON oi.product_id = p.PR_id AND p.shop_id = ?
    WHERE oi.order_id = ?
");
$itemsStmt->bind_param("ii", $shop_id, $order['id']);
$itemsStmt->execute();
$orderItems = $itemsStmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get currency symbol for this order
$currencySymbol = '₭';
$currencyRate = 1;
if (!empty($order['currencys_id'])) {
    $curStmt = $conn->prepare("SELECT currencys_symbol FROM currencys WHERE currencys_id = ? AND shop_id = ?");
    $curStmt->bind_param("si", $order['currencys_id'], $shop_id);
    $curStmt->execute();
    $curResult = $curStmt->get_result()->fetch_assoc();
    if ($curResult) {
        $currencySymbol = $curResult['currencys_symbol'] ?? '₭';
    }
    $curStmt->close();
    $currencyRate = $order['currency_rate'] ?? 1;
}

// Get products for adding more items
$products = $conn->query("SELECT p.*, c.category_name 
    FROM products p 
    LEFT JOIN product_categories c ON p.category_id = c.id 
    WHERE p.shop_id = $shop_id AND p.deleted_at IS NULL AND p.quantity > 0 
    ORDER BY p.product_name")->fetch_all(MYSQLI_ASSOC);

// Get categories
$categories = $conn->query("SELECT id, category_name FROM product_categories WHERE shop_id = $shop_id AND deleted_at IS NULL ORDER BY category_name")->fetch_all(MYSQLI_ASSOC);

// Get customers
$customers = $conn->query("SELECT id, CU_id, name, phone, address FROM customers WHERE shop_id = $shop_id AND deleted_at IS NULL ORDER BY name")->fetch_all(MYSQLI_ASSOC);

// Get shipping companies
$shippingCompanies = $conn->query("SELECT id, SC_id, company_name FROM shipping_companies WHERE shop_id = $shop_id AND deleted_at IS NULL AND is_active = 1 ORDER BY company_name")->fetch_all(MYSQLI_ASSOC);

// Get shipping branches
$shippingBranches = $conn->query("SELECT id, SB_id, company_id, branch_name FROM shipping_branches WHERE shop_id = $shop_id AND deleted_at IS NULL AND is_active = 1 ORDER BY branch_name")->fetch_all(MYSQLI_ASSOC);

// Get currencies
$currencies = $conn->query("
    SELECT c.*, r.rate_id, r.rate_amount
    FROM currencys c
    LEFT JOIN (
        SELECT r1.* FROM rates r1
        INNER JOIN (SELECT currencys_id, MAX(rate_datetime) as max_dt FROM rates WHERE shop_id = $shop_id GROUP BY currencys_id) r2 
        ON r1.currencys_id = r2.currencys_id AND r1.rate_datetime = r2.max_dt WHERE r1.shop_id = $shop_id
    ) r ON c.currencys_id = r.currencys_id
    WHERE c.shop_id = $shop_id ORDER BY c.base_currency DESC
")->fetch_all(MYSQLI_ASSOC);

if (empty($currencies)) {
    $currencies = [['currencys_id' => 'LAK', 'currencys_name' => 'Lao Kip', 'currencys_symbol' => '₭', 'rate_amount' => 1]];
}
?>

<style>
    /* Prevent SweetAlert2 from shifting body - keep scrollbar always visible */
    body {
        overflow-y: scroll !important;
    }
    body.swal2-shown {
        padding-right: 0 !important;
        overflow-y: scroll !important;
    }
    body.swal2-height-auto {
        height: auto !important;
    }
    .swal2-container {
        padding-right: 0 !important;
    }

    .page-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        flex-wrap: wrap;
        gap: 15px;
    }

    .page-header h1 {
        font-size: 24px;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
        margin: 0;
    }

    .page-header h1 i {
        color: var(--accent);
    }

    .order-badge {
        padding: 8px 16px;
        background: var(--accent);
        color: #fff;
        border-radius: 20px;
        font-size: 14px;
        font-weight: 600;
    }

    .edit-wrapper {
        display: grid;
        grid-template-columns: 1fr 480px;
        gap: 20px;
        min-height: calc(100vh - 180px);
    }

    /* Products Panel */
    .products-panel {
        display: flex;
        flex-direction: column;
        background: var(--card-bg);
        border-radius: 16px;
        border: 1px solid var(--border-color);
        overflow: hidden;
        max-height: calc(100vh - 180px);
    }

    .products-header {
        padding: 16px;
        border-bottom: 1px solid var(--border-color);
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
        align-items: center;
    }

    .search-box {
        flex: 1;
        min-width: 180px;
        position: relative;
    }

    .search-box input {
        width: 100%;
        padding: 12px 15px 12px 42px;
        border-radius: 10px;
        border: 2px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--text-primary);
        font-size: 16px;
    }

    .search-box input:focus {
        border-color: var(--accent);
        outline: none;
    }

    .search-box i {
        position: absolute;
        left: 14px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--text-secondary);
    }

    .filter-select {
        padding: 12px;
        border-radius: 10px;
        border: 2px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--text-primary);
        font-size: 16px;
        min-width: 120px;
    }

    .products-grid {
        flex: 1;
        overflow-y: auto;
        padding: 16px;
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(140px, 1fr));
        gap: 12px;
        align-content: start;
    }

    .product-card {
        background: var(--bg-primary);
        border: 2px solid var(--border-color);
        border-radius: 12px;
        padding: 12px;
        text-align: center;
        cursor: pointer;
        transition: all 0.2s;
    }

    .product-card:hover {
        border-color: var(--accent);
        transform: translateY(-3px);
        box-shadow: 0 6px 15px rgba(0, 0, 0, 0.1);
    }

    .product-card h4 {
        font-size: 16px;
        margin-bottom: 6px;
        color: var(--text-primary);
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }

    .product-card .price {
        font-size: 20px;
        font-weight: 700;
        color: var(--accent);
    }

    .product-card .stock {
        font-size: 14px;
        color: var(--text-secondary);
    }

    .product-card .product-img {
        width: 100%;
        height: 80px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(var(--accent-rgb), 0.1);
        border-radius: 8px;
        margin-bottom: 10px;
        overflow: hidden;
    }
    .product-card .product-img img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    .product-card .product-img i {
        font-size: 28px;
        color: var(--accent);
        opacity: 0.5;
    }

    /* Currency Colors */
    .currency-lak {
        color: #5bc0de !important;
    }

    .currency-thb {
        color: #5bc0de !important;
    }

    .currency-usd {
        color: #28a745 !important;
    }

    /* Quick Add Styles */
    .input-with-btn {
        display: flex;
        gap: 8px;
    }

    .input-with-btn select {
        flex: 1;
    }

    .quick-add-btn {
        width: 38px;
        height: 38px;
        border-radius: 8px;
        border: 2px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--accent);
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
        flex-shrink: 0;
    }

    .quick-add-btn:hover {
        background: var(--accent);
        color: white;
        border-color: var(--accent);
        transform: scale(1.05);
    }

    .quick-add-form {
        margin-top: 10px;
        padding: 12px;
        background: var(--bg-primary);
        border: 2px solid var(--accent);
        border-radius: 10px;
        animation: slideDown 0.2s ease;
    }

    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }

        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .quick-add-form input {
        margin-bottom: 10px;
    }

    .quick-add-actions {
        display: flex;
        gap: 8px;
    }

    .btn-quick-save {
        flex: 1;
        padding: 8px 12px;
        border-radius: 8px;
        border: none;
        background: var(--accent);
        color: white;
        font-size: 12px;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 5px;
        transition: all 0.2s;
    }

    .btn-quick-save:hover {
        opacity: 0.9;
        transform: translateY(-1px);
    }

    .btn-quick-cancel {
        padding: 8px 12px;
        border-radius: 8px;
        border: 2px solid var(--border-color);
        background: transparent;
        color: var(--text-secondary);
        cursor: pointer;
        transition: all 0.2s;
    }

    .btn-quick-cancel:hover {
        background: rgba(220, 53, 69, 0.1);
        color: #dc3545;
        border-color: #dc3545;
    }

    .currency-note {
        font-size: 12px;
        color: var(--text-secondary);
        margin-top: 10px;
    }

    /* VAT Toggle Button */
    .vat-toggle {
        padding: 12px 16px;
        border-radius: 10px;
        border: 2px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--text-secondary);
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 6px;
        transition: all 0.2s;
    }

    .vat-toggle.active {
        background: linear-gradient(135deg, #28a745, #20c997);
        color: white;
        border-color: transparent;
    }

    .vat-toggle:hover {
        transform: translateY(-2px);
    }

    .vat-row {
        color: #28a745;
    }

    .refund-row {
        color: #dc3545;
    }

    /* Radio Groups */
    .radio-group {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
        margin-top: 5px;
    }

    .radio-label {
        flex: 1;
        min-width: 80px;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        padding: 8px;
        text-align: center;
        cursor: pointer;
        transition: all 0.2s;
        font-weight: 500;
        font-size: 16px;
        background: var(--bg-primary);
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 5px;
    }

    .radio-label:hover {
        background: var(--bg-secondary);
    }

    .radio-label input {
        margin: 0; 
    }

    .radio-label.active, 
    .radio-label:has(input:checked) {
        border-color: var(--accent);
        background: rgba(var(--accent-rgb), 0.05);
        color: var(--accent);
        font-weight: 600;
    }

    /* Cart/Edit Panel */
    .cart-panel {
        background: var(--card-bg);
        border: 1px solid var(--border-color);
        border-radius: 16px;
        display: flex;
        flex-direction: column;
        overflow: hidden;
        max-height: calc(100vh - 180px);
        overflow-y: auto;
    }

    .cart-header {
        padding: 16px;
        border-bottom: 1px solid var(--border-color);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .cart-header h3 {
        font-size: 20px;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .cart-items {
        min-height: 150px;
        max-height: 250px;
        overflow-y: auto;
        padding: 12px;
        background: var(--bg-primary);
    }

    .cart-item {
        background: var(--bg-primary);
        border: 1px solid var(--border-color);
        border-radius: 10px;
        padding: 12px;
        margin-bottom: 10px;
    }

    .cart-item-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 8px;
    }

    .cart-item-name {
        font-weight: 600;
        font-size: 16px;
    }

    .cart-item-remove {
        background: none;
        border: none;
        color: var(--text-secondary);
        cursor: pointer;
        font-size: 16px;
    }

    .cart-item-remove:hover {
        color: #dc3545;
    }

    .cart-item-controls {
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .qty-controls {
        display: flex;
        align-items: center;
        gap: 6px;
    }

    .qty-btn {
        width: 30px;
        height: 30px;
        border-radius: 8px;
        border: 1px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--text-primary);
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .qty-btn:hover {
        background: var(--accent);
        color: #fff;
        border-color: var(--accent);
    }

    .qty-display {
        font-weight: 600;
        width: 35px;
        text-align: center;
    }

    .item-total {
        font-weight: 700;
        color: var(--accent);
        font-size: 18px;
    }

    .price-input {
        width: 100px;
        padding: 6px 8px;
        border-radius: 6px;
        border: 1px solid var(--border-color);
        background: var(--bg-primary);
        color: var(--text-primary);
        font-size: 16px;
        text-align: right;
    }

    /* Order Info */
    .order-info-section {
        padding: 12px;
        border-top: 1px solid var(--border-color);
        background: var(--card-bg);
        flex-shrink: 0;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 12px;
        margin-bottom: 12px;
    }

    .form-group {
        margin-bottom: 10px;
    }

    .form-group label {
        display: block;
        font-size: 15px;
        color: var(--text-secondary);
        margin-bottom: 4px;
        font-weight: 500;
    }

    .form-control {
        width: 100%;
        padding: 10px 12px;
        border-radius: 8px;
        border: 1px solid var(--border-color);
        background: var(--card-bg);
        color: var(--text-primary);
        font-size: 16px;
    }

    .form-control:focus {
        border-color: var(--accent);
        outline: none;
    }

    /* Summary */
    .cart-summary {
        padding: 16px;
        border-top: 2px dashed var(--border-color);
        background: var(--card-bg);
        flex-shrink: 0;
    }

    .summary-row {
        display: flex;
        justify-content: space-between;
        margin-bottom: 8px;
        font-size: 16px;
    }

    .summary-row.total-row {
        font-size: 24px;
        font-weight: 700;
        color: var(--accent);
        padding-top: 10px;
        border-top: 2px solid var(--border-color);
        margin-top: 8px;
    }

    /* Buttons */
    .btn-pro {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 12px 24px;
        border-radius: 10px;
        border: none;
        font-weight: 600;
        cursor: pointer;
        font-size: 16px;
        transition: all 0.2s;
    }

    .btn-pro.primary {
        background: linear-gradient(135deg, var(--accent), var(--accent-soft));
        color: #fff;
    }

    .btn-pro.primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(74, 144, 217, 0.4);
    }

    .btn-pro.secondary {
        background: var(--border-color);
        color: var(--text-primary);
    }

    .btn-pro.danger {
        background: #dc3545;
        color: #fff;
    }

    .btn-pro:disabled {
        opacity: 0.6;
        cursor: not-allowed;
    }

    .action-buttons {
        display: flex;
        gap: 10px;
        padding: 16px;
        border-top: 1px solid var(--border-color);
    }

    .action-buttons .btn-pro {
        flex: 1;
    }

    @media (max-width: 900px) {
        .edit-wrapper {
            grid-template-columns: 1fr;
            height: auto;
        }

        .products-panel,
        .cart-panel {
            max-height: 50vh;
        }
    }

    .input-group-custom {
        position: relative;
        display: flex;
        align-items: center;
    }

    .input-group-custom .form-control {
        padding-right: 40px; /* Make space for symbol */
    }

    .currency-addon {
        position: absolute;
        right: 12px;
        color: var(--text-secondary);
        font-weight: 600;
        pointer-events: none;
    }
</style>

<div class="page-header">
    <h1><i class="fas fa-edit"></i> ແກ້ໄຂໃບບິນ</h1>
    <span class="order-badge">
        <?= htmlspecialchars($order_id) ?>
    </span>
    <a href="<?= htmlspecialchars($return_url) ?>" class="btn-pro secondary"><i class="fas fa-arrow-left"></i> ກັບໄປ</a>
</div>

<div class="edit-wrapper">
    <!-- Products Panel -->
    <div class="products-panel">
        <div class="products-header">
            <div class="search-box">
                <i class="fas fa-search"></i>
                <input type="text" id="searchProduct" placeholder="ຄົ້ນຫາສິນຄ້າ...">
            </div>
            <select id="categoryFilter" class="filter-select">
                <option value="">ທຸກປະເພດ</option>
                <?php foreach ($categories as $cat): ?>
                    <option value="<?= $cat['id'] ?>">
                        <?= htmlspecialchars($cat['category_name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <select id="currencySelect" class="filter-select">
                <?php foreach ($currencies as $cur): ?>
                    <option value="<?= $cur['currencys_id'] ?>" data-symbol="<?= $cur['currencys_symbol'] ?? '₭' ?>"
                        data-rate="<?= $cur['rate_amount'] ?? 1 ?>" <?= ($cur['currencys_id'] == ($order['currencys_id'] ?? 'CUR_1')) ? 'selected' : '' ?>>
                        <?= $cur['currencys_symbol'] ?? '₭' ?>     <?= htmlspecialchars($cur['currencys_name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button id="vatToggle" class="vat-toggle <?= ($order['vat_enabled'] ?? 0) ? 'active' : '' ?>"
                onclick="toggleVAT()">
                <i class="fas fa-percent"></i> VAT 10%
            </button>
        </div>
        <div class="products-grid" id="productsGrid">
            <?php foreach ($products as $p): ?>
                <div class="product-card" data-id="<?= $p['id'] ?>" data-category="<?= $p['category_id'] ?>"
                    data-name="<?= htmlspecialchars($p['product_name']) ?>"
                    onclick="addToCart(<?= htmlspecialchars(json_encode($p)) ?>)">
                    <div class="product-img">
                        <?php if (!empty($p['product_image'])): ?>
                            <img src="<?= htmlspecialchars($p['product_image']) ?>" alt="">
                        <?php else: ?>
                            <i class="fas fa-box"></i>
                        <?php endif; ?>
                    </div>
                    <h4><?= htmlspecialchars($p['product_name']) ?></h4>
                    <div class="price" data-base-price="<?= $p['selling_price'] ?>">
                        <?= number_format($p['selling_price']) ?> ₭
                    </div>
                    <div class="stock">ຍັງເຫຼືອ: <?= number_format($p['quantity']) ?></div>
                </div>
            <?php endforeach; ?>
            <?php if (empty($products)): ?>
                <div style="grid-column:1/-1;text-align:center;color:var(--text-secondary);padding:40px">
                    <i class="fas fa-box-open" style="font-size:32px;opacity:.4"></i>
                    <p>ບໍ່ພົບສິນຄ້າ</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Cart/Edit Panel -->
    <div class="cart-panel">
        <div class="cart-header">
            <h3><i class="fas fa-shopping-cart"></i> ລາຍການສິນຄ້າ</h3>
            <button class="btn-pro danger" onclick="clearCart()" style="padding:8px 12px;font-size:13px">
                <i class="fas fa-trash"></i> ລ້າງ
            </button>
        </div>

        <div class="cart-items" id="cartItems">
            <!-- Items will be rendered here -->
        </div>

        <!-- Order Info -->
        <div class="order-info-section">
            <div class="form-row">
                <div class="form-group">
                    <label>ລູກຄ້າ</label>
                    <div class="input-with-btn">
                        <select id="customerId" class="form-control">
                            <option value="">-- ເລືອກລູກຄ້າ --</option>
                            <?php foreach ($customers as $c): ?>
                                <option value="<?= $c['CU_id'] ?>" data-phone="<?= htmlspecialchars($c['phone']) ?>"
                                    data-address="<?= htmlspecialchars($c['address']) ?>"
                                    <?= ($c['CU_id'] == $order['customer_id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($c['name']) ?> -
                                    <?= htmlspecialchars($c['phone']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <button type="button" class="quick-add-btn" onclick="showQuickAddCustomer()"
                            title="ເພີ່ມລູກຄ້າໃໝ່">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <div id="quickAddCustomerForm" class="quick-add-form" style="display:none;">
                        <input type="text" id="newCustomerName" class="form-control" placeholder="ຊື່ລູກຄ້າ"
                            style="margin-bottom:8px;">
                        <input type="text" id="newCustomerPhone" class="form-control" placeholder="ເບີໂທ"
                            style="margin-bottom:10px;">
                        <div class="quick-add-actions">
                            <button type="button" class="btn-quick-save" onclick="saveQuickCustomer()">
                                <i class="fas fa-check"></i> ບັນທຶກ
                            </button>
                            <button type="button" class="btn-quick-cancel" onclick="hideQuickAddCustomer()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label>ສະຖານະຈ່າຍ</label>
                    <select id="paymentStatus" class="form-control" onchange="handlePaymentStatusChange(this.value)">
                        <option value="pending" <?= $order['payment_status'] == 'pending' ? 'selected' : '' ?>>ຍັງບໍ່ຈ່າຍ
                        </option>
                        <option value="partial" <?= $order['payment_status'] == 'partial' ? 'selected' : '' ?>>ມັດຈຳ
                        </option>
                        <option value="paid" <?= $order['payment_status'] == 'paid' ? 'selected' : '' ?>>ຈ່າຍແລ້ວ</option>
                        <option value="partial_refund" <?= $order['payment_status'] == 'partial_refund' ? 'selected' : '' ?>>ຄືນເງີນບາງສ່ວນ
                        </option>
                        <option value="refunded" <?= $order['payment_status'] == 'refunded' ? 'selected' : '' ?>>ຄືນເງີນເຕັມຈຳນວນ
                        </option>
                    </select>
                </div>
                <!-- Refund Amount Display -->
                <div class="form-group" id="refundAmountDisplay" 
                    style="display:<?= in_array($order['payment_status'], ['partial_refund', 'refunded']) ? 'block' : 'none' ?>">
                    <label style="color: #dc3545;"><i class="fas fa-undo-alt"></i> ຈຳນວນເງິນທີ່ຄືນ</label>
                    <div id="refundAmountValue" style="font-size: 20px; color: #dc3545; font-weight: 700; padding: 8px 12px; background: rgba(220, 53, 69, 0.1); border-radius: 8px;">
                        <?= $currencySymbol ?><?= number_format($order['refund_amount'] ?? 0) ?>
                    </div>
                    <input type="hidden" id="refundAmount" value="<?= $order['refund_amount'] ?? 0 ?>">
                </div>
                <div class="form-group">
                    <label>ສະຖານະອໍເດີ</label>
                    <select id="orderStatus" class="form-control">
                        <option value="pending" <?= $order['order_status'] == 'pending' ? 'selected' : '' ?>>ກະກຽມຈັດສົ່ງ</option>
                        <option value="processing" <?= $order['order_status'] == 'processing' ? 'selected' : '' ?>>ກຳລັງຈັດສົ່ງ</option>
                        <option value="shipped" <?= $order['order_status'] == 'shipped' ? 'selected' : '' ?>>ສົ່ງອອກແລ້ວ</option>
                        <option value="delivered" <?= $order['order_status'] == 'delivered' ? 'selected' : '' ?>>ລູກຄ້າຮັບແລ້ວ</option>
                        <option value="returned" <?= $order['order_status'] == 'returned' ? 'selected' : '' ?>>ສິນຄ້າຕີກັບ</option>
                        <option value="cancelled" <?= $order['order_status'] == 'cancelled' ? 'selected' : '' ?>>ຍົກເລີກ</option>
                    </select>
                </div>

                <!-- Payment Method (Show if not pending) -->
                <div class="form-group" id="paymentMethodGroup"
                    style="display:<?= $order['payment_status'] == 'pending' ? 'none' : 'block' ?>">
                    <label>ວິທີຊໍາລະ</label>
                    <div class="radio-group">
                        <label class="radio-label">
                            <input type="radio" name="paymentMethod" value="cash"
                                <?= (stripos($order['notes'], 'Payment: cash') !== false) || !stripos($order['notes'], 'Payment: transfer') ? 'checked' : '' ?>> ເງິນສົດ
                        </label>
                        <label class="radio-label">
                            <input type="radio" name="paymentMethod" value="transfer"
                                <?= stripos($order['notes'], 'Payment: transfer') !== false ? 'checked' : '' ?>> ໂອນ
                        </label>
                    </div>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>ບໍລິສັດຂົນສົ່ງ</label>
                    <div class="input-with-btn">
                        <select id="shippingCompany" class="form-control" onchange="filterBranches()">
                            <option value="">-- ເລືອກ --</option>
                            <?php foreach ($shippingCompanies as $sc): ?>
                                <option value="<?= $sc['SC_id'] ?>" data-int-id="<?= $sc['id'] ?>"
                                    <?= ($sc['id'] == $order['shipping_company_id_int']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($sc['company_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <button type="button" class="quick-add-btn" onclick="showQuickAddCompany()"
                            title="ເພີ່ມບໍລິສັດໃໝ່">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <div id="quickAddCompanyForm" class="quick-add-form" style="display:none;">
                        <input type="text" id="newCompanyName" class="form-control" placeholder="ຊື່ບໍລິສັດໃໝ່">
                        <div class="quick-add-actions">
                            <button type="button" class="btn-quick-save" onclick="saveQuickCompany()">
                                <i class="fas fa-check"></i> ບັນທຶກ
                            </button>
                            <button type="button" class="btn-quick-cancel" onclick="hideQuickAddCompany()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label>ສາຂາຂົນສົ່ງ</label>
                    <div class="input-with-btn">
                        <select id="shippingBranch" class="form-control">
                            <option value="">-- ເລືອກ --</option>
                            <?php foreach ($shippingBranches as $sb): ?>
                                <option value="<?= $sb['SB_id'] ?>" data-company="<?= $sb['company_id'] ?>"
                                    <?= ($sb['id'] == $order['shipping_branch_id_int']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($sb['branch_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <button type="button" class="quick-add-btn" onclick="showQuickAddBranch()" title="ເພີ່ມສາຂາໃໝ່">
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <div id="quickAddBranchForm" class="quick-add-form" style="display:none;">
                        <input type="text" id="newBranchName" class="form-control" placeholder="ຊື່ສາຂາໃໝ່">
                        <div class="quick-add-actions">
                            <button type="button" class="btn-quick-save" onclick="saveQuickBranch()">
                                <i class="fas fa-check"></i> ບັນທຶກ
                            </button>
                            <button type="button" class="btn-quick-cancel" onclick="hideQuickAddBranch()">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>  
            <div class="form-row">

                <div class="form-group" id="depositGroup"
                    style="display:<?= $order['payment_status'] == 'partial' ? 'block' : 'none' ?>">
                    <label>ຈຳນວນມັດຈຳ</label>
                    <div class="input-group-custom">
                        <input type="number" id="depositAmount" class="form-control"
                            value="<?= $order['deposit_amount'] ?? 0 ?>" oninput="updateSummary()">
                        <span class="currency-addon"><?= htmlspecialchars($currencySymbol) ?></span>
                    </div>
                </div>
            </div>

            <!-- COD Options -->
            <div class="form-group" style="margin-bottom:15px">
                <label>ຮູບແບບຄ່າຝາກ (COD)</label>
                <div class="radio-group" id="codTypeGroup">
                    <label class="radio-label">
                        <input type="radio" name="codType" value="ຫັກ COD" <?= strpos($order['notes'], 'ຫັກ COD') !== false ? 'checked' : '' ?>> ຫັກ COD
                    </label>
                    <label class="radio-label">
                        <input type="radio" name="codType" value="ຄ່າຝາກປາຍທາງ" <?= strpos($order['notes'], 'ຄ່າຝາກປາຍທາງ') !== false ? 'checked' : '' ?>> ຄ່າຝາກປາຍທາງ
                    </label>
                    <label class="radio-label">
                        <input type="radio" name="codType" value="" <?= (strpos($order['notes'], 'ຫັກ COD') === false && strpos($order['notes'], 'ຄ່າຝາກປາຍທາງ') === false) ? 'checked' : '' ?>> ບໍ່ລະບຸ
                    </label>
                </div>
            </div>

            <div class="form-group">
                <label>ໝາຍເຫດ</label>
                <textarea id="notes" class="form-control" rows="2"
                    placeholder="ໝາຍເຫດ..."><?= htmlspecialchars($order['notes'] ?? '') ?></textarea>
            </div>
        </div>

        <!-- Summary -->
        <div class="cart-summary">
            <div class="summary-row">
                <span>ລວມຍ່ອຍ:</span>
                <span id="subtotal">₭0</span>
            </div>
            <div class="summary-row">
                <span>ສ່ວນຫຼຸດ:</span>
                <div class="input-group-custom">
                    <input type="number" id="discountAmount" class="price-input form-control"
                        value="<?= $order['discount_amount'] ?? 0 ?>" oninput="updateSummary()">
                    <span class="currency-addon"><?= htmlspecialchars($currencySymbol) ?></span>
                </div>
            </div>

            <div class="summary-row refund-row" id="refundRow"
                style="display:<?= in_array($order['payment_status'], ['partial_refund', 'refunded']) ? 'flex' : 'none' ?>">
                <span>ການຄືນເງິນ:</span>
                <span id="refundDisplay">-<?= $currencySymbol ?><?= number_format($order['refund_amount'] ?? 0) ?></span>
            </div>

            <div class="summary-row vat-row" id="vatRow"
                style="display:<?= ($order['vat_enabled'] ?? 0) ? 'flex' : 'none' ?>">
                <span>VAT (10%):</span>
                <span id="vatAmount">₭0</span>
            </div>
            <div class="summary-row total-row">
                <span>ຍອດລວມ:</span>
                <span id="grandTotal">₭0</span>
            </div>
            <div class="currency-note" id="currencyNote">ລາຄາເປັນ: ກີບລາວ</div>
        </div>

        <div class="action-buttons">
            <a href="<?= htmlspecialchars($return_url) ?>" class="btn-pro secondary"><i class="fas fa-times"></i> ຍົກເລີກ</a>
            <button type="button" class="btn-pro primary" id="saveBtn" onclick="saveOrder()">
                <i class="fas fa-save"></i> ບັນທຶກ
            </button>
        </div>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>

<script>
    // ===== State =====
    let cart = [];
    let cartModified = false; // Track if cart has been modified
    const orderId = '<?= addslashes($order_id) ?>';
    const orderIntId = <?= intval($order['id']) ?>;
    const originalItems = <?= json_encode($orderItems) ?>;
    const allBranches = <?= json_encode($shippingBranches) ?>;
    const shopId = <?= $shop_id ?>;
    const returnUrl = '<?= addslashes($return_url) ?>';

    // Original order values from database (for display reference)
    const originalOrderData = {
        subtotal: <?= floatval($order['subtotal'] ?? 0) ?>,
        subtotal_lak: <?= floatval($order['subtotal_lak'] ?? 0) ?>,
        discount: <?= floatval($order['discount_amount'] ?? 0) ?>,
        discount_lak: <?= floatval($order['discount_lak'] ?? 0) ?>,
        vat_amount: <?= floatval($order['tax_amount'] ?? 0) ?>,
        vat_amount_lak: <?= floatval($order['vat_amount_lak'] ?? 0) ?>,
        total: <?= floatval($order['total_amount'] ?? 0) ?>,
        total_lak: <?= floatval($order['total_amount_lak'] ?? 0) ?>
    };

    // Currency state
    let currentCurrency = {
        id: '<?= addslashes($order['currencys_id'] ?? 'CUR_1') ?>',
        symbol: '<?= addslashes($currencySymbol) ?>',
        rate: <?= floatval($currencyRate) ?>
    };

    // Original currency info (for rate protection when switching back)
    const originalCurrencyId = '<?= addslashes($order['currencys_id'] ?? 'CUR_1') ?>';
    const originalCurrencyRate = <?= floatval($currencyRate) ?>;

    // Base values in LAK
    let baseDiscountLAK = 0;
    let baseDepositLAK = 0;

    // VAT state
    let vatEnabled = <?= ($order['vat_enabled'] ?? 0) ? 'true' : 'false' ?>;
 
    // Initialize cart from existing order items
    document.addEventListener('DOMContentLoaded', function () {
        if (originalItems && originalItems.length > 0) {
            originalItems.forEach(item => {
                // ใช้ current_price จาก products table (ราคา LAK เสมอ) เป็น base price
                // ถ้าไม่มี current_price ให้คำนวณจาก unit_price * rate
                const basePriceLAK = parseFloat(item.current_price) ||
                    (parseFloat(item.unit_price) * (currentCurrency.rate || 1));
                cart.push({
                    id: item.product_id,
                    name: item.display_name || item.product_name || 'ສິນຄ້າ',
                    basePriceLAK: basePriceLAK,
                    price: convertPrice(basePriceLAK), // แปลงเป็นสกุลเงินปัจจุบัน
                    quantity: parseFloat(item.quantity) || 1,
                    originalQty: parseFloat(item.quantity) || 0
                });
            });
        }
        renderCart();
        // Initialize base values
        // Initialize base values from LAK (database source of truth) to prevent rounding drift
        if (originalOrderData.discount_lak) {
            baseDiscountLAK = parseFloat(originalOrderData.discount_lak);
        } else {
            baseDiscountLAK = (parseFloat(document.getElementById('discountAmount').value) || 0) * currentCurrency.rate;
        }

        if (originalOrderData.deposit_lak) { // Assuming deposit_lak might exist or we can infer it
             // Actually originalOrderData might verify structure later. For safety:
             // If we are in LAK mode, depositAmount is LAK. If not, converting back from display might be safe ENOUGH for deposit, 
             // but ideally we should track deposit LAK. 
             // Let's stick to the plan: use originalOrderData LAK values if possible.
             baseDepositLAK = (parseFloat(document.getElementById('depositAmount').value) || 0) * currentCurrency.rate;
        } else {
             baseDepositLAK = (parseFloat(document.getElementById('depositAmount').value) || 0) * currentCurrency.rate;
        }

        updateSummary();
        updateCurrencyNote();

        // Initialize product card prices in order's currency
        updateAllPrices();

        // Event listeners
        document.getElementById('searchProduct').addEventListener('input', filterProducts);
        document.getElementById('categoryFilter').addEventListener('change', filterProducts);

        // Input listeners to update base values
        document.getElementById('discountAmount').addEventListener('input', function () {
            baseDiscountLAK = (parseFloat(this.value) || 0) * currentCurrency.rate;
            cartModified = true; // Mark as modified when user changes discount
            updateSummary();
        });
        document.getElementById('depositAmount').addEventListener('input', function () {
            baseDepositLAK = (parseFloat(this.value) || 0) * currentCurrency.rate;
            cartModified = true; // Mark as modified when user changes deposit
            updateSummary();
        });

        document.getElementById('currencySelect').addEventListener('change', function () {
            const opt = this.options[this.selectedIndex];
            const selectedCurrencyId = this.value;
            
            // Check if switching back to original currency
            // If yes, use original rate to prevent accidental rate changes
            let rateToUse;
            if (selectedCurrencyId === originalCurrencyId) {
                rateToUse = originalCurrencyRate;
            } else {
                rateToUse = parseFloat(opt.dataset.rate) || 1;
            }
            
            currentCurrency = {
                id: selectedCurrencyId,
                symbol: opt.dataset.symbol,
                rate: rateToUse
            };

            // If cart not modified, update discount display from originalOrderData
            if (!cartModified) {
                let discountDisplay;
                if (rateToUse === 1 || selectedCurrencyId === 'CUR_1') {
                    // LAK - use discount_lak
                    discountDisplay = originalOrderData.discount_lak;
                } else if (selectedCurrencyId === originalCurrencyId) {
                    // Original currency - use original discount
                    discountDisplay = originalOrderData.discount;
                } else {
                    // Different foreign currency - convert from discount_lak
                    discountDisplay = Math.ceil(originalOrderData.discount_lak / rateToUse);
                }
                document.getElementById('discountAmount').value = discountDisplay;
            }

            updateAllPrices();
            updateSummary();
            updateCurrencyNote();
        });
    });

    // ===== Currency Functions =====
    function convertPrice(priceLAK) {
        if (currentCurrency.rate === 1) return priceLAK;
        const rawPrice = priceLAK / currentCurrency.rate;
        // For foreign currencies, round up to prevent loss
        return roundUpPrice(rawPrice);
    }

    // Round up price for foreign currencies to prevent selling below cost
    function roundUpPrice(price) {
        // CUR_1 = LAK (no rounding needed)
        if (currentCurrency.id === 'CUR_1' || currentCurrency.rate === 1) {
            return price;
        }
        // CUR_3 = USD (round up to 2 decimal places)
        if (currentCurrency.id === 'CUR_3') {
            return Math.ceil(price * 100) / 100;
        }
        // THB and others: round up to integer
        return Math.ceil(price);
    }

    // Get original price without rounding (for tip calculation)
    function convertPriceRaw(priceLAK) {
        return currentCurrency.rate === 1 ? priceLAK : priceLAK / currentCurrency.rate;
    }

    function updateAllPrices() {
        // Update product card prices
        document.querySelectorAll('.product-card .price').forEach(el => {
            const basePrice = parseFloat(el.dataset.basePrice);
            el.textContent = formatNumber(convertPrice(basePrice)) + ' ' + currentCurrency.symbol;
            el.className = 'price ' + getCurrencyColorClass();
        });

        // Update modal input currency symbols
        document.querySelectorAll('.currency-addon').forEach(el => {
            el.textContent = currentCurrency.symbol;
        });

        // Update cart item prices
        cart.forEach(item => {
            if (item.basePriceLAK) {
                item.price = convertPrice(item.basePriceLAK);
            }
        });

        // Only update inputs if cart was modified (user changed discount values)
        // Otherwise currency change handler already set the correct values from database
        if (cartModified) {
            document.getElementById('discountAmount').value = parseFloat(convertPrice(baseDiscountLAK)).toFixed(2).replace(/\.00$/, '');
            document.getElementById('depositAmount').value = parseFloat(convertPrice(baseDepositLAK)).toFixed(2).replace(/\.00$/, '');
        }

        renderCart();
    }

    function getCurrencyColorClass() {
        if (currentCurrency.symbol === '₭') return 'currency-lak';
        if (currentCurrency.symbol === '฿') return 'currency-thb';
        if (currentCurrency.symbol === '$') return 'currency-usd';
        return '';
    }

    function updateCurrencyNote() {
        const select = document.getElementById('currencySelect');
        const selectedText = select.options[select.selectedIndex]?.text || 'ກີບລາວ';
        document.getElementById('currencyNote').textContent = 'ລາຄາເປັນ: ' + selectedText;
    }

    // ===== Product Filter =====
    function filterProducts() {
        const search = document.getElementById('searchProduct').value.toLowerCase();
        const category = document.getElementById('categoryFilter').value;

        document.querySelectorAll('.product-card').forEach(card => {
            const name = card.dataset.name.toLowerCase();
            const cat = card.dataset.category;
            const matchSearch = name.includes(search);
            const matchCat = !category || cat == category;
            card.style.display = matchSearch && matchCat ? '' : 'none';
        });
    }

    // ===== Cart Functions =====
    function addToCart(product) {
        const existing = cart.find(item => item.id == product.PR_id || item.id == product.id);
        if (existing) {
            existing.quantity++;
        } else {
            const basePriceLAK = parseFloat(product.selling_price);
            cart.push({
                id: product.PR_id || product.id,
                name: product.product_name,
                basePriceLAK: basePriceLAK,
                price: convertPrice(basePriceLAK),
                quantity: 1,
                originalQty: 0
            });
        }
        cartModified = true;
        renderCart();
        updateSummary();
    }

    function removeFromCart(index) {
        cart.splice(index, 1);
        cartModified = true;
        renderCart();
        updateSummary();
    }

    function updateQuantity(index, delta) {
        cart[index].quantity += delta;
        cartModified = true;
        if (cart[index].quantity < 1) {
            removeFromCart(index);
        } else {
            renderCart();
            updateSummary();
        }
    }

    function clearCart() {
        if (cart.length === 0) return;
        Swal.fire({
            title: 'ລ້າງລາຍການທັງໝົດ?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            confirmButtonText: 'ລ້າງ',
            cancelButtonText: 'ຍົກເລີກ'
        }).then(r => { if (r.isConfirmed) { cart = []; renderCart(); updateSummary(); } });
    }

    function renderCart() {
        const container = document.getElementById('cartItems');
        const colorClass = getCurrencyColorClass();

        if (cart.length === 0) {
            container.innerHTML = '<div style="text-align:center;padding:40px;color:var(--text-secondary)"><i class="fas fa-shopping-cart" style="font-size:32px;opacity:.4"></i><p>ບໍ່ມີສິນຄ້າ</p></div>';
            return;
        }

        container.innerHTML = cart.map((item, index) => `
            <div class="cart-item">
                <div class="cart-item-header">
                    <div class="cart-item-name">${escapeHtml(item.name)}</div>
                    <button class="cart-item-remove" onclick="removeFromCart(${index})"><i class="fas fa-times"></i></button>
                </div>
                <div class="cart-item-controls">
                    <div class="qty-controls">
                        <button class="qty-btn" onclick="updateQuantity(${index}, -1)">-</button>
                        <span class="qty-display">${item.quantity}</span>
                        <button class="qty-btn" onclick="updateQuantity(${index}, 1)">+</button>
                    </div>
                    <div class="item-total ${colorClass}">${formatNumber(item.price * item.quantity)} ${currentCurrency.symbol}</div>
                </div>
            </div>
        `).join('');
    }

    function updateSummary() {
        let subtotal, discount, vatAmount, total;
        const refundAmount = parseFloat(document.getElementById('refundAmount').value) || 0;
        const colorClass = getCurrencyColorClass();

        // If cart was NOT modified, use original database values
        if (!cartModified) {
            // LAK currency - always use _lak values directly
            if (currentCurrency.rate === 1 || currentCurrency.id === 'CUR_1') {
                subtotal = originalOrderData.subtotal_lak;
                discount = originalOrderData.discount_lak;
                vatAmount = originalOrderData.vat_amount_lak;
                total = originalOrderData.total_lak - refundAmount;
            } 
            // Original currency - use display values from database
            else if (currentCurrency.id === originalCurrencyId) {
                subtotal = originalOrderData.subtotal;
                discount = originalOrderData.discount;
                vatAmount = originalOrderData.vat_amount;
                total = originalOrderData.total - refundAmount;
            }
            // Different foreign currency - calculate from LAK then convert (same as pos.php)
            else {
                // คำนวณจาก LAK ก่อน แล้วค่อยแปลง - เหมือนกับ pos.php
                const subtotalLAK = cart.reduce((sum, item) => sum + ((item.basePriceLAK || item.price) * item.quantity), 0);
                const discountInput = parseFloat(document.getElementById('discountAmount').value) || 0;
                const discountLAK = discountInput * currentCurrency.rate; // แปลง discount กลับไปเป็น LAK
                const taxableAmountLAK = subtotalLAK - discountLAK;
                const vatAmountLAK = vatEnabled ? taxableAmountLAK * 0.1 : 0;
                const totalLAK = taxableAmountLAK + vatAmountLAK;
                
                // แปลงเป็นสกุลเงินปัจจุบัน
                subtotal = convertPrice(subtotalLAK);
                discount = discountInput;
                vatAmount = convertPrice(vatAmountLAK);
                total = convertPrice(totalLAK) - refundAmount;
            }
        } else {
            // Cart was modified - calculate from LAK then convert (same as pos.php)
            // คำนวณจาก LAK ก่อน แล้วค่อยแปลง - เหมือนกับ pos.php
            const subtotalLAK = cart.reduce((sum, item) => sum + ((item.basePriceLAK || item.price) * item.quantity), 0);
            const discountInput = parseFloat(document.getElementById('discountAmount').value) || 0;
            const discountLAK = currentCurrency.rate === 1 ? discountInput : discountInput * currentCurrency.rate;
            const taxableAmountLAK = subtotalLAK - discountLAK;
            const vatAmountLAK = vatEnabled ? taxableAmountLAK * 0.1 : 0;
            const totalLAK = taxableAmountLAK + vatAmountLAK;
            
            // แปลงเป็นสกุลเงินปัจจุบัน
            if (currentCurrency.rate === 1) {
                subtotal = subtotalLAK;
                discount = discountInput;
                vatAmount = vatAmountLAK;
                total = totalLAK - refundAmount;
            } else {
                subtotal = convertPrice(subtotalLAK);
                discount = discountInput;
                vatAmount = convertPrice(vatAmountLAK);
                total = convertPrice(totalLAK) - refundAmount;
            }
        }

        const subtotalEl = document.getElementById('subtotal');
        const grandTotalEl = document.getElementById('grandTotal');
        const vatAmountEl = document.getElementById('vatAmount');
        const refundDisplayEl = document.getElementById('refundDisplay');

        subtotalEl.textContent = formatNumber(subtotal) + ' ' + currentCurrency.symbol;
        subtotalEl.className = colorClass;

        vatAmountEl.textContent = formatNumber(vatAmount) + ' ' + currentCurrency.symbol;

        // Update refund display
        if (refundDisplayEl) {
            refundDisplayEl.textContent = '-' + formatNumber(refundAmount) + ' ' + currentCurrency.symbol;
        }

        grandTotalEl.textContent = formatTotal(total) + ' ' + currentCurrency.symbol;
        grandTotalEl.className = colorClass;
    }

    // Get total in LAK (base currency) for tip calculation
    function getTotalLAK() {
        const subtotalLAK = cart.reduce((sum, item) => sum + ((item.basePriceLAK || item.price) * item.quantity), 0);
        const discountInput = parseFloat(document.getElementById('discountAmount').value) || 0;
        const discountLAK = currentCurrency.rate === 1 ? discountInput : discountInput * currentCurrency.rate;
        const vatAmountLAK = vatEnabled ? (subtotalLAK - discountLAK) * 0.1 : 0;
        return subtotalLAK - discountLAK + vatAmountLAK;
    }

    // ===== VAT Toggle =====
    function toggleVAT() {
        vatEnabled = !vatEnabled;
        const btn = document.getElementById('vatToggle');
        const vatRow = document.getElementById('vatRow');

        if (vatEnabled) {
            btn.classList.add('active');
            btn.innerHTML = '<i class="fas fa-percent"></i> VAT 10%';
            vatRow.style.display = 'flex';
        } else {
            btn.classList.remove('active');
            btn.innerHTML = '<i class="fas fa-percent"></i> ບໍ່ເອົາ VAT';
            vatRow.style.display = 'none';
        }
        updateSummary();
    }

    // ===== Quick Add Customer =====
    function showQuickAddCustomer() {
        document.getElementById('quickAddCustomerForm').style.display = 'block';
        document.getElementById('newCustomerName').focus();
    }

    function hideQuickAddCustomer() {
        document.getElementById('quickAddCustomerForm').style.display = 'none';
        document.getElementById('newCustomerName').value = '';
        document.getElementById('newCustomerPhone').value = '';
    }

    function saveQuickCustomer() {
        const name = document.getElementById('newCustomerName').value.trim();
        const phone = document.getElementById('newCustomerPhone').value.trim();
        if (!name || !phone) {
            Swal.fire({ icon: 'warning', title: 'ກະລຸນາປ້ອນຊື່ ແລະ ເບີໂທ', timer: 2000, showConfirmButton: false });
            return;
        }

        fetch('api/pos_quick_add.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ type: 'customer', name: name, phone: phone, shop_id: shopId })
        })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    const select = document.getElementById('customerId');
                    const option = document.createElement('option');
                    option.value = data.id;
                    option.textContent = name + ' - ' + phone;
                    option.selected = true;
                    select.appendChild(option);

                    hideQuickAddCustomer();

                    Swal.fire({ icon: 'success', title: 'ເພີ່ມລູກຄ້າແລ້ວ', timer: 1500, showConfirmButton: false });
                } else {
                    throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
                }
            })
            .catch(err => {
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: err.message });
            });
    }

    // Previous payment status to revert if user cancels
    let previousPaymentStatus = '<?= $order['payment_status'] ?? 'pending' ?>';
    const orderTotalAmount = <?= $order['total_amount'] ?? 0 ?>;
    const orderTotalAmountLAK = <?= $order['total_amount_lak'] ?? 0 ?>;

    function handlePaymentStatusChange(newStatus) {
        // Show/hide deposit field
        document.getElementById('depositGroup').style.display = newStatus === 'partial' ? 'block' : 'none';
        // Show/hide payment method
        document.getElementById('paymentMethodGroup').style.display = newStatus === 'pending' ? 'none' : 'block';
        
        // Handle refund options
        if (newStatus === 'partial_refund') {
            showPartialRefundDialog();
        } else if (newStatus === 'refunded') {
            showFullRefundDialog();
        } else {
            // Hide refund display for other statuses
            document.getElementById('refundAmountDisplay').style.display = 'none';
            document.getElementById('refundRow').style.display = 'none';
            document.getElementById('refundAmount').value = 0;
            updateSummary();
            previousPaymentStatus = newStatus;
        }
    }

    function showPartialRefundDialog() {
        // Step 1: Ask for confirmation
        Swal.fire({
            title: 'ທ່ານຕ້ອງການ ຄືນເງີນບາງສ່ວນໃຫ້ລູກຄ້າ ບໍ?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: '<i class="fas fa-check"></i> ແມ່ນ',
            cancelButtonText: '<i class="fas fa-times"></i> ຍົກເລີກ',
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d',
            scrollbarPadding: false
        }).then((result) => {
            if (result.isConfirmed) {
                // Step 2: Show input form
                Swal.fire({
                    title: 'ຂໍ້ມູນຄືນເງິນບາງສ່ວນ',
                    html: `
                        <div style="text-align:left;margin-bottom:15px;">
                            <label style="display:block;margin-bottom:5px;font-weight:600;">ຈຳນວນເງິນທີ່ຄືນ <span style="color:red">*</span></label>
                            <input type="number" id="swal-refund-amount" class="swal2-input" placeholder="ປ້ອນຈຳນວນເງິນ" style="margin:0;width:100%;">
                        </div>
                        <div style="text-align:left;">
                            <label style="display:block;margin-bottom:5px;font-weight:600;">ເຫດຜົນທີ່ຄືນ <span style="color:red">*</span></label>
                            <textarea id="swal-refund-reason" class="swal2-textarea" placeholder="ລະບຸເຫດຜົນ..." style="margin:0;width:100%;min-height:80px;"></textarea>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check"></i> ສຳເລັດ',
                    cancelButtonText: '<i class="fas fa-times"></i> ຍົກເລີກ',
                    confirmButtonColor: '#28a745',
                    cancelButtonColor: '#dc3545',
                    scrollbarPadding: false,
                    preConfirm: () => {
                        const amount = document.getElementById('swal-refund-amount').value;
                        const reason = document.getElementById('swal-refund-reason').value;
                        
                        if (!amount || parseFloat(amount) <= 0) {
                            Swal.showValidationMessage('ກະລຸນາປ້ອນຈຳນວນເງິນທີ່ຄືນ');
                            return false;
                        }
                        if (!reason.trim()) {
                            Swal.showValidationMessage('ກະລຸນາລະບຸເຫດຜົນ');
                            return false;
                        }
                        return { amount: parseFloat(amount), reason: reason.trim() };
                    }
                }).then((inputResult) => {
                    if (inputResult.isConfirmed && inputResult.value) {
                        // Set refund amount and reason
                        document.getElementById('refundAmount').value = inputResult.value.amount;
                        document.getElementById('refundAmountValue').textContent = currentCurrency.symbol + formatNumber(inputResult.value.amount);
                        document.getElementById('refundAmountDisplay').style.display = 'block';
                        document.getElementById('refundRow').style.display = 'flex';
                        
                        // Update notes with reason
                        const notesField = document.getElementById('notes');
                        notesField.value = 'ເຫດຜົນຄືນເງິນ: ' + inputResult.value.reason;
                        
                        updateSummary();
                        previousPaymentStatus = 'partial_refund';
                        
                        Swal.fire({
                            icon: 'success',
                            title: 'ບັນທຶກຂໍ້ມູນຄືນເງິນແລ້ວ',
                            text: 'ກະລຸນາກົດ "ບັນທຶກ" ເພື່ອຢືນຢັນການປ່ຽນແປງ',
                            timer: 2000,
                            showConfirmButton: false,
                            scrollbarPadding: false
                        });
                    } else {
                        // User cancelled input - revert to previous status
                        document.getElementById('paymentStatus').value = previousPaymentStatus;
                    }
                });
            } else {
                // User cancelled confirmation - revert to previous status
                document.getElementById('paymentStatus').value = previousPaymentStatus;
            }
        });
    }

    function showFullRefundDialog() {
        const displayAmount = currentCurrency.rate === 1 ? orderTotalAmountLAK : orderTotalAmount;
        
        // Step 1: Ask for confirmation
        Swal.fire({
            title: 'ທ່ານຕ້ອງການ ຄືນເງີນເຕັມຈຳນວນໃຫ້ລູກຄ້າ ບໍ?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: '<i class="fas fa-check"></i> ແມ່ນ',
            cancelButtonText: '<i class="fas fa-times"></i> ຍົກເລີກ',
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            scrollbarPadding: false
        }).then((result) => {
            if (result.isConfirmed) {
                // Step 2: Show input form with amount display
                Swal.fire({
                    title: 'ຂໍ້ມູນຄືນເງິນເຕັມຈຳນວນ',
                    html: `
                        <div style="text-align:center;margin-bottom:20px;padding:15px;background:rgba(220,53,69,0.1);border-radius:10px;">
                            <p style="margin:0;color:#666;">ຍອດທີ່ຈະຄືນ:</p>
                            <h2 style="margin:10px 0;color:#dc3545;">${currentCurrency.symbol}${formatNumber(displayAmount)}</h2>
                        </div>
                        <div style="text-align:left;">
                            <label style="display:block;margin-bottom:5px;font-weight:600;">ເຫດຜົນທີ່ຄືນ <span style="color:red">*</span></label>
                            <textarea id="swal-refund-reason" class="swal2-textarea" placeholder="ລະບຸເຫດຜົນ..." style="margin:0;width:100%;min-height:80px;"></textarea>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check"></i> ສຳເລັດ',
                    cancelButtonText: '<i class="fas fa-times"></i> ຍົກເລີກ',
                    confirmButtonColor: '#dc3545',
                    cancelButtonColor: '#6c757d',
                    scrollbarPadding: false,
                    preConfirm: () => {
                        const reason = document.getElementById('swal-refund-reason').value;
                        if (!reason.trim()) {
                            Swal.showValidationMessage('ກະລຸນາລະບຸເຫດຜົນ');
                            return false;
                        }
                        return { reason: reason.trim() };
                    }
                }).then((inputResult) => {
                    if (inputResult.isConfirmed && inputResult.value) {
                        // Set full refund amount
                        const refundAmount = currentCurrency.rate === 1 ? orderTotalAmountLAK : orderTotalAmount;
                        document.getElementById('refundAmount').value = refundAmount;
                        document.getElementById('refundAmountValue').textContent = currentCurrency.symbol + formatNumber(refundAmount);
                        document.getElementById('refundAmountDisplay').style.display = 'block';
                        document.getElementById('refundRow').style.display = 'flex';
                        
                        // Update notes with reason
                        const notesField = document.getElementById('notes');
                        notesField.value = 'ເຫດຜົນຄືນເງິນເຕັມຈຳນວນ: ' + inputResult.value.reason;
                        
                        updateSummary();
                        previousPaymentStatus = 'refunded';
                        
                        Swal.fire({
                            icon: 'success',
                            title: 'ບັນທຶກຂໍ້ມູນຄືນເງິນແລ້ວ',
                            text: 'ກະລຸນາກົດ "ບັນທຶກ" ເພື່ອຢືນຢັນການປ່ຽນແປງ',
                            timer: 2000,
                            showConfirmButton: false,
                            scrollbarPadding: false
                        });
                    } else {
                        // User cancelled input - revert to previous status
                        document.getElementById('paymentStatus').value = previousPaymentStatus;
                    }
                });
            } else {
                // User cancelled confirmation - revert to previous status
                document.getElementById('paymentStatus').value = previousPaymentStatus;
            }
        });
    }


    function toggleDepositField() {
        const status = document.getElementById('paymentStatus').value;
        document.getElementById('depositGroup').style.display = status === 'partial' ? 'block' : 'none';

        // Show/Hide Payment Method
        document.getElementById('paymentMethodGroup').style.display = status === 'pending' ? 'none' : 'block';
    }


    function filterBranches() {
        const companySelect = document.getElementById('shippingCompany');
        const branchSelect = document.getElementById('shippingBranch');
        const selectedCompanyId = companySelect.value;

        branchSelect.innerHTML = '<option value="">-- ເລືອກ --</option>';

        if (selectedCompanyId) {
            allBranches.filter(b => b.company_id == selectedCompanyId).forEach(b => {
                branchSelect.innerHTML += `<option value="${b.SB_id}">${b.branch_name}</option>`;
            });
        }
    }

    // ===== Quick Add Shipping Company =====
    function showQuickAddCompany() {
        document.getElementById('quickAddCompanyForm').style.display = 'block';
        document.getElementById('newCompanyName').focus();
    }

    function hideQuickAddCompany() {
        document.getElementById('quickAddCompanyForm').style.display = 'none';
        document.getElementById('newCompanyName').value = '';
    }

    function saveQuickCompany() {
        const name = document.getElementById('newCompanyName').value.trim();
        if (!name) {
            Swal.fire({ icon: 'warning', title: 'ກະລຸນາປ້ອນຊື່ບໍລິສັດ', timer: 2000, showConfirmButton: false });
            return;
        }

        fetch('api/pos_quick_add.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ type: 'company', name: name, shop_id: shopId })
        })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    const select = document.getElementById('shippingCompany');
                    const option = document.createElement('option');
                    option.value = data.id;
                    option.textContent = name;
                    option.selected = true;
                    select.appendChild(option);

                    hideQuickAddCompany();
                    filterBranches();

                    Swal.fire({ icon: 'success', title: 'ເພີ່ມບໍລິສັດແລ້ວ', timer: 1500, showConfirmButton: false });
                } else {
                    throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
                }
            })
            .catch(err => {
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: err.message });
            });
    }

    // ===== Quick Add Shipping Branch =====
    function showQuickAddBranch() {
        const companyId = document.getElementById('shippingCompany').value;
        if (!companyId) {
            Swal.fire({ icon: 'warning', title: 'ກະລຸນາເລືອກບໍລິສັດກ່ອນ', timer: 2000, showConfirmButton: false });
            return;
        }
        document.getElementById('quickAddBranchForm').style.display = 'block';
        document.getElementById('newBranchName').focus();
    }

    function hideQuickAddBranch() {
        document.getElementById('quickAddBranchForm').style.display = 'none';
        document.getElementById('newBranchName').value = '';
    }

    function saveQuickBranch() {
        const name = document.getElementById('newBranchName').value.trim();
        const companySelect = document.getElementById('shippingCompany');
        const companyId = companySelect.options[companySelect.selectedIndex]?.dataset?.intId || companySelect.value;

        if (!name) {
            Swal.fire({ icon: 'warning', title: 'ກະລຸນາປ້ອນຊື່ສາຂາ', timer: 2000, showConfirmButton: false });
            return;
        }

        fetch('api/pos_quick_add.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ type: 'branch', name: name, company_id: companyId, shop_id: shopId })
        })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    const select = document.getElementById('shippingBranch');
                    const option = document.createElement('option');
                    option.value = data.id;
                    option.textContent = name;
                    option.selected = true;
                    select.appendChild(option);

                    allBranches.push({ id: data.id, SB_id: data.id, company_id: companySelect.value, branch_name: name });

                    hideQuickAddBranch();

                    Swal.fire({ icon: 'success', title: 'ເພີ່ມສາຂາແລ້ວ', timer: 1500, showConfirmButton: false });
                } else {
                    throw new Error(data.message || 'ເກີດຂໍ້ຜິດພາດ');
                }
            })
            .catch(err => {
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: err.message });
            });
    }

    // ===== Calculate Tip Data (Rounding Difference) =====
    function calculateTipData(totalLAK) {
        // Only calculate tip for foreign currencies
        if (currentCurrency.id === 'CUR_1' || currentCurrency.rate === 1) {
            return null; // No tip for LAK
        }

        // Calculate raw total (before rounding)
        const rawTotal = convertPriceRaw(totalLAK);
        // Calculate rounded total
        const roundedTotal = convertPrice(totalLAK);
        // Tip is the difference
        const tipForeign = roundedTotal - rawTotal;

        if (tipForeign <= 0) {
            return null; // No tip if no rounding occurred
        }

        // Convert tip to LAK
        const tipLAK = tipForeign * currentCurrency.rate;

        return {
            currencys_id: currentCurrency.id,
            currency_rate: currentCurrency.rate,
            original_total: rawTotal,
            rounded_total: roundedTotal,
            tip_amount_foreign: tipForeign,
            tip_amount_lak: tipLAK,
            calculation_note: `ยอดเดิม: ${rawTotal.toFixed(4)} ${currentCurrency.symbol} → ปัดเป็น: ${roundedTotal} ${currentCurrency.symbol} | Tip: ${tipForeign.toFixed(4)} ${currentCurrency.symbol} = ${tipLAK.toFixed(2)} ₭ (Rate: ${currentCurrency.rate})`
        };
    }

    // ===== Save Order =====
    function saveOrder() {
        if (cart.length === 0) {
            Swal.fire({ icon: 'warning', title: 'ກະລຸນາເພີ່ມສິນຄ້າ' });
            return;
        }

        const saveBtn = document.getElementById('saveBtn');
        saveBtn.disabled = true;
        saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ກຳລັງບັນທຶກ...';

        // Calculate LAK values separately to ensure precision and avoid rounding drift
        // 1. Subtotal LAK: sum of all items' base prices (which are LAK)
        const subtotalLAK = cart.reduce((sum, item) => sum + ((item.basePriceLAK || item.price) * item.quantity), 0);
        
        // 2. Discount LAK: use the baseDiscountLAK tracking variable
        const discountLAK = baseDiscountLAK;
        
        // 3. VAT LAK
        const vatAmountLAK = vatEnabled ? (subtotalLAK - discountLAK) * 0.1 : 0;
        
        // 4. Total LAK
        const totalLAK = subtotalLAK - discountLAK + vatAmountLAK;

        const data = {
            order_id: orderId,
            customer_id: document.getElementById('customerId').value,
            payment_status: document.getElementById('paymentStatus').value,
            order_status: document.getElementById('orderStatus').value,
            cod_type: document.querySelector('input[name="codType"]:checked')?.value || '',
            payment_method: document.querySelector('input[name="paymentMethod"]:checked')?.value || '',
            deposit_amount: parseFloat(document.getElementById('depositAmount').value) || 0,
            refund_amount: parseFloat(document.getElementById('refundAmount').value) || 0,
            shipping_company_id: document.getElementById('shippingCompany').value,
            shipping_branch_id: document.getElementById('shippingBranch').value,

            discount_amount: parseFloat(document.getElementById('discountAmount').value) || 0,
            notes: document.getElementById('notes').value,
            currencys_id: currentCurrency.id,
            currency_rate: currentCurrency.rate,
            vat_enabled: vatEnabled ? 1 : 0,
            items: cart.map(item => ({
                product_id: item.id,
                product_name: item.name,
                quantity: item.quantity,
                unit_price: item.price,
                unit_price_lak: item.basePriceLAK || item.price,
                original_qty: item.originalQty || 0
            })),
            // Calculate definitive LAK values for backend
            subtotal_lak: subtotalLAK,
            discount_lak: discountLAK,
            vat_amount_lak: vatAmountLAK,
            total_amount_lak: totalLAK,

            // Calculate foreign values (displayed values) for backend to ensure consistency
            subtotal: convertPrice(subtotalLAK),
            tax_amount: convertPrice(vatAmountLAK),
            total_amount: convertPrice(totalLAK) - parseFloat(document.getElementById('refundAmount').value || 0),

            // Tip data (rounding difference)
            tip_data: calculateTipData(totalLAK)
        };

        fetch('api/update_order_full.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        })
            .then(response => response.json())
            .then(result => {
                saveBtn.disabled = false;
                saveBtn.innerHTML = '<i class="fas fa-save"></i> ບັນທຶກ';

                if (result.status === 'success') {
                    Swal.fire({
                        icon: 'success',
                        title: 'ບັນທຶກສຳເລັດ!',
                        timer: 1500,
                        showConfirmButton: false
                    }).then(() => {
                        // Use returnUrl and append highlight params
                        const separator = returnUrl.includes('?') ? '&' : '?';
                        window.location.href = returnUrl + separator + 'highlight=1&order_id=' + orderId;
                    });
                } else {
                    Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: result.message });
                }
            })
            .catch(error => {
                saveBtn.disabled = false;
                saveBtn.innerHTML = '<i class="fas fa-save"></i> ບັນທຶກ';
                Swal.fire({ icon: 'error', title: 'ຜິດພາດ', text: 'ການເຊື່ອມຕໍ່ມີປັນຫາ' });
            });
    }

    // ===== Utility =====
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function formatNumber(num) {
        if (currentCurrency.id === 'CUR_3') {
            return new Intl.NumberFormat('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(num);
        }
        return new Intl.NumberFormat('en-US', { minimumFractionDigits: 0, maximumFractionDigits: 0 }).format(Math.round(num));
    }

    // Format grand total with rounding (to avoid confusion with small decimal differences)
    function formatTotal(num) {
        if (currentCurrency.id === 'CUR_3') {
            // USD: round to 2 decimal places
            return new Intl.NumberFormat('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(Math.round(num * 100) / 100);
        }
        if (currentCurrency.id === 'CUR_2') {
            // THB: round to integer
            return new Intl.NumberFormat('en-US', { minimumFractionDigits: 0, maximumFractionDigits: 0 }).format(Math.round(num));
        }
        // LAK: already integer
        return new Intl.NumberFormat('en-US', { minimumFractionDigits: 0, maximumFractionDigits: 0 }).format(Math.round(num));
    }
</script>