<?php
/**
 * AJAX endpoint for loading cancelled orders table body
 * Shows orders with: order_status = 'returned', 'cancelled' OR payment_status = 'refunded'
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo '<tr><td colspan="9" class="empty-state">Unauthorized</td></tr>';
    exit;
}

require_once dirname(__DIR__) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo '<tr><td colspan="9" class="empty-state">ບໍ່ພົບຂໍ້ມູນຮ້ານຄ້າ</td></tr>';
    exit;
}

// Get parameters
$search_query = isset($_GET['search']) ? trim($_GET['search']) : '';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$filter_type = isset($_GET['filter_type']) ? $_GET['filter_type'] : 'all';

// Build query
$sql = "SELECT 
            o.id,
            o.OR_id,
            o.order_type,
            o.order_date,
            o.order_status,
            o.payment_status,
            o.total_amount,
            c.name as customer_name,
            c.phone as customer_phone
        FROM orders o
        LEFT JOIN customers c ON o.customer_id = c.CU_id
        WHERE o.shop_id = ? AND o.deleted_at IS NULL
        AND (o.order_status IN ('cancelled', 'returned') OR o.payment_status = 'refunded')";

$params = [$shop_id];
$types = "i";

// Filter by type
if ($filter_type === 'cancelled') {
    $sql = str_replace(
        "AND (o.order_status IN ('cancelled', 'returned') OR o.payment_status = 'refunded')",
        "AND o.order_status = 'cancelled'",
        $sql
    );
} elseif ($filter_type === 'returned') {
    $sql = str_replace(
        "AND (o.order_status IN ('cancelled', 'returned') OR o.payment_status = 'refunded')",
        "AND o.order_status = 'returned'",
        $sql
    );
} elseif ($filter_type === 'refunded') {
    $sql = str_replace(
        "AND (o.order_status IN ('cancelled', 'returned') OR o.payment_status = 'refunded')",
        "AND o.payment_status = 'refunded'",
        $sql
    );
}

// Add search condition
if (!empty($search_query)) {
    $sql .= " AND (o.OR_id LIKE ? OR c.name LIKE ? OR c.phone LIKE ?)";
    $search_param = "%" . $search_query . "%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $types .= "sss";
}

// Add date filter
if (!empty($date_from)) {
    $sql .= " AND DATE(o.order_date) >= ?";
    $params[] = $date_from;
    $types .= "s";
}
if (!empty($date_to)) {
    $sql .= " AND DATE(o.order_date) <= ?";
    $params[] = $date_to;
    $types .= "s";
}

// Order by
$sql .= " ORDER BY o.created_at ASC";

// Execute query
$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

// Output rows
if ($result->num_rows > 0) {
    $count = 1;
    while ($row = $result->fetch_assoc()) {
        $order_type_class = '';
        $order_type_text = '-';

        switch ($row['order_type']) {
            case 'pos':
                $order_type_class = 'type-pos';
                $order_type_text = 'ໜ້າຮ້ານ';
                break;
            case 'online':
                $order_type_class = 'type-online';
                $order_type_text = 'ອອນລາຍ';
                break;
        }

        // Determine status to display
        $status_class = '';
        $status_text = '-';
        $data_status = '';

        if ($row['payment_status'] === 'refunded') {
            $status_class = 'status-refunded';
            $status_text = 'ຄືນເງິນ';
            $data_status = 'refunded';
        } elseif ($row['order_status'] === 'cancelled') {
            $status_class = 'status-cancelled';
            $status_text = 'ຍົກເລີກ';
            $data_status = 'cancelled';
        } elseif ($row['order_status'] === 'returned') {
            $status_class = 'status-returned';
            $status_text = 'ສິນຄ້າຕີກັບ';
            $data_status = 'returned';
        }

        $customer_name = htmlspecialchars($row['customer_name'] ?? '-');
        $customer_phone = htmlspecialchars($row['customer_phone'] ?? '-');

        echo '<tr data-order-id="' . htmlspecialchars($row['OR_id']) . '" data-status="' . $data_status . '">';
        echo '<td>' . $count . '</td>';
        echo '<td><span class="badge ' . $order_type_class . '">' . $order_type_text . '</span></td>';
        echo '<td>' . htmlspecialchars($row['OR_id']) . '</td>';
        echo '<td>' . date('d/m/Y H:i', strtotime($row['order_date'])) . '</td>';
        echo '<td title="' . $customer_name . '">' . (mb_strlen($customer_name) > 15 ? mb_substr($customer_name, 0, 13) . '…' : $customer_name) . '</td>';
        echo '<td>' . $customer_phone . '</td>';
        echo '<td><span class="badge ' . $status_class . '">' . $status_text . '</span></td>';
        echo '<td class="amount-cell">₭' . number_format($row['total_amount'] ?? 0, 0) . '</td>';
        echo '<td>';
        echo '<div class="action-btns">';
        echo '<button class="btn-icon view" onclick="viewOrder(\'' . $row['OR_id'] . '\')" title="ເບິ່ງ"><i class="fas fa-eye"></i></button>';
        echo '<button class="btn-icon restore" onclick="restoreOrder(\'' . $row['OR_id'] . '\')" title="ກູ້ຄືນ"><i class="fas fa-undo"></i></button>';
        echo '</div>';
        echo '</td>';
        echo '</tr>';
        $count++;
    }
} else {
    echo '<tr><td colspan="9" class="empty-state"><i class="fas fa-inbox" style="font-size:32px;opacity:.4;margin-bottom:12px;display:block"></i>ບໍ່ພົບຂໍ້ມູນ</td></tr>';
}

$stmt->close();
?>