/**
 * SPA Navigation System
 * Implements persistent layout by intercepting links and swapping content
 * Includes compatibility layer for legacy valid const/let redeclarations
 */

// Helper to run scripts immediately if DOM is already ready (for SPA)
window.spaOnLoad = function (callback) {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', callback);
    } else {
        callback();
    }
};

document.addEventListener('DOMContentLoaded', () => {
    // Only initialize once
    if (window.spaInitialized) return;
    window.spaInitialized = true;

    const contentSelector = '#app-content';
    const linkSelector = 'a:not([target="_blank"]):not([href^="#"]):not([data-no-spa])';

    // Intercept clicks
    document.addEventListener('click', (e) => {
        const link = e.target.closest(linkSelector);
        if (!link) return;

        // Check if internal link
        const href = link.getAttribute('href');
        if (!href || href.startsWith('http') && !href.includes(window.location.origin)) return;
        if (href.startsWith('javascript:')) return;

        e.preventDefault();
        navigateTo(href);
    });

    // Initial active menu update
    updateActiveMenu();

    // Handle Back/Forward buttons
    window.addEventListener('popstate', () => {
        loadContent(window.location.href);
    });

    async function navigateTo(url) {
        history.pushState(null, '', url);
        updateActiveMenu(); // Update immediately for responsiveness
        await loadContent(url);
    }

    // Active Menu Logic (Realtime)
    function updateActiveMenu() {
        const currentPage = location.pathname.split('/').pop() || 'index.php';

        // Remove active class from all links
        document.querySelectorAll('.sidebar .menu-item > a, .sidebar .submenu li a').forEach(link => {
            link.classList.remove('active');

            // Check if this link matches the current page
            // We use getAttribute('href') to get the raw value (e.g. 'pos.php')
            // instead of link.href which is absolute
            const linkHref = link.getAttribute('href');
            if (linkHref === currentPage || (currentPage === 'index.php' && linkHref === './')) {
                link.classList.add('active');

                // If it's a submenu item, ensure the parent menu is open
                const parentItem = link.closest('.menu-item');
                if (parentItem && link.closest('.submenu')) {
                    parentItem.classList.add('open');
                }
            }
        });
    }

    async function loadContent(url) {
        const container = document.querySelector(contentSelector);
        if (!container) {
            console.error('SPA: Content container not found');
            window.location.href = url; // Fallback
            return;
        }

        // Show loading state (optional: progress bar)
        container.style.opacity = '0.7';
        container.style.pointerEvents = 'none';

        try {
            const response = await fetch(url);
            const html = await response.text();

            // Parse HTML
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            const newContent = doc.querySelector(contentSelector);

            if (!newContent) {
                console.warn('SPA: No content container in response, full reload');
                window.location.href = url;
                return;
            }

            // Update Page Title
            document.title = doc.title;

            // Swap Content
            container.innerHTML = newContent.innerHTML;

            // Re-execute Scripts with Transpilation
            await executeScripts(container);

            // Re-initialize plugins (dispatch event)
            window.dispatchEvent(new Event('app:navigated'));
            window.dispatchEvent(new Event('resize')); // Trigger chart resizing

            // Restore state
            container.style.opacity = '';
            container.style.pointerEvents = '';

            // Scroll to top of content (not page, to keep sidebar position)
            window.scrollTo(0, 0);

        } catch (err) {
            console.error('SPA Navigation Error:', err);
            window.location.href = url; // Fallback
        }
    }

    async function executeScripts(container) {
        const scripts = container.querySelectorAll('script');

        for (const script of scripts) {
            const newScript = document.createElement('script');

            // Copy attributes
            Array.from(script.attributes).forEach(attr => {
                newScript.setAttribute(attr.name, attr.value);
            });

            // Transpile const/let to var to prevent redeclaration errors
            if (script.innerHTML) {
                let content = script.innerHTML;

                // Simple regex based transpilation - not perfect but catches most top-levels
                // Replace "const x =" with "var x ="
                content = content.replace(/\bconst\s+/g, 'var ');
                content = content.replace(/\blet\s+/g, 'var ');

                // Handle DOMContentLoaded for SPA navigation
                // Replace document.addEventListener('DOMContentLoaded', ...) with window.spaOnLoad(...)
                content = content.replace(/document\.addEventListener\s*\(\s*['"]DOMContentLoaded['"]\s*,/g, 'window.spaOnLoad(');

                newScript.innerHTML = content;
            }

            // Replace in DOM
            script.parentNode.replaceChild(newScript, script);

            // Wait for src scripts to load
            if (newScript.src) {
                await new Promise((resolve) => {
                    newScript.onload = resolve;
                    newScript.onerror = resolve;
                });
            }
        }
    }
});
