<?php
/**
 * API to update print status for orders
 * Updates printed_online/printed_shop and their counts
 */
header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
    exit;
}

require_once dirname(dirname(__DIR__)) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo json_encode(['status' => 'error', 'message' => 'Shop not found']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['order_ids']) || !isset($input['type'])) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid input']);
    exit;
}

$order_ids = $input['order_ids'];
$type = $input['type']; // 'online' or 'shop'

if (!is_array($order_ids) || empty($order_ids)) {
    echo json_encode(['status' => 'error', 'message' => 'No orders specified']);
    exit;
}

if (!in_array($type, ['online', 'shop'])) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid print type']);
    exit;
}

try {
    $updated_count = 0;

    foreach ($order_ids as $order_id) {
        $order_id = $conn->real_escape_string(trim($order_id));

        if ($type === 'online') {
            // Update online print status + set order_status to 'processing' (ກຳລັງຈັດສົ່ງ)
            $sql = "UPDATE orders 
                    SET printed_online = 1, 
                        printed_count_online = printed_count_online + 1,
                        order_status = 'processing',
                        updated_at = CURRENT_TIMESTAMP
                    WHERE OR_id = ? AND shop_id = ?";
        } else {
            // Update shop print status
            $sql = "UPDATE orders 
                    SET printed_shop = 1, 
                        printed_count_shop = printed_count_shop + 1,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE OR_id = ? AND shop_id = ?";
        }

        $stmt = $conn->prepare($sql);
        $stmt->bind_param("si", $order_id, $shop_id);

        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                $updated_count++;
            }
        }

        $stmt->close();
    }

    echo json_encode([
        'status' => 'success',
        'message' => "Updated $updated_count orders",
        'updated_count' => $updated_count
    ]);

} catch (Exception $e) {
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>