<?php
/**
 * API: Full Order Update
 * Updates all order details including items with stock adjustment
 */

@ob_end_clean();
@ini_set('display_errors', 0);
@error_reporting(0);

header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check authentication
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
    exit;
}

require_once dirname(__DIR__, 2) . '/includes/con_database.php';
require_once dirname(__DIR__, 2) . '/includes/secondary_ids.php';

$shop_id = $_SESSION['shop_id'] ?? null;
$user_id = $_SESSION['user_id'] ?? null;

if (!$shop_id) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບຂໍ້ມູນຮ້ານຄ້າ']);
    exit;
}

// Get input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || empty($input['order_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid request']);
    exit;
}

$order_id = $input['order_id'];
$customer_id = $input['customer_id'] ?? '';
$payment_status = $input['payment_status'] ?? 'pending';
$order_status = $input['order_status'] ?? 'pending';
$deposit_amount = floatval($input['deposit_amount'] ?? 0);
$refund_amount = floatval($input['refund_amount'] ?? 0);
$shipping_company_id = $input['shipping_company_id'] ?? '';
$shipping_branch_id = $input['shipping_branch_id'] ?? '';
$shipping_code = trim($input['shipping_code'] ?? '');
$shipping_fee = floatval($input['shipping_fee'] ?? 0);
$discount_amount = floatval($input['discount_amount'] ?? 0);
$notes = trim($input['notes'] ?? '');

$cod_type = trim($input['cod_type'] ?? '');
$payment_method = trim($input['payment_method'] ?? '');

// Append special info to notes if not already present
if ($cod_type) {
    if (strpos($notes, "COD: $cod_type") === false) {
        $notes .= ($notes ? "\n" : "") . "COD: $cod_type";
    }
}
if ($payment_method) {
    if (strpos($notes, "Payment: $payment_method") === false) {
        $notes .= ($notes ? "\n" : "") . "Payment: $payment_method";
    }
}

$items = $input['items'] ?? [];

// Currency fields
$currencys_id = !empty($input['currencys_id']) ? $conn->real_escape_string($input['currencys_id']) : null;
$rate_id = !empty($input['rate_id']) ? $conn->real_escape_string($input['rate_id']) : null;
$currency_rate = floatval($input['currency_rate'] ?? 1);
$vat_enabled = isset($input['vat_enabled']) ? intval($input['vat_enabled']) : 0;

// Validate
if (empty($items)) {
    echo json_encode(['status' => 'error', 'message' => 'ກະລຸນາເພີ່ມສິນຄ້າ']);
    exit;
}

$valid_statuses = ['pending', 'partial', 'paid', 'partial_refund', 'refunded'];
if (!in_array($payment_status, $valid_statuses)) {
    echo json_encode(['status' => 'error', 'message' => 'ສະຖານະຈ່າຍບໍ່ຖືກຕ້ອງ']);
    exit;
}

$valid_order_statuses = ['pending', 'processing', 'shipped', 'delivered', 'returned', 'cancelled'];
if (!in_array($order_status, $valid_order_statuses)) {
    echo json_encode(['status' => 'error', 'message' => 'ສະຖານະອໍເດີບໍ່ຖືກຕ້ອງ']);
    exit;
}

// Check if order exists and get INT id
$check_sql = "SELECT id, OR_id FROM orders WHERE OR_id = ? AND shop_id = ? AND deleted_at IS NULL";
$check_stmt = $conn->prepare($check_sql);
$check_stmt->bind_param("si", $order_id, $shop_id);
$check_stmt->execute();
$check_result = $check_stmt->get_result();

if ($check_result->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບໃບບິນນີ້']);
    exit;
}
$order_row = $check_result->fetch_assoc();
$order_int_id = intval($order_row['id']); // Use INT id for order_items
$check_stmt->close();

// Get user secondary ID
$user_secondary_id = getUserSecondaryId($conn, $user_id);

// Start transaction
$conn->begin_transaction();

try {
    // 1. Get old order items for stock adjustment (use INT id)
    $old_items_stmt = $conn->prepare("SELECT product_id, quantity FROM order_items WHERE order_id = ?");
    $old_items_stmt->bind_param("i", $order_int_id);
    $old_items_stmt->execute();
    $old_items_result = $old_items_stmt->get_result();

    $old_items = [];
    while ($row = $old_items_result->fetch_assoc()) {
        $old_items[$row['product_id']] = floatval($row['quantity']);
    }
    $old_items_stmt->close();

    // 2. Delete old order items (use INT id)
    $delete_items_stmt = $conn->prepare("DELETE FROM order_items WHERE order_id = ?");
    $delete_items_stmt->bind_param("i", $order_int_id);
    $delete_items_stmt->execute();
    $delete_items_stmt->close();

    // 3. Calculate totals and insert new items
    $subtotal = 0;

    foreach ($items as $item) {
        $product_id = $item['product_id'];
        $quantity = intval($item['quantity']);
        $unit_price = floatval($item['unit_price']);
        $item_discount = floatval($item['item_discount'] ?? 0);
        $item_total = ($unit_price * $quantity) - $item_discount;

        // Get product secondary ID if numeric
        $product_secondary_id = $product_id;
        if (is_numeric($product_id)) {
            $pr_stmt = $conn->prepare("SELECT PR_id FROM products WHERE id = ? AND shop_id = ?");
            $pr_stmt->bind_param("ii", $product_id, $shop_id);
            $pr_stmt->execute();
            $pr_result = $pr_stmt->get_result();
            if ($pr_row = $pr_result->fetch_assoc()) {
                $product_secondary_id = $pr_row['PR_id'];
            }
            $pr_stmt->close();
        }

        // Insert order item (use INT id)
        $insert_item_stmt = $conn->prepare("
            INSERT INTO order_items (order_id, product_id, product_name, quantity, unit_price, discount_amount, total)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $product_name = $item['product_name'] ?? '';
        $insert_item_stmt->bind_param(
            "issdddd",
            $order_int_id,
            $product_secondary_id,
            $product_name,
            $quantity,
            $unit_price,
            $item_discount,
            $item_total
        );
        $insert_item_stmt->execute();
        $insert_item_stmt->close();

        $subtotal += $item_total;

        // 4. Adjust stock
        $old_qty = $old_items[$product_secondary_id] ?? 0;
        $qty_diff = $old_qty - $quantity; // Positive = return to stock, Negative = take from stock

        if ($qty_diff != 0) {
            $stock_stmt = $conn->prepare("UPDATE products SET quantity = quantity + ? WHERE PR_id = ? AND shop_id = ?");
            $stock_stmt->bind_param("isi", $qty_diff, $product_secondary_id, $shop_id);
            $stock_stmt->execute();
            $stock_stmt->close();
        }

        // Remove from old_items (any remaining will be returned to stock)
        unset($old_items[$product_secondary_id]);
    }

    // 5. Return stock for removed items
    foreach ($old_items as $product_id => $qty) {
        $return_stmt = $conn->prepare("UPDATE products SET quantity = quantity + ? WHERE PR_id = ? AND shop_id = ?");
        $return_stmt->bind_param("isi", $qty, $product_id, $shop_id);
        $return_stmt->execute();
        $return_stmt->close();
    }

    // 6. Calculate final total with VAT
    // Use values from frontend if provided to avoid rounding discrepancy
    if (isset($input['subtotal'])) {
        $subtotal = floatval($input['subtotal']);
    }

    // Check for tax amount from frontend
    if (isset($input['tax_amount'])) {
        $tax_amount = floatval($input['tax_amount']);
    } else {
        $taxable_amount = $subtotal - $discount_amount;
        $tax_amount = $vat_enabled ? ($taxable_amount * 0.1) : 0;
    }

    // Check for total amount from frontend
    if (isset($input['total_amount'])) {
        $total_amount = floatval($input['total_amount']);
    } else {
        $taxable_amount = $subtotal - $discount_amount;
        // Recalculate tax if not provided (though logic above handles it)
        if (!isset($input['tax_amount'])) {
            $tax_amount = $vat_enabled ? ($taxable_amount * 0.1) : 0;
        }
        $total_amount = $taxable_amount + $tax_amount + $shipping_fee;
    }

    // 7. Get LAK values (prefer input values to avoid rounding drift)
    $subtotal_lak = isset($input['subtotal_lak']) ? floatval($input['subtotal_lak']) : ($subtotal * $currency_rate);
    $discount_lak = isset($input['discount_lak']) ? floatval($input['discount_lak']) : ($discount_amount * $currency_rate);
    $vat_amount_lak = isset($input['vat_amount_lak']) ? floatval($input['vat_amount_lak']) : ($tax_amount * $currency_rate);
    $total_amount_lak = isset($input['total_amount_lak']) ? floatval($input['total_amount_lak']) : ($total_amount * $currency_rate);

    // 8. Update order
    $update_sql = "UPDATE orders SET 
        customer_id = NULLIF(?, ''),
        payment_status = ?,
        order_status = ?,
        deposit_amount = ?,
        refund_amount = ?,
        shipping_company_id = NULLIF(?, ''),
        shipping_branch_id = NULLIF(?, ''),
        shipping_code = ?,
        shipping_fee = ?,
        subtotal = ?,
        discount_amount = ?,
        tax_amount = ?,
        total_amount = ?,
        notes = ?,
        currencys_id = NULLIF(?, ''),
        rate_id = NULLIF(?, ''),
        currency_rate = ?,
        vat_enabled = ?,
        subtotal_lak = ?,
        discount_lak = ?,
        vat_amount_lak = ?,
        total_amount_lak = ?,
        updated_by = ?,
        updated_at = NOW()
    WHERE OR_id = ? AND shop_id = ?";

    $update_stmt = $conn->prepare($update_sql);
    $update_stmt->bind_param(
        "sssddsssdddddsssdiddddssi",
        $customer_id,
        $payment_status,
        $order_status,
        $deposit_amount,
        $refund_amount,
        $shipping_company_id,
        $shipping_branch_id,
        $shipping_code,
        $shipping_fee,
        $subtotal,
        $discount_amount,
        $tax_amount,
        $total_amount,
        $notes,
        $currencys_id,
        $rate_id,
        $currency_rate,
        $vat_enabled,
        $subtotal_lak,
        $discount_lak,
        $vat_amount_lak,
        $total_amount_lak,
        $user_secondary_id,
        $order_id,
        $shop_id
    );
    $update_stmt->execute();
    $update_stmt->close();

    // Handle tip data
    // First, delete any existing tip for this order
    $delete_tip_sql = "DELETE FROM tips WHERE order_id = ? AND shop_id = ?";
    $delete_tip_stmt = $conn->prepare($delete_tip_sql);
    $delete_tip_stmt->bind_param("si", $order_id, $shop_id);
    $delete_tip_stmt->execute();
    $delete_tip_stmt->close();

    // Save new tip data if present and order is paid
    $tip_id = null;
    if (!empty($input['tip_data']) && $payment_status === 'paid') {
        $tip_data = $input['tip_data'];
        $tip_currencys_id = $conn->real_escape_string($tip_data['currencys_id']);
        $tip_currency_rate = floatval($tip_data['currency_rate']);
        $tip_original_total = floatval($tip_data['original_total']);
        $tip_rounded_total = floatval($tip_data['rounded_total']);
        $tip_amount_foreign = floatval($tip_data['tip_amount_foreign']);
        $tip_amount_lak = floatval($tip_data['tip_amount_lak']);
        $tip_calculation_note = $conn->real_escape_string($tip_data['calculation_note'] ?? '');

        $tip_sql = "INSERT INTO tips (
            shop_id, order_id, currencys_id, currency_rate,
            original_total, rounded_total, tip_amount_foreign, tip_amount_lak,
            calculation_note, created_by
        ) VALUES (
            $shop_id,
            '$order_id',
            '$tip_currencys_id',
            $tip_currency_rate,
            $tip_original_total,
            $tip_rounded_total,
            $tip_amount_foreign,
            $tip_amount_lak,
            '$tip_calculation_note',
            '" . $conn->real_escape_string($user_secondary_id) . "'
        )";

        if ($conn->query($tip_sql)) {
            $tip_id = $conn->insert_id;
        }
    }

    // Commit transaction
    $conn->commit();

    echo json_encode([
        'status' => 'success',
        'message' => 'ອັບເດດສຳເລັດ',
        'order_id' => $order_id,
        'total' => $total_amount,
        'tip_id' => $tip_id
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode([
        'status' => 'error',
        'message' => 'ເກີດຂໍ້ຜິດພາງ: ' . $e->getMessage()
    ]);
}

$conn->close();
?>