<?php
/**
 * API: Update Order
 * Updates order payment status, shipping code, deposit amount, and notes
 */

// Prevent any output before JSON
@ob_end_clean();
@ini_set('display_errors', 0);
@error_reporting(0);

header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check authentication
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
    exit;
}

require_once dirname(__DIR__, 2) . '/includes/con_database.php';
require_once dirname(__DIR__, 2) . '/includes/secondary_ids.php';

$shop_id = $_SESSION['shop_id'] ?? null;
$user_id = $_SESSION['user_id'] ?? null;

if (!$shop_id) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບຂໍ້ມູນຮ້ານຄ້າ']);
    exit;
}

// Get input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid request']);
    exit;
}

$order_id = $input['order_id'] ?? '';
$payment_status = $input['payment_status'] ?? '';
$deposit_amount = floatval($input['deposit_amount'] ?? 0);
$shipping_code = trim($input['shipping_code'] ?? '');
$notes = trim($input['notes'] ?? '');

// Validate
if (empty($order_id)) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບໃບບິນ']);
    exit;
}

$valid_statuses = ['pending', 'partial', 'paid', 'refunded'];
if (!in_array($payment_status, $valid_statuses)) {
    echo json_encode(['status' => 'error', 'message' => 'ສະຖານະບໍ່ຖືກຕ້ອງ']);
    exit;
}

// Check if order exists and belongs to this shop
$check_sql = "SELECT id FROM orders WHERE OR_id = ? AND shop_id = ? AND deleted_at IS NULL";
$check_stmt = $conn->prepare($check_sql);
$check_stmt->bind_param("si", $order_id, $shop_id);
$check_stmt->execute();
$check_result = $check_stmt->get_result();

if ($check_result->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບໃບບິນນີ້']);
    exit;
}

$check_stmt->close();

// Get user secondary ID
$user_secondary_id = getUserSecondaryId($conn, $user_id);

// Update order
$update_sql = "UPDATE orders SET 
    payment_status = ?,
    deposit_amount = ?,
    shipping_code = ?,
    notes = ?,
    updated_by = ?,
    updated_at = NOW()
WHERE OR_id = ? AND shop_id = ?";

$update_stmt = $conn->prepare($update_sql);
$update_stmt->bind_param(
    "sdssssi",
    $payment_status,
    $deposit_amount,
    $shipping_code,
    $notes,
    $user_secondary_id,
    $order_id,
    $shop_id
);

if ($update_stmt->execute()) {
    echo json_encode([
        'status' => 'success',
        'message' => 'ອັບເດດສຳເລັດ'
    ]);
} else {
    echo json_encode([
        'status' => 'error',
        'message' => 'ເກີດຂໍ້ຜິດພາດໃນການບັນທຶກ: ' . $conn->error
    ]);
}

$update_stmt->close();
$conn->close();
?>