    <?php
    /**
     * API to restore cancelled/returned/refunded orders
     * Sets order_status back to 'pending' and payment_status back to 'pending'
     */
    header('Content-Type: application/json');

    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Check if logged in and is admin
    if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
        http_response_code(403);
        echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
        exit;
    }

    require_once dirname(dirname(__DIR__)) . '/includes/con_database.php';
    require_once dirname(dirname(__DIR__)) . '/includes/secondary_ids.php';

    $shop_id = $_SESSION['shop_id'] ?? null;
    $user_id = $_SESSION['user_id'] ?? null;

    if (!$shop_id) {
        echo json_encode(['status' => 'error', 'message' => 'Shop not found']);
        exit;
    }

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input || !isset($input['order_id'])) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid input']);
        exit;
    }

    $order_id = $conn->real_escape_string(trim($input['order_id']));
    $user_secondary_id = getUserSecondaryId($conn, $user_id);

    try {
        // Restore order - set status back to pending
        $sql = "UPDATE orders 
                SET order_status = 'pending', 
                    payment_status = 'pending',
                    updated_by = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE OR_id = ? AND shop_id = ?";

        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssi", $user_secondary_id, $order_id, $shop_id);

        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'ກູ້ຄືນອໍເດີສຳເລັດ'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'ບໍ່ພົບອໍເດີນີ້'
                ]);
            }
        } else {
            throw new Exception($stmt->error);
        }

        $stmt->close();

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
    ?>