<?php
/**
 * POS Order Submission API
 * Column names match actual server database schema
 * Uses secondary IDs (US_id, CU_id, PR_id, SC_id, SB_id) instead of auto-increment ids
 * 
 * Now saves BOTH:
 * - subtotal/total_amount = in selected currency (e.g., Baht, USD)
 * - subtotal_lak/total_amount_lak = in LAK (base currency for reference)
 */

session_start();
header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

require_once '../../includes/con_database.php';
require_once '../../includes/secondary_ids.php';

$user_id = $_SESSION['user_id'];
$user_secondary_id = getUserSecondaryId($conn, $user_id);

try {
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input || empty($input['cart'])) {
        throw new Exception('ຂໍ້ມູນບໍ່ຖືກຕ້ອງ');
    }

    $shop_id = intval($input['shop_id']);
    $sale_type = $input['sale_type'] ?? 'pos';
    $cart = $input['cart'];

    // Customer - store CU_id instead of id
    $customer_name = trim($input['customer_name'] ?? '');
    $customer_phone = trim($input['customer_phone'] ?? '');
    $customer_secondary_id = null;

    if (!empty($customer_name)) {
        $stmt = $conn->prepare("SELECT id, CU_id FROM customers WHERE shop_id = ? AND name = ? LIMIT 1");
        $stmt->bind_param("is", $shop_id, $customer_name);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($row = $result->fetch_assoc()) {
            $customer_secondary_id = $row['CU_id'];
        } else {
            $stmt = $conn->prepare("INSERT INTO customers (shop_id, name, phone, created_by) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("isss", $shop_id, $customer_name, $customer_phone, $user_secondary_id);
            $stmt->execute();
            $new_customer_id = $conn->insert_id;
            $customer_secondary_id = getCustomerSecondaryId($conn, $new_customer_id);
        }
    }

    // Shipping - store SC_id and SB_id instead of integer ids
    $shipping_company_secondary_id = null;
    $shipping_branch_secondary_id = null;

    if (!empty($input['shipping_company_id'])) {
        $shipping_company_secondary_id = getShippingCompanySecondaryId($conn, intval($input['shipping_company_id']));
    }
    if (!empty($input['shipping_branch_id'])) {
        $shipping_branch_secondary_id = getShippingBranchSecondaryId($conn, intval($input['shipping_branch_id']));
    }

    $cod_type = $input['cod_type'] ?? null;

    // Payment
    $payment_status = $input['payment_status'] ?? 'pending';
    $db_payment_status = 'pending';
    if ($payment_status === 'paid')
        $db_payment_status = 'paid';
    elseif ($payment_status === 'partial')
        $db_payment_status = 'partial';

    // LAK amounts (base currency - always saved for reference)
    $subtotal_lak = floatval($input['subtotal_lak'] ?? 0);
    $discount_lak = floatval($input['discount_lak'] ?? 0);
    $vat_amount_lak = floatval($input['vat_amount_lak'] ?? 0);
    $total_amount_lak = floatval($input['total_amount_lak'] ?? 0);

    // Display currency amounts (selected currency - e.g., Baht, USD)
    $subtotal_display = floatval($input['subtotal_display'] ?? $subtotal_lak);
    $discount_display = floatval($input['discount_display'] ?? $discount_lak);
    $vat_display = floatval($input['vat_display'] ?? $vat_amount_lak);
    $total_display = floatval($input['total_display'] ?? $total_amount_lak);

    // Currency info
    $currencys_id = !empty($input['currencys_id']) ? $conn->real_escape_string($input['currencys_id']) : null;
    $currency_rate = floatval($input['currency_rate'] ?? 1);
    $rate_id = !empty($input['rate_id']) ? $conn->real_escape_string($input['rate_id']) : null;
    $vat_enabled = intval($input['vat_enabled'] ?? 1);

    // Cash (in selected currency)
    $cash_received = floatval($input['amount_received'] ?? 0);
    $cash_change = floatval($input['change_amount'] ?? 0);

    // Deposit amount for partial payments
    $deposit_amount = 0;
    if ($db_payment_status === 'partial') {
        // Try to get deposit from input, fallback to cash_received if not provided
        $deposit_amount = floatval($input['deposit_amount'] ?? $cash_received);
    }

    $order_type = $sale_type === 'online' ? 'online' : 'pos';

    // Set order_status based on order type
    // online = 'pending' (ກະກຽມຈັດສົ່ງ), pos = 'delivered' (ลูกค้ารับสินค้าเลย)
    $order_status = ($order_type === 'online') ? 'pending' : 'delivered';

    $notes = $cod_type ? $conn->real_escape_string("COD: $cod_type") : null;

    $conn->begin_transaction();

    // INSERT INTO orders
    // subtotal/total_amount = display currency (Baht, USD, etc.)
    // subtotal_lak/total_amount_lak = LAK (base currency for reference)
    $sql = "INSERT INTO orders (
        shop_id, customer_id, order_type, order_status, payment_status,
        shipping_company_id, shipping_branch_id,
        subtotal, discount_amount, tax_amount, total_amount, deposit_amount,
        cash_received, cash_change, notes,
        vat_enabled, currencys_id, rate_id, currency_rate,
        subtotal_lak, vat_amount_lak, total_amount_lak, discount_lak,
        created_by
    ) VALUES (
        $shop_id,
        " . ($customer_secondary_id ? "'" . $conn->real_escape_string($customer_secondary_id) . "'" : 'NULL') . ",
        '$order_type',
        '$order_status',
        '$db_payment_status',
        " . ($shipping_company_secondary_id ? "'" . $conn->real_escape_string($shipping_company_secondary_id) . "'" : 'NULL') . ",
        " . ($shipping_branch_secondary_id ? "'" . $conn->real_escape_string($shipping_branch_secondary_id) . "'" : 'NULL') . ",
        $subtotal_display,
        $discount_display,
        $vat_display,
        $total_display,
        $deposit_amount,
        $cash_received,
        $cash_change,
        " . ($notes ? "'$notes'" : 'NULL') . ",
        $vat_enabled,
        " . ($currencys_id ? "'$currencys_id'" : 'NULL') . ",
        " . ($rate_id ? "'$rate_id'" : 'NULL') . ",
        $currency_rate,
        $subtotal_lak,
        $vat_amount_lak,
        $total_amount_lak,
        $discount_lak,
        '" . $conn->real_escape_string($user_secondary_id) . "'
    )";

    if (!$conn->query($sql)) {
        throw new Exception('ບໍ່ສາມາດບັນທຶກອໍເດີໄດ້: ' . $conn->error);
    }

    $order_id = $conn->insert_id;

    // INSERT INTO order_items
    // unit_price/total = in selected currency
    $stmt_item = $conn->prepare("INSERT INTO order_items (order_id, product_id, product_name, quantity, unit_price, total) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt_stock = $conn->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ? AND shop_id = ?");

    foreach ($cart as $item) {
        $product_id = intval($item['product_id']);
        $product_secondary_id = getProductSecondaryId($conn, $product_id);
        $product_name = $item['product_name'];
        $qty = floatval($item['quantity']);

        // Use display currency price if available, otherwise use LAK
        $unit_price = floatval($item['unit_price_display'] ?? $item['unit_price_lak']);
        $item_total = floatval($item['subtotal_display'] ?? $item['subtotal_lak']);

        $stmt_item->bind_param("issddd", $order_id, $product_secondary_id, $product_name, $qty, $unit_price, $item_total);
        if (!$stmt_item->execute()) {
            throw new Exception('ບໍ່ສາມາດບັນທຶກລາຍການສິນຄ້າໄດ້: ' . $stmt_item->error);
        }

        // Update stock using original product_id (integer) for the WHERE clause
        $stmt_stock->bind_param("dii", $qty, $product_id, $shop_id);
        $stmt_stock->execute();
    }

    // Get the OR_id for the order just created
    $stmt_or_id = $conn->prepare("SELECT OR_id FROM orders WHERE id = ?");
    $stmt_or_id->bind_param("i", $order_id);
    $stmt_or_id->execute();
    $order_result = $stmt_or_id->get_result()->fetch_assoc();
    $or_id = $order_result['OR_id'];

    // Save tip data if present and order is paid
    $tip_id = null;
    if (!empty($input['tip_data']) && $db_payment_status === 'paid') {
        $tip_data = $input['tip_data'];
        $tip_currencys_id = $conn->real_escape_string($tip_data['currencys_id']);
        $tip_currency_rate = floatval($tip_data['currency_rate']);
        $tip_original_total = floatval($tip_data['original_total']);
        $tip_rounded_total = floatval($tip_data['rounded_total']);
        $tip_amount_foreign = floatval($tip_data['tip_amount_foreign']);
        $tip_amount_lak = floatval($tip_data['tip_amount_lak']);
        $tip_calculation_note = $conn->real_escape_string($tip_data['calculation_note'] ?? '');

        $tip_sql = "INSERT INTO tips (
            shop_id, order_id, currencys_id, currency_rate,
            original_total, rounded_total, tip_amount_foreign, tip_amount_lak,
            calculation_note, created_by
        ) VALUES (
            $shop_id,
            '$or_id',
            '$tip_currencys_id',
            $tip_currency_rate,
            $tip_original_total,
            $tip_rounded_total,
            $tip_amount_foreign,
            $tip_amount_lak,
            '$tip_calculation_note',
            '" . $conn->real_escape_string($user_secondary_id) . "'
        )";

        if ($conn->query($tip_sql)) {
            $tip_id = $conn->insert_id;
        }
    }

    $conn->commit();

    echo json_encode([
        'success' => true,
        'message' => 'ບັນທຶກສຳເລັດ',
        'order_id' => $order_id,
        'or_id' => $or_id,
        'tip_id' => $tip_id
    ]);

} catch (Exception $e) {
    if (isset($conn) && $conn->ping()) {
        $conn->rollback();
    }
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
