<?php
/**
 * POS Quick Add API
 * Handles quick add for shipping companies and branches from POS modal
 */

session_start();
header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// Correct path: from admin/api/ to includes/
require_once '../../includes/con_database.php';
require_once '../../includes/secondary_ids.php';

$user_id = $_SESSION['user_id'];
$user_secondary_id = getUserSecondaryId($conn, $user_id);

try {
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input) {
        throw new Exception('Invalid JSON input');
    }

    $type = $input['type'] ?? '';
    $name = trim($input['name'] ?? '');
    $shop_id = intval($input['shop_id'] ?? 0);

    if (!$name || !$shop_id) {
        throw new Exception('ຂໍ້ມູນບໍ່ຄົບຖ້ວນ');
    }

    if ($type === 'company') {
        // Add shipping company
        $sql = "INSERT INTO shipping_companies (shop_id, company_name, is_active, created_by, created_at) VALUES (?, ?, 1, ?, NOW())";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }

        $stmt->bind_param("iss", $shop_id, $name, $user_secondary_id);

        if ($stmt->execute()) {
            echo json_encode([
                'success' => true,
                'id' => $conn->insert_id,
                'message' => 'ເພີ່ມບໍລິສັດແລ້ວ'
            ]);
        } else {
            throw new Exception('Execute failed: ' . $stmt->error);
        }

    } elseif ($type === 'branch') {
        $company_id = intval($input['company_id'] ?? 0);

        if (!$company_id) {
            throw new Exception('ກະລຸນາເລືອກບໍລິສັດກ່ອນ');
        }

        // Get SC_id from company_id
        $company_secondary_id = getShippingCompanySecondaryId($conn, $company_id);

        // Add shipping branch
        $sql = "INSERT INTO shipping_branches (shop_id, company_id, branch_name, is_active, created_by, created_at) VALUES (?, ?, ?, 1, ?, NOW())";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }

        $stmt->bind_param("isss", $shop_id, $company_secondary_id, $name, $user_secondary_id);

        if ($stmt->execute()) {
            echo json_encode([
                'success' => true,
                'id' => $conn->insert_id,
                'message' => 'ເພີ່ມສາຂາແລ້ວ'
            ]);
        } else {
            throw new Exception('Execute failed: ' . $stmt->error);
        }

    } elseif ($type === 'customer') {
        $phone = trim($input['phone'] ?? '');

        if (!$phone) {
            throw new Exception('ກະລຸນາປ້ອນເບີໂທ');
        }

        // Add customer
        $sql = "INSERT INTO customers (shop_id, name, phone, is_active, created_by, created_at) VALUES (?, ?, ?, 1, ?, NOW())";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }

        $stmt->bind_param("isss", $shop_id, $name, $phone, $user_secondary_id);

        if ($stmt->execute()) {
            // Get the CU_id that was generated by trigger
            $new_id = $conn->insert_id;
            $cu_id_stmt = $conn->prepare("SELECT CU_id FROM customers WHERE id = ?");
            $cu_id_stmt->bind_param("i", $new_id);
            $cu_id_stmt->execute();
            $cu_result = $cu_id_stmt->get_result()->fetch_assoc();

            echo json_encode([
                'success' => true,
                'id' => $cu_result['CU_id'] ?? $new_id,
                'message' => 'ເພີ່ມລູກຄ້າແລ້ວ'
            ]);
        } else {
            throw new Exception('Execute failed: ' . $stmt->error);
        }

    } else {
        throw new Exception('ປະເພດບໍ່ຖືກຕ້ອງ');
    }

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
