<?php
// API endpoint for getting order details
header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_code'] !== 'admin') {
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
    exit;
}

require_once dirname(__DIR__, 2) . '/includes/con_database.php';

$shop_id = $_SESSION['shop_id'] ?? null;
if (!$shop_id) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບຂໍ້ມູນຮ້ານຄ້າ']);
    exit;
}

$order_id = isset($_GET['order_id']) ? trim($_GET['order_id']) : '';
if (empty($order_id)) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບ Order ID']);
    exit;
}

// Get order details
$sql = "SELECT 
            o.*,
            c.name as customer_name,
            c.phone as customer_phone,
            c.address as customer_address,
            sc.company_name as shipping_company,
            sb.branch_name as shipping_branch,
            cu.currencys_symbol as currency_symbol
        FROM orders o
        LEFT JOIN customers c ON o.customer_id = c.CU_id
        LEFT JOIN shipping_companies sc ON o.shipping_company_id = sc.SC_id
        LEFT JOIN shipping_branches sb ON o.shipping_branch_id = sb.SB_id
        LEFT JOIN currencys cu ON o.currencys_id = cu.currencys_id
        WHERE o.shop_id = ? AND o.OR_id = ? AND o.deleted_at IS NULL";

$stmt = $conn->prepare($sql);
$stmt->bind_param("is", $shop_id, $order_id);
$stmt->execute();
$result = $stmt->get_result();
$order = $result->fetch_assoc();
$stmt->close();

if (!$order) {
    echo json_encode(['status' => 'error', 'message' => 'ບໍ່ພົບຂໍ້ມູນໃບບິນ']);
    exit;
}

// Get order items
$sql = "SELECT 
            oi.*,
            p.product_name as product_name_current,
            p.product_code,
            p.product_image
        FROM order_items oi
        LEFT JOIN products p ON oi.product_id = p.PR_id
        WHERE oi.order_id = ?";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $order['id']);
$stmt->execute();
$result = $stmt->get_result();
$items = [];
while ($row = $result->fetch_assoc()) {
    $items[] = [
        'id' => $row['id'],
        'product_id' => $row['product_id'],
        'product_name' => $row['product_name'] ?? $row['product_name_current'] ?? '-',
        'product_code' => $row['product_code'] ?? '',
        'product_image' => $row['product_image'] ?? '',
        'quantity' => floatval($row['quantity']),
        'unit_price' => floatval($row['unit_price']),
        'discount' => floatval($row['discount_amount'] ?? 0),
        'total' => floatval($row['total'])
    ];
}
$stmt->close();

// Format order data
$order_type_text = '-';
switch ($order['order_type']) {
    case 'pos':
        $order_type_text = 'ຂາຍໜ້າຮ້ານ';
        break;
    case 'online':
        $order_type_text = 'ຂາຍອອນລາຍ';
        break;
    case 'wholesale':
        $order_type_text = 'ຂາຍສົ່ງ';
        break;
}

$payment_status_text = '-';
switch ($order['payment_status']) {
    case 'paid':
        $payment_status_text = 'ຈ່າຍແລ້ວ';
        break;
    case 'partial':
        $payment_status_text = 'ມັດຈຳ';
        break;
    case 'pending':
        $payment_status_text = 'ຍັງບໍ່ຈ່າຍ';
        break;
    case 'partial_refund':
        $payment_status_text = 'ຄືນເງີນບາງສ່ວນ';
        break;
    case 'refunded':
        $payment_status_text = 'ຄືນເງີນເຕັມຈຳນວນ';
        break;
}

// Get values directly from database
$subtotal = floatval($order['subtotal'] ?? 0);
$discount_amount = floatval($order['discount_amount'] ?? 0);
$shipping_fee = floatval($order['shipping_fee'] ?? 0);
$vat_enabled = intval($order['vat_enabled'] ?? 0);
$currency_rate = floatval($order['currency_rate'] ?? 1);

// Use tax_amount from database instead of recalculating
$vat_amount = floatval($order['tax_amount'] ?? 0);

// Use total_amount from database instead of recalculating
$display_total = floatval($order['total_amount'] ?? 0);

$response = [
    'status' => 'success',
    'order' => [
        'order_id' => $order['OR_id'],
        'order_type' => $order['order_type'],
        'order_type_text' => $order_type_text,
        'order_date' => date('d/m/Y H:i', strtotime($order['order_date'])),
        'customer_name' => $order['customer_name'] ?? '-',
        'customer_phone' => $order['customer_phone'] ?? '-',
        'customer_address' => $order['customer_address'] ?? '-',
        'shipping_company' => $order['shipping_company'] ?? '-',
        'shipping_branch' => $order['shipping_branch'] ?? '-',
        'shipping_code' => $order['shipping_code'] ?? '-',
        'payment_status' => $order['payment_status'],
        'payment_status_text' => $payment_status_text,
        'subtotal' => $subtotal,
        'discount_amount' => $discount_amount,
        'shipping_fee' => $shipping_fee,
        'vat_enabled' => $vat_enabled,
        'vat_amount' => $vat_amount,
        'total_amount' => $display_total,
        'deposit_amount' => floatval($order['deposit_amount'] ?? 0),
        'cash_received' => floatval($order['cash_received'] ?? 0),
        'cash_change' => floatval($order['cash_change'] ?? 0),
        'refund_amount' => floatval($order['refund_amount'] ?? 0),
        'notes' => $order['notes'] ?? '',
        'currency_symbol' => $order['currency_symbol'] ?? '₭',
        'currencys_id' => $order['currencys_id'] ?? 'CUR_1'
    ],
    'items' => $items
];

echo json_encode($response);
?>